#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2024. All rights reserved.
import argparse
import json
import re
import os
import shutil
import stat
import subprocess
from multiprocessing import Pool
from pathlib import Path
from datetime import datetime, timedelta, timezone
import yaml


TIMEOUT_SEC = 15 * 60
DEFAULT_TOOL_NAME = "bingo"
BACKUP_TOOL_NAME = "bmcgo"
BMCGO_CONF_PATH = "/etc/bmcgo.conf"


def build_component(app_name: str, app_data: dict, last_remote: str):
    app_path = app_data.get("path", "")
    if not os.path.exists(app_path):
        raise RuntimeError(f"{app_name}组件仓库路径{app_path}不存在")
    cmd = ["git", "stash", "push"]
    ignored_files = app_data.get("ignoredFiles", [])
    if ignored_files:
        cmd.extend(ignored_files)
        subprocess.run(cmd, cwd=app_path, check=True)
    try:
        cmd = [BuildToolConfig.tool_name, "build"]
        if BuildToolConfig.need_remote and len(last_remote) > 0:
            cmd.extend(["-r", last_remote])
        if BuildToolConfig.is_partner_mode:
            cmd.append("-jit")
        proc = subprocess.run(cmd, cwd=app_path, capture_output=True)
        if proc.returncode != 0:
            raise RuntimeError(proc.stderr.decode())
        ok, result = parse_conan_package(app_name, proc.stdout.decode())
        if not ok:
            raise RuntimeError(result)
        return result
    finally:
        if ignored_files:
            subprocess.run(["git", "stash", "apply"], cwd=app_path, check=True)


def parse_conan_package(app_name: str, build_output: str):
    for line in reversed(build_output.split('\n')):
        result = re.search(r'\S+/dev', line)
        if result:
            return True, result.group(0)
    return False, f"获取{app_name}组件conan包失败: {build_output}"


def check_is_partner_mode():
    if not os.path.exists(BMCGO_CONF_PATH):
        return False
    with open(BMCGO_CONF_PATH, "r") as f:
        file_content = f.readlines()
    found = False
    for line in file_content:
        if found and re.match("\\s*enable\\s*=\\s*true", line) is not None:
            return True
        trimmed_line = line.strip()
        if trimmed_line.startswith("["):
            found = "[partner]" in trimmed_line
    return False


class BuildToolConfig:
    tool_name = BACKUP_TOOL_NAME
    is_partner_mode = False
    is_community_mode = False
    need_remote = False

    def __init__(self):
        pass

    @staticmethod
    def check_tools_name():
        path = shutil.which(BACKUP_TOOL_NAME)
        if not path:
            BuildToolConfig.tool_name = DEFAULT_TOOL_NAME

    @staticmethod
    def check_env():
        BuildToolConfig.is_partner_mode = check_is_partner_mode()
        BuildToolConfig.is_community_mode = (BuildToolConfig.tool_name == DEFAULT_TOOL_NAME)
        BuildToolConfig.need_remote = BuildToolConfig.is_partner_mode or BuildToolConfig.is_community_mode


class ManifestBuilder:
    def __init__(self, args):
        self.input_file = args.input
        self.output_file = args.output
        if self.input_file is None or not os.path.exists(self.input_file) or self.output_file is None:
            self.raise_error("缺少参数或输入文件不存在")
        with open(self.input_file, "r") as file_descriptor:
            input_content = json.load(file_descriptor)
        self.manifest_path = input_content.get("manifestPath")
        if not self.manifest_path or not os.path.exists(self.manifest_path):
            self.raise_error("manifest仓库不存在")
        self.current_product = input_content.get("currentProduct")
        if not self.current_product:
            self.raise_error("未选择当前产品")
        self.file_changes = input_content.get("fileChanges", {})
        self.sdk_path = input_content.get("sdkPath", "")
        self.contains_sdk = input_content.get("containsSdk", False)
        self.build_options = input_content.get("options", [])
        self.yaml_files_content = {}
        self.manifest_yaml = self.find_manifest_yaml()
        with open(self.manifest_yaml, "r") as file_descriptor:
            str_data = file_descriptor.read()
        self.yaml_files_content[self.manifest_yaml] = str_data
        self.manifest_yaml_data = yaml.safe_load(str_data)
        self.sdk_modified = False

    @staticmethod
    def save_yaml_file(yaml_data: dict, yaml_path: str, str_data: str):
        schema_comment = ""
        for line in str_data.split("\n"):
            if line.startswith("#") and "schema" in line:
                schema_comment = line
                break
        with os.fdopen(os.open(yaml_path, os.O_WRONLY | os.O_CREAT | os.O_TRUNC,
                               stat.S_IWUSR | stat.S_IRUSR), "w") as file_descriptor:
            if schema_comment:
                file_descriptor.write(schema_comment + "\n")
            yaml.safe_dump(yaml_data, file_descriptor)

    def get_last_remote(self):
        conan = shutil.which("conan")
        if conan is None:
            raise Exception("缺失conan工具")
        proc = subprocess.run([conan, "remote", "list"], capture_output=True)
        if proc.returncode != 0:
            self.raise_error(proc.stderr.decode())
        remote_list = proc.stdout.split(b'\n')
        last_remote = b""
        for remote in remote_list:
            if remote:
                last_remote = remote.strip().split(b':')[0]
        return last_remote.decode('utf-8')

    def build_components(self):
        packages = {}
        if not self.file_changes:
            return packages
        pool = Pool(processes=min(len(self.file_changes), os.cpu_count()))
        error_msg = ""
        last_remote = self.get_last_remote()

        def error_handler(error: BaseException, app_name: str):
            nonlocal error_msg
            error_msg += f"组件{app_name}构建失败：{error}\n"
        for app_name, app_data in self.file_changes.items():
            result = pool.apply_async(func=build_component, args=(app_name, app_data, last_remote),
                        error_callback=lambda error: error_handler(error, app_name))
            packages[app_name] = result.get(timeout=TIMEOUT_SEC)
        pool.close()
        pool.join()
        if error_msg:
            self.raise_error(error_msg)
        return packages

    def search_subsys_config(self, manifest_or_sdk_path: str, packages: dict, is_sdk: bool):
        manifest_rc = os.path.join(manifest_or_sdk_path, "build", "subsys", "rc")
        if not os.path.isdir(manifest_rc):
            return
        found = False
        for yaml_file in os.scandir(manifest_rc):
            if not yaml_file.name.endswith(".yml"):
                continue
            with open(yaml_file, 'r') as file_descriptor:
                str_data = file_descriptor.read()
            yaml_data = yaml.safe_load(str_data)
            found_in_subsys = False
            for dep in yaml_data.get("dependencies", []):
                slices = dep.get("conan", "").split("/")
                if slices and slices[0] in packages:
                    found_in_subsys = True
                    dep["conan"] = packages[slices[0]]
                    del packages[slices[0]]
            if not found_in_subsys:
                continue
            found = True
            self.yaml_files_content[yaml_file.path] = str_data
            self.save_yaml_file(yaml_data, yaml_file.path, str_data)
        self.sdk_modified = self.sdk_modified or (found and is_sdk)

    def build_sdk(self):
        proc = subprocess.run([BuildToolConfig.tool_name, "build"], cwd=self.sdk_path, capture_output=True)
        if proc.returncode != 0:
            self.raise_error(proc.stderr.decode())
        ok, result = parse_conan_package("ibmc_sdk", proc.stdout.decode())
        if not ok:
            self.raise_error(result)
        platform_data = self.manifest_yaml_data.get("platform", {})
        platform_data["conan"] = result
        self.manifest_yaml_data["platform"] = platform_data

    def build_product(self, start_time: datetime):
        time_pass = datetime.now(tz=timezone.utc) - start_time
        max_time = timedelta(minutes=15)
        if time_pass >= max_time:
            self.copy_log("/tmp/bmcgo/bmcgo.log")
            self.raise_error("组件构建执行超时")
        time_left = (max_time - time_pass).seconds
        cmd = [BuildToolConfig.tool_name, "build", "-b", self.current_product, "-t", "target_personal"]
        remote = self.get_last_remote()
        if BuildToolConfig.need_remote and len(remote) > 0:
            cmd.extend(["-r", remote])
        cmd.extend(self.build_options)
        task_log_path = os.path.join(self.manifest_path, "temp", "log", "task.log")
        try:
            proc = subprocess.run(cmd, cwd=self.manifest_path, capture_output=True, timeout=time_left)
            if proc.returncode != 0:
                return False, proc.stderr.decode()
            self.copy_log(task_log_path)
            return True, ""
        except subprocess.TimeoutExpired:
            self.copy_log(task_log_path)
            return False, "产品构建执行超时"
        except Exception as e:
            return False, str(e)
        finally:
            self.restore_yaml_files()

    def find_manifest_yaml(self):
        products_path = os.path.join(self.manifest_path, "build", "product")
        for entry in os.scandir(products_path):
            if not entry.is_dir():
                continue
            for product_dir in os.scandir(entry):
                if not product_dir.is_dir() or product_dir.name != self.current_product:
                    continue
                yaml_path = os.path.join(product_dir, "manifest.yml")
                if not os.path.exists(yaml_path):
                    self.raise_error(f"{yaml_path}文件不存在")
                return yaml_path
        self.raise_error(f"产品{self.current_product}在{products_path}中不存在")

    def update_manifest_yaml(self, packages: dict):
        dependencies = self.manifest_yaml_data.get("dependencies", [])
        for dep in dependencies:
            conan_package = dep.get("conan", "")
            if not conan_package:
                continue
            if "/" in conan_package:
                conan_name = conan_package.split("/")[0]
            else:
                conan_name = conan_package
            if conan_name in packages:
                dep["conan"] = packages[conan_name]
                del packages[conan_name]
        for package in packages.values():
            dependencies.append({"conan": package})
        self.manifest_yaml_data["dependencies"] = dependencies

    def restore_yaml_files(self):
        for yaml_path, str_data in self.yaml_files_content.items():
            with os.fdopen(os.open(yaml_path, os.O_WRONLY | os.O_CREAT | os.O_TRUNC,
                                   stat.S_IWUSR | stat.S_IRUSR), "w") as file_descriptor:
                file_descriptor.write(str_data)

    def copy_log(self, log_file: str):
        if not os.path.exists(log_file):
            return
        with open(log_file, "r") as fp:
            self.dump_output(fp.read() + "\n")

    def dump_output(self, content: str):
        parent_dir = Path(self.output_file).parent
        if not os.path.exists(parent_dir):
            os.makedirs(parent_dir, exist_ok=True)
        with os.fdopen(os.open(self.output_file, os.O_WRONLY | os.O_CREAT | os.O_APPEND,
                               stat.S_IWUSR | stat.S_IRUSR), "a") as file_descriptor:
            file_descriptor.write(content)

    def raise_error(self, error_msg: str):
        self.dump_output(error_msg)
        raise RuntimeError("manifest构建失败")

    def run(self):
        start_time = datetime.now(tz=timezone.utc)
        packages = self.build_components()
        if self.contains_sdk and self.sdk_path:
            self.search_subsys_config(self.sdk_path, packages, True)
        self.search_subsys_config(self.manifest_path, packages, False)
        if packages:
            self.update_manifest_yaml(packages)
        if self.sdk_modified and not BuildToolConfig.need_remote:
            self.build_sdk()
        self.save_yaml_file(self.manifest_yaml_data, self.manifest_yaml, self.yaml_files_content[self.manifest_yaml])
        ok, error = self.build_product(start_time)
        if not ok:
            self.raise_error(error)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="执行组件和manifest构建")
    parser.add_argument("-i", "--input", help="输入的配置文件")
    parser.add_argument("-o", "--output", help="输出的日志文件")
    args, _ = parser.parse_known_args()
    BuildToolConfig.check_tools_name()
    BuildToolConfig.check_env()
    ManifestBuilder(args).run()