//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { traduction } from '@/utils/language';
import { IRule } from '@/model/Service/vnc-interface';
import { PWD_VALIDATOR } from '@/model/special-str';
import { MIN_CHARTYPE_2, SPACE_SPECIAL_CHAR } from '@/utils/regular';
import { getStoreData } from '@/utils/composition';
import {
  strVerification,
  passwordVerification,
  lengthVerification,
  matchRuleVerification,
  containsEmptyStringVerification,
} from '@/validators/validation-functions';

function getOrdinary(setting?: IRule): IRule {
  const rule = {
    type: 'password',
    required: false,
    minLength: 1,
    maxLength: 8,
    containsEmptyString: true,
    validator: strVerification,
    message: traduction('VALIDTOR_FORMAT_ERROR'),
    tipText: traduction('VALIDATOR_PWD_STANDARD', [1, 8]),
    trigger: 'blur',
  };

  if (setting) {
    Object.assign(rule, setting);
  }
  return rule;
}

// 长度验证
export function strLengthVerification(setting?: IRule): IRule {
  const rule = {
    // 是否展示前面的验证正确或者错误的icon
    icon: true,
    // 长度区间
    rangeSize: [8, 8],
    // 验证方法
    validator: lengthVerification,
    // 提示文本
    label: traduction('COMMON_VNC_MUST_CONTAIN'),
    // 验证结果
    state: false,
  };

  if (setting) {
    Object.assign(rule, setting);
  }
  return rule;
}

// 至少包含多少种字符验证
export function atLeastContainVerification(setting?: IRule): IRule {
  const rule = {
    icon: true,
    // 匹配规则
    matchRule: MIN_CHARTYPE_2,
    // 至少要满足的条目
    minCharType: 2,
    validator: matchRuleVerification,
    label: traduction('COMMON_SPECIAL_CHARACTERS'),
    state: false,
  };
  if (setting) {
    Object.assign(rule, setting);
  }
  return rule;
}

// 至少包含一个空格或以下特殊字符：`~!@#$%^&*()-_=+\\|[{}];:'\",<.>/?  验证
export function containSpecialStrVerification(setting?: IRule): IRule {
  const rule = {
    icon: true,
    matchRule: SPACE_SPECIAL_CHAR,
    validator: matchRuleVerification,
    label: traduction('VALIDATOR_PWD_VALIDATOR3') + PWD_VALIDATOR,
    state: false,
  };
  if (setting) {
    Object.assign(rule, setting);
  }
  return rule;
}

// 包含空格验证
export function noSpaceVerification(setting?: IRule): IRule {
  const rule = {
    icon: true,
    // 是否可以包含空格
    containsEmptyString: false,
    validator: containsEmptyStringVerification,
    label: traduction('COMMON_CANNOT_CONTAIN_SPACES'),
    state: false,
  };
  if (setting) {
    Object.assign(rule, setting);
  }
  return rule;
}

// 安全程度验证
export function securityVerification(setting?: IRule): IRule {
  const rule = {
    icon: false,
    // 是否开启安全等级 指示器
    security: true,
    // 安全等级描述 强/中/弱
    securityDescription: '',
    // 指示器的背景颜色
    circleAndLineStyles: [],
    validator: () => {},
    label: traduction('COMMON_SECURITY_LEVEL'),
  };
  if (setting) {
    Object.assign(rule, setting);
  }
  return rule;
}

export function regVerification(reg: string, type: string): any {
  let label = '';
  switch (type) {
    case 'local': {
      label = traduction('SECURITY_LOCAL_USER_TIP');
      break;
    }
    case 'vnc': {
      label = traduction('SECURITY_VNC_TIP');
      break;
    }
    case 'snmp': {
      label = traduction('SECURITY_SNMP_TIP');
      break;
    }
    default: break;
  }
  const rule = {
    icon: true,
    matchRule: reg,
    validator: matchRuleVerification,
    label,
    state: false,
  };
  return rule;
}

function getSpecial(setting?: IRule): IRule {
  const vncPasswordRulePolicy = getStoreData('glob', 'vncPasswordRulePolicy');
  const vncPasswordPattern = getStoreData('glob', 'vncPasswordPattern');
  let verificationList: IRule[] = [];
  if (vncPasswordRulePolicy === 'Customized') {
    verificationList = [
      strLengthVerification(),
      regVerification(vncPasswordPattern, 'vnc'),
      securityVerification(),
    ];
  } else if (vncPasswordRulePolicy === 'Hybrid') {
    verificationList = [
      strLengthVerification(),
      atLeastContainVerification(),
      containSpecialStrVerification(),
      regVerification(vncPasswordPattern, 'vnc'),
      securityVerification(),
    ];
  } else {
    verificationList = [
      strLengthVerification(),
      atLeastContainVerification(),
      containSpecialStrVerification(),
      securityVerification(),
    ];
  }

  const rule = {
    type: 'password',
    required: false,
    containsEmptyString: true,
    verificationList: verificationList,
    // 自定义校验规则
    validator: passwordVerification,
    message: traduction('VALIDTOR_FORMAT_ERROR'),
    trigger: 'blur',
  };

  if (setting) {
    Object.assign(rule, setting);
  }
  return rule;
}

// 设置密码检查规则 flag：是否启用密码检查(用户&安全-安全配置-密码检查)
export function initRules(flag: boolean) {
  if (flag) {
    return getSpecial();
  }
  return getOrdinary();
}