//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import $http from '@/utils/http-service';
import { IPort, IPortListItem } from '@/model/service-interface';
import { UI_REST_SERVICE_PORT } from '@/api/api';
import { traduction } from '@/utils/language';
import { IResponse } from '@/model/base-interface';

export function getPortInfo(): Promise<IPortListItem[]> {
  return new Promise<IPortListItem[]>((resolve, reject) => {
    const dataArr: IPortListItem[] = [];
    $http
      .get(UI_REST_SERVICE_PORT)
      .then((res: IResponse) => {
        const data: IPort = res.data;
        // V3接口变更，不支持的属性也会在接口中存在，只是取值为null, 因此不显示的属性需过滤掉为null的
        Object.keys(data).forEach(key => {
          if (data[key] === null) {
            return;
          }

          const port1 = data[key]?.Port || null;
          const port2 = data[key]?.SparePort || null;
          const configInfo = getConfigInfo(key);
          if (Object.keys(configInfo).length > 0) {
            const idx = configInfo.priority;
            const _tempalate: IPortListItem = {
              name: key,
              tipState1: false,
              tipState2: false,
              label: traduction(configInfo.i18nKey),
              port1: port1,
              port1Edit: port1,
              port1Default: configInfo.defaultPort,
              port2: port2,
              port2Edit: port2,
              port2Default: configInfo.defaultPort2,
              enabled: data[key]?.Enabled,
              enabledEdit: data[key]?.Enabled,
              id1: configInfo.id1,
              id2: configInfo.id2,
              index: idx,
              alias: configInfo.alias,
            };
            if (key === 'IPMILANRMCP') {
              _tempalate.port2Default = configInfo.defaultPort2;
            }
            dataArr.push(_tempalate);
          }
        });
        // 对最终数据做排序，根据priority升序显示
        dataArr.sort((a: IPortListItem, b: IPortListItem) => {
          return a.index - b.index;
        });
        resolve(dataArr);
      })
      .catch();
  });
}

export function setPortInfo(params: any) {
  return new Promise((resolve, reject) => {
    $http.patch(UI_REST_SERVICE_PORT, params).then(
      res => {
        resolve(res.data);
      },
      error => {
        reject(error);
      },
    );
  });
}

const configInfoMap = {
  SSH: {
    priority: 1,
    id1: 'editSSH',
    id2: '',
    i18nKey: 'SERVICE_SSH',
    defaultPort: '22',
    alias: 'SSH',
  },
  NAT: {
    priority: 2,
    id1: 'editNAT',
    id2: '',
    i18nKey: 'SERVICE_NAT',
    defaultPort: '30022',
    alias: 'NAT',
  },
  SNMPAgent: {
    priority: 3,
    id1: 'editSNMPAgent',
    id2: '',
    i18nKey: 'SERVICE_SNMP_AGENT',
    defaultPort: '161',
    alias: 'SNMP',
  },
  KVM: {
    priority: 4,
    id1: 'editKVM',
    id2: '',
    i18nKey: 'SERVICE_KVM',
    defaultPort: '2198',
    alias: 'KVM',
  },
  VMM: {
    priority: 5,
    id1: 'editVMM',
    id2: '',
    i18nKey: 'SERVICE_VMM',
    defaultPort: '8208',
    alias: 'VMM',
  },
  Video: {
    priority: 6,
    id1: 'editVideo',
    id2: '',
    i18nKey: 'SERVICE_VIDEO',
    defaultPort: '2199',
    alias: 'VIDEO',
  },
  VNC: {
    priority: 7,
    id1: 'editVNC',
    id2: '',
    i18nKey: 'SERVICE_VNC',
    defaultPort: '5900',
    alias: 'VNC',
  },
  HTTP: {
    priority: 8,
    id1: 'editHTTP',
    id2: '',
    i18nKey: 'SERVICE_HTTP',
    defaultPort: '80',
    alias: 'HTTP',
  },
  HTTPS: {
    priority: 9,
    id1: 'editHTTPS',
    id2: '',
    i18nKey: 'SERVICE_HTTPS',
    defaultPort: '443',
    alias: 'HTTPS',
  },
  IPMILANRMCP: {
    priority: 10,
    id1: 'editIPMI',
    id2: 'editStandbyIPMI',
    i18nKey: 'SERVICE_IPMI',
    defaultPort: '623',
    defaultPort2: '664',
    alias: 'IPMI',
  },
  IPMILANRMCPPlus: {
    priority: 11,
    id1: '',
    id2: '',
    i18nKey: 'SERVICE_IPMI_RMCP',
    defaultPort: '',
    alias: 'IPMI1',
  },
};
function getConfigInfo(name: string) {
  let temp: { [key: string]: any } = {};
  if (Object.prototype.hasOwnProperty.call(configInfoMap, name)) {
    temp = configInfoMap[name];
  }
  return temp;
}
