//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.

export enum TpcmTabName {
    BOOT_METRICS = 'BOOT_METRICS',
    SECURE_BOOT = 'SECURE_BOOT',
    HARDWARE_ID = 'HARDWARE_ID',
}

export interface SecureBootEntry {
    signature: string;
    verificationResult: number;
    name: string;
    version: string;
}

export interface IdentityEntryDetails {
    actualDigest: string;
    actualIdentifier: string;
    name: string;
    standardDigest: string;
    standardIdentifier: string;
    status: number;
}

export interface IdentityEntry {
    comparedResult: number;
    details: IdentityEntryDetails[];
    name: string;
}

enum DetailStatus {
    NORMAL = 0,
    PLUGIN = 1,
    REMOVE = 2,
    UPDATE = 3,
}

const detailStatusText = new Map<DetailStatus, string>([
    [DetailStatus.NORMAL, 'NORMAL'],
    [DetailStatus.PLUGIN, 'PLUGIN'],
    [DetailStatus.REMOVE, 'REMOVE'],
    [DetailStatus.UPDATE, 'UPDATE'],
]);

export const detailKeyList = ['name', 'actualIdentifier', 'standardIdentifier', 'status', 'actualDigest', 'standardDigest'] as const;
export const miniViewDetailKeyList = ['name', 'status', 'actualIdentifier', 'standardIdentifier', 'actualDigest', 'standardDigest'] as const;

const detailLabel = new Map<keyof IdentityEntryDetails, string>([
    ['name', 'TPCM_DEVICE_POSITION'],
    ['status', 'TPCM_DEVICE_STATUS'],
    ['actualIdentifier', 'TPCM_ACTUAL_IDENTIFIER'],
    ['standardIdentifier', 'TPCM_STANDARD_IDENTIFIER'],
    ['actualDigest', 'TPCM_ACTUAL_DIGEST'],
    ['standardDigest', 'TPCM_STANDARD_DIGEST'],
]);

export function getDetailLabel(key: keyof IdentityEntryDetails): string {
    return detailLabel.get(key) as string;
}

export function getStateStyleClass(status: number): string {
    return status === DetailStatus.NORMAL ? 'primary' : 'danger';
}

function getStatusText(status: number): string {
    switch (status) {
        case DetailStatus.NORMAL:
        case DetailStatus.PLUGIN:
        case DetailStatus.REMOVE:
        case DetailStatus.UPDATE:
            return detailStatusText.get(status) as string;
        default:
            return 'FAILED';
    }
}

export function getStatusLabel(status: number): string {
    return `TPCM_DEVICE_STATUS_${getStatusText(status)}`;
}