//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { traduction } from '@/utils/language';
import { IRule } from '@/model/Service/vnc-interface';
import { PWD_VALIDATOR } from '@/model/special-str';
import { SPECIAL_CHAR } from '@/utils/regular';
import {
  strLengthVerification,
  atLeastContainVerification,
  containSpecialStrVerification,
  noSpaceVerification,
  securityVerification,
} from '@/validators/password-rules';
import { getStoreData } from '@/utils/composition';
import { regVerification } from '@/validators/password-rules';
import { strVerification, passwordVerification } from '@/validators/validation-functions';

// 没有开启 密码检查的时候的校验规则
const ordinaryRule = (minLength: number, maxLength: number, message: string): IRule => {
  return {
    type: 'password',
    required: false,
    minLength: minLength,
    maxLength: maxLength,
    containsEmptyString: true,
    validator: strVerification,
    message: message,
    trigger: 'blur',
  };
};

export const specialRule = (minLength: number, maxLength: number): IRule => {
  const snmpCommunityPasswordRulePolicy = getStoreData('glob', 'snmpCommunityPasswordRulePolicy');
  const snmpCommunityPasswordPattern = getStoreData('glob', 'snmpCommunityPasswordPattern');
  let verificationList: IRule[] = [];
  if (snmpCommunityPasswordRulePolicy === 'Customized') {
    verificationList = [
      strLengthVerification({
        rangeSize: [minLength, maxLength],
        label: traduction('VALIDATOR_PWD_VALIDATOR1', [minLength, maxLength]),
      }),
      regVerification(snmpCommunityPasswordPattern, 'snmp'),
      securityVerification(),
    ];
  } else if (snmpCommunityPasswordRulePolicy === 'Hybrid') {
    verificationList = [
      strLengthVerification({
        rangeSize: [minLength, maxLength],
        label: traduction('VALIDATOR_PWD_VALIDATOR1', [minLength, maxLength]),
      }),
      atLeastContainVerification(),
      containSpecialStrVerification({
        matchRule: SPECIAL_CHAR,
        label: traduction('COMMON_ALLOWING_CHARACTERS') + PWD_VALIDATOR,
      }),
      noSpaceVerification(),
      regVerification(snmpCommunityPasswordPattern, 'snmp'),
      securityVerification(),
    ];
  } else {
    verificationList = [
      strLengthVerification({
        rangeSize: [minLength, maxLength],
        label: traduction('VALIDATOR_PWD_VALIDATOR1', [minLength, maxLength]),
      }),
      atLeastContainVerification(),
      containSpecialStrVerification({
        matchRule: SPECIAL_CHAR,
        label: traduction('COMMON_ALLOWING_CHARACTERS') + PWD_VALIDATOR,
      }),
      noSpaceVerification(),
      securityVerification(),
    ];
  }
  return {
    type: 'password',
    required: false,
    verificationList: verificationList,
    validator: passwordVerification,
    message: traduction('VALIDTOR_FORMAT_ERROR'),
    trigger: 'blur',
  };
};

/**
 * 根据是否开启密码检查和是否开启超长口令初始化password类型输入框的校验规则
 * passwordCheck：是否启用密码检查(用户&安全-安全配置-密码检查)
 * longPasswordEnabled: 是否开启超长口令
 * Notice:
 *  开启密码检查功能时：
 *    若已启用超长口令，则团体名可设置为长度为16～32个字符的字符串。
 *    若已禁用超长口令，则团体名可设置为长度为8～32个字符的字符串。
 *    至少包含以下字符中的两种：大写字母：A～Z, 小写字母：a～z, 数字：0～9
 *    至少包含以下特殊字符：`~!@#$%^&*()-_=+\|[{}];:'",<.>/?
 *    不能包含空格。
 *  关闭密码检查功能时：
 *    若已启用超长口令，则团体名可设置为长度为16～32个字符的字符串，字符串不能包含空格。
 *    若已禁用超长口令，则团体名可设置为长度为1～32个字符的字符串，字符串不能包含空格
 */
export function initPasswordRules(passwordCheck: boolean, longPasswordEnabled: boolean): IRule {
  if (passwordCheck) {
    if (longPasswordEnabled) {
      return specialRule(16, 32);
    }
    return specialRule(8, 32);
  } else {
    if (longPasswordEnabled) {
      return ordinaryRule(16, 32, traduction('VALIDTOR_FORMAT_ERROR'));
    }
    return ordinaryRule(1, 32, traduction('VALIDTOR_FORMAT_ERROR'));
  }
}
