//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { traduction } from '@/utils/language';
import { initIalizingData } from './fdmFunctions';

const cpuLabels = [
  traduction('COMMON_NAME'),
  traduction('FDM_ALARM_MAIN_FREQUENCY'),
  traduction('FDM_ALARM_LIVE_TIME'),
  traduction('FDM_ALARM_EVENT_STATUS'),
  traduction('COMMON_MANUFACTURER'),
  traduction('FDM_CORECOUNT_THREADCOUNT'),
  traduction('FDM_ALARM_MODEL'),
  traduction('FDM_ALARM_L1L2L3_CACHE'),
  traduction('FDM_ALARM_PROCESSOR_ID'),
  traduction('OTHER_PART_NUMBER'),
  traduction('FDM_ALARM_OTHER_PARAMETERS'),
  'PPIN',
];

function getL1L2L3Cache(bodyData: any) {
  return (
    `${initIalizingData(bodyData.L1CacheKiB)}/${initIalizingData(bodyData.L2CacheKiB)}/${initIalizingData(bodyData.L3CacheKiB)}KB`
  );
}

// 加工cpu详情数据
export function createCPUData(bodyData: any, liveTime: string) {
  const list: any = [
    {
      value: initIalizingData(bodyData.DeviceName),
    },
    {
      value: initIalizingData(bodyData.FrequencyMHz) ? initIalizingData(bodyData.FrequencyMHz) + ' MHz' : '--',
    },
    {
      value: liveTime || '--',
    },
    {
      value: initIalizingData(bodyData.State),
    },
    {
      value: initIalizingData(bodyData.Manufacturer),
    },
    {
      value: (initIalizingData(bodyData.TotalCores)) + '/' + (initIalizingData(bodyData.TotalThreads)),
    },
    {
      value: initIalizingData(bodyData.Model),
    },
    {
      value: getL1L2L3Cache(bodyData),
    },
    {
      value: initIalizingData(bodyData.IdentificationRegisters),
    },
    {
      value: initIalizingData(bodyData.PartNumber),
    },
    {
      value: initIalizingData(bodyData.OtherParameters),
    },
    {
      value: initIalizingData(bodyData.SerialNumber),
    },
  ];
  list.forEach((item: any, index: number) => {
    item.label = cpuLabels[index];
  });
  return list;
}

const dimmLabels = [
  traduction('COMMON_NAME'),
  traduction('COMMON_SERIALNUMBER'),
  traduction('FDM_ALARM_LIVE_TIME'),
  traduction('FDM_ALARM_TYPE'),
  traduction('FDM_ALARM_VENDOR'),
  traduction('FDM_MIN_VOLTAGE'),
  traduction('FDM_LOCATION'),
  traduction('OTHER_PART_NUMBER'),
  traduction('COMMON_CAPACITY'),
  traduction('FDM_DATAWIDTH_BITS'),
  traduction('FDM_ALARM_MAIN_FREQUENCY'),
  traduction('FDM_TECHNOLOGY'),
  traduction('FDM_RANK_COL'),
];
export function createDIMM(bodyData: any, liveTime: string) {
  const list: any = [
    {
      value: initIalizingData(bodyData.DeviceName),
    },
    {
      value: initIalizingData(bodyData.SerialNumber),
    },
    {
      value: liveTime || '--',
    },
    {
      value: initIalizingData(bodyData.MemoryDeviceType),
    },
    {
      value: initIalizingData(bodyData.Manufacturer),
    },
    {
      value: initIalizingData(bodyData.MinVoltageMillivolt) ? bodyData.MinVoltageMillivolt + ' mV' : '--',
    },
    {
      value: initIalizingData(bodyData.Position),
    },
    {
      value: initIalizingData(bodyData.PartNumber),
    },
    {
      value: initIalizingData(bodyData.CapacityMiB) ? bodyData.CapacityMiB + ' MB' : '--',
    },
    {
      value: initIalizingData(bodyData.DataWidthBits) ? bodyData.DataWidthBits + ' bit' : '--',
    },
    {
      value: initIalizingData(bodyData.OperatingSpeedMhz) ? bodyData.OperatingSpeedMhz + ' MT/S' : '--',
    },
    {
      value: initIalizingData(bodyData.Technology),
    },
    {
      value: initIalizingData(bodyData.RankCount),
    },
  ];

  list.forEach((item: any, index: number) => {
    item.label = dimmLabels[index];
  });

  return list;
}

function getTmpData(rebuildState: any, bodyData: any) {
  let tmpRebuildState = '';
  if (rebuildState === 'DoneOrNotRebuilt') {
    tmpRebuildState = traduction('FDM_STOPPED');
  } else if (rebuildState === 'Rebuilding') {
    const rebuildProgress = initIalizingData(bodyData.RebuildProgress) ? bodyData.RebuildProgress : '';
    tmpRebuildState = `${traduction('FDM_RECONSTRUCTION')}  ${rebuildProgress}`;
  }

  let tmpIndicatorLED = '';
  if (bodyData.indicatorLED === 'Off') {
    tmpIndicatorLED = traduction('FDM_DISABLED');
  } else if (bodyData.indicatorLED === 'Blinking') {
    tmpIndicatorLED = traduction('FDM_ENABLED');
  } else if (!bodyData.indicatorLED) {
    tmpIndicatorLED = traduction('FDM_DISABLED');
  }

  let capacity = null;
  if (bodyData.CapacityBytes === 0 || bodyData.CapacityBytes) {
    capacity = bodyData.CapacityBytes / 1024 / 1024 / 1024 / 1024;
    capacity = `${capacity.toFixed(3)} TB`;
  }

  const patrolState = bodyData.PatrolState;
  let tmpPatrolState = '';
  if (patrolState === 'DoneOrNotPatrolled') {
    tmpPatrolState = traduction('FDM_STOPPED');
  } else if (patrolState === 'Patrolling') {
    tmpPatrolState = traduction('FDM_CHECKING');
  }

  return { tmpRebuildState, tmpIndicatorLED, capacity, tmpPatrolState };
}

const diskLabels = [
  traduction('FDM_ALARM_LIVE_TIME'),
  traduction('FDM_ALARM_MODEL'),
  traduction('FDM_INTERFACE_TYPE'),
  traduction('COMMON_FIXED_VERSION'),
  traduction('FDM_ALARM_VENDOR'),
  traduction('FDM_TEMPERATURE'),
  traduction('COMMON_SERIALNUMBER'),
  traduction('FDM_SAS_ADDRESS0'),
  traduction('FDM_MEDIA_TYPE'),
  traduction('COMMON_CAPACITY'),
  traduction('FDM_SAS_ADDRESS1'),
  traduction('FDM_NEGOTIATED_SPEEDGBS'),
  traduction('FDM_SUPPORTED_RATE'),
  traduction('FDM_PATROL_STATE'),
  traduction('FDM_POWER_STATUS'),
  traduction('FDM_FIRMWARE_STATUS'),
  traduction('FDM_HOT_STATUS'),
  traduction('FDM_TOTAL_POWER_TIME'),
  traduction('FDM_LOCATION_STATUS'),
  traduction('FDM_CONSTRUCTION_STATE'),
  traduction('WEAR_RATE'),
  traduction('OTHER_PART_NUMBER'),
  traduction('COMMON_RESOURCE_OWNER'),
];
const diskValues = (
  liveTime: string,
  bodyData: any,
  capacity: any,
  tmpPatrolState: string,
  tmpIndicatorLED: string,
  tmpRebuildState: string,
) => {
  return [
    liveTime,
    initIalizingData(bodyData.Model),
    initIalizingData(bodyData.Protocol),
    initIalizingData(bodyData.Revision),
    initIalizingData(bodyData.Manufacturer),
    initIalizingData(bodyData.TemperatureCelsius, ' ℃'),
    initIalizingData(bodyData.SerialNumber),
    initIalizingData(bodyData.SASAddress[0]),
    initIalizingData(bodyData.MediaType),
    capacity,
    initIalizingData(bodyData.SASAddress[1]),
    initIalizingData(bodyData.NegotiatedSpeedGbs, ' Gbps'),
    initIalizingData(bodyData.CapableSpeedGbs, ' Gbps'),
    tmpPatrolState,
    initIalizingData(bodyData.PowerState),
    initIalizingData(bodyData.FirmwareStatus),
    initIalizingData(bodyData.HotspareType),
    initIalizingData(bodyData.HoursOfPoweredUp, ' h'),
    tmpIndicatorLED,
    tmpRebuildState,
    initIalizingData(bodyData.PredictedMediaLifeLeftPercent, '%'),
    initIalizingData(bodyData.PartNum),
    initIalizingData(bodyData.AssociatedResource),
  ];
};
export function createDisk(bodyData: any, liveTime: string) {
  const rebuildState = initIalizingData(bodyData.RebuildState);
  const { tmpRebuildState, tmpIndicatorLED, capacity, tmpPatrolState } = getTmpData(rebuildState, bodyData);
  const list: any = [];
  const values: any = diskValues(liveTime, bodyData, capacity, tmpPatrolState, tmpIndicatorLED, tmpRebuildState);
  diskLabels.forEach((label: string, index: number) => {
    list.push({
      label,
      value: values(index) || '--',
    });
  });
  return list;
}

export function createPSU(bodyData: any, liveTime: string, deviceInfoUrl: string) {
  return [
    {
      label: traduction('FDM_ALARM_LIVE_TIME'),
      value: liveTime || '--',
    },
    {
      label: traduction('COMMON_FIXED_VERSION'),
      value: initIalizingData(bodyData.FirmwareVersion),
    },
    {
      label: traduction('FDM_SLOT'),
      value: initIalizingData(bodyData.MemberId + 1),
    },
    {
      label: traduction('FDM_RATED_POWER'),
      value: initIalizingData(bodyData.PowerCapacityWatts) ? bodyData.PowerCapacityWatts + ' W' : '--',
    },
    {
      label: traduction('FDM_ALARM_VENDOR'),
      value: initIalizingData(bodyData.Manufacturer),
    },
    {
      label: traduction('FDM_INPUT_MODE'),
      value: initIalizingData(bodyData.PowerSupplyType),
    },
    {
      label: traduction('FDM_TYPE'),
      value: initIalizingData(bodyData.Model),
    },
    {
      label: traduction('OTHER_PART_NUMBER'),
      value: initIalizingData(bodyData.PartNumber),
    },
    {
      label: traduction('COMMON_SERIALNUMBER'),
      value: initIalizingData(bodyData.SerialNumber),
    },
  ];
}

export function createFans(bodyData: any, liveTime: string) {
  let nameFan = '';
  if (bodyData.Speed.indexOf('/') > 0) {
    nameFan =
      `${traduction('FDM_FAN_NAME1')}${bodyData.Name.substr(3, bodyData.Name.length - 1)} ${traduction('FAN_LOCATION_NAME')}`;
  } else {
    nameFan = traduction('FDM_FAN_NAME1') + bodyData.Name.substr(3, bodyData.Name.length - 1);
  }
  return [
    {
      label: traduction('COMMON_NAME'),
      value: nameFan || '--',
    },
    {
      label: traduction('OTHER_PART_NUMBER'),
      value: initIalizingData(bodyData.PartNumber),
    },
    {
      label: traduction('FDM_ALARM_LIVE_TIME'),
      value: liveTime || '--',
    },
    {
      label: traduction('FDM_ROTATIONAL_SPEED'),
      value: initIalizingData(bodyData.Speed),
    },
    {
      label: traduction('FDM_ALARM_MODEL'),
      value: initIalizingData(bodyData.Model),
    },
    {
      label: traduction('FAN_HEAT_RATE'),
      value: initIalizingData(bodyData.SpeedRatio),
    },
  ];
}

export function createDiskBP(bodyData: any, liveTime: string) {
  return [
    {
      label: traduction('FDM_ALARM_LIVE_TIME'),
      value: liveTime || '--',
    },
    {
      label: traduction('FDM_ALARM_TYPE'),
      value: initIalizingData(bodyData.Description),
    },
    {
      label: traduction('FDM_LOCATION'),
      value: initIalizingData(bodyData.Location),
    },
    {
      label: traduction('COMMON_ALARM_PCB_VERSION'),
      value: initIalizingData(bodyData.PCBVersion),
    },
    {
      label: traduction('FDM_ALARM_VENDOR'),
      value: initIalizingData(bodyData.Manufacturer),
    },
    {
      label: traduction('COMMIN_CPLD_VERSION'),
      value: initIalizingData(bodyData.CPLDVersion),
    },
    {
      label: traduction('COMMON_NUMBER'),
      value: initIalizingData(bodyData.CardNo),
    },
    {
      label: traduction('OTHER_BOARD_ID'),
      value: initIalizingData(bodyData.BoardId),
    },
  ];
}

export function createRaid(bodyData: any, liveTime: string) {
  return [
    {
      label: traduction('COMMON_NAME'),
      value: initIalizingData(bodyData.ProductName),
    },
    {
      label: traduction('FDM_ALARM_EVENT_LEVEL'),
      value: initIalizingData(bodyData.SupportedRAIDLevels),
    },
    {
      label: traduction('FDM_ALARM_LIVE_TIME'),
      value: liveTime || '--',
    },
    {
      label: traduction('COMMON_ALARM_PCB_VERSION'),
      value: initIalizingData(bodyData.PCBVersion),
    },
    {
      label: traduction('FDM_LOCATION'),
      value: initIalizingData(bodyData.Location),
    },
    {
      label: traduction('COMMIN_CPLD_VERSION'),
      value: initIalizingData(bodyData.CPLDVersion),
    },
    {
      label: traduction('FDM_ALARM_VENDOR'),
      value: initIalizingData(bodyData.Manufacturer),
    },
    {
      label: traduction('OTHER_BOARD_ID'),
      value: initIalizingData(bodyData.BoardId),
    },
    {
      label: traduction('COMMON_NUMBER'),
      value: initIalizingData(bodyData.CardNo),
    },
    {
      label: traduction('COMMON_RESOURCE_OWNER'),
      value: initIalizingData(bodyData.AssociatedResource),
    },
    {
      label: traduction('FDM_ALARM_TYPE'),
      value: initIalizingData(bodyData.Description),
    },
  ];
}

export function createPcieCard(bodyData: any, liveTime: string) {
  return [
    {
      label: traduction('COMMON_NAME'),
      value: initIalizingData(bodyData.ProductName),
    },
    {
      label: traduction('FDM_VENDOR_ID'),
      value: initIalizingData(bodyData.VendorId),
    },
    {
      label: traduction('FDM_ALARM_LIVE_TIME'),
      value: liveTime || '--',
    },
    {
      label: traduction('FDM_DEVICEID'),
      value: initIalizingData(bodyData.DeviceId),
    },
    {
      label: traduction('COMMON_DESCRIPTION'),
      value: initIalizingData(bodyData.Description),
    },
    {
      label: traduction('OTHER_SON_VENDOR_ID'),
      value: initIalizingData(bodyData.SubsystemVendorId),
    },
    {
      label: traduction('FDM_ALARM_VENDOR'),
      value: initIalizingData(bodyData.Manufacturer),
    },
    {
      label: traduction('FDM_SUBSYSTEM_ID'),
      value: initIalizingData(bodyData.SubsystemId),
    },
    {
      label: traduction('FDM_SLOT'),
      value: initIalizingData(bodyData.PcieCardSlot),
    },
    {
      label: traduction('COMMON_RESOURCE_OWNER'),
      value: initIalizingData(bodyData.AssociatedResource),
    },
  ];
}

export function createLom(bodyData: any, liveTime: string) {
  return [
    {
      label: traduction('COMMON_NAME'),
      value: initIalizingData(bodyData.Name),
    },
    {
      label: traduction('NET_CHIP_MODEL'),
      value: initIalizingData(bodyData.Model),
    },
    {
      label: traduction('FDM_ALARM_LIVE_TIME'),
      value: liveTime || '--',
    },
    {
      label: traduction('COMMON_MANUFACTURER'),
      value: initIalizingData(bodyData.CardManufacturer),
    },
    {
      label: traduction('FDM_ALARM_MODEL'),
      value: initIalizingData(bodyData.CardModel),
    },
    {
      label: traduction('COMMON_ALARM_PCB_VERSION'),
      value: initIalizingData(bodyData.PCBVersion),
    },
    {
      label: traduction('OTHER_BOARD_ID'),
      value: initIalizingData(bodyData.BoardId),
    },
    {
      label: traduction('NET_CHIP_VENDOR'),
      value: initIalizingData(bodyData.Manufacturer),
    },
  ];
}

export function createRiserCard(bodyData: any, liveTime: string) {
  return [
    {
      label: traduction('COMMON_NAME'),
      value: initIalizingData(bodyData.Name),
    },
    {
      label: traduction('FDM_ALARM_TYPE'),
      value: initIalizingData(bodyData.Description),
    },
    {
      label: traduction('FDM_ALARM_LIVE_TIME'),
      value: liveTime || '--',
    },
    {
      label: traduction('COMMON_ALARM_PCB_VERSION'),
      value: initIalizingData(bodyData.PCBVersion),
    },
    {
      label: traduction('FDM_ALARM_VENDOR'),
      value: initIalizingData(bodyData.Manufacturer),
    },
    {
      label: traduction('OTHER_BOARD_ID'),
      value: initIalizingData(bodyData.BoardId),
    },
    {
      label: traduction('FDM_SLOT'),
      value: initIalizingData(bodyData.Slot),
    },
  ];
}

const mainBoardLabels = [
  traduction('FDM_ALARM_LIVE_TIME'),
  traduction('OTHER_BOARD_ID'),
  traduction('INFO_MAIN_BOARD_IBMC_VERSION'),
  traduction('FDM_ALARM_MAINBORD_VENDOR'),
  traduction('IBMC_FDM_ALARM_BIOS_VERSION'),
  traduction('COMMON_ALARM_PCB_VERSION'),
  traduction('COMMIN_CPLD_VERSION'),
  traduction('FDM_PCH_VERSION'),
  traduction('FDM_UBOOT_VERSION'),
  traduction('COMMIN_CPLD_VERSION'),
  traduction('FDM_MAINBOARD_MODEL'),
  traduction('FDM_BY_UBOOT_VERSION'),
  traduction('FDM_MAINBOARD_SN'),
  traduction('OTHER_PART_NUMBER'),
];
export function createMainboard(bodyData: any, liveTime: string) {
  const list: any = [
    {
      value: liveTime || '--',
    },
    {
      value: initIalizingData(bodyData.BoardId),
    },
    {
      value: initIalizingData(bodyData.PMEVersion),
    },
    {
      value: initIalizingData(bodyData.Manufacturer),
    },
    {
      value: initIalizingData(bodyData.BiosVersion),
    },
    {
      value: initIalizingData(bodyData.PCBVersion),
    },
    {
      value: initIalizingData(bodyData.UbootVersion),
    },
    {
      value: initIalizingData(bodyData.CPLDVersion),
    },
    {
      value: initIalizingData(bodyData.PchModel),
    },
    {
      value: initIalizingData(bodyData.UbootVersion),
    },
    {
      value: initIalizingData(bodyData.BoardModel),
    },
    {
      value: initIalizingData(bodyData.UbootBackupVer),
    },
    {
      value: initIalizingData(bodyData.SerialNumber),
    },
    {
      value: initIalizingData(bodyData.PartNumber),
    },
  ];
  list.forEach((item: any, index: number) => {
    item.label = mainBoardLabels[index];
  });
  return list;
}

export function createChassis(liveTime: string) {
  return [
    {
      label: traduction('COMMON_NAME'),
      value: 'chassis',
    },
    {
      label: traduction('FDM_ALARM_LIVE_TIME'),
      value: liveTime || '--',
    },
  ];
}
