//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { findAimClassDom, findAimlabelingDom } from '@/utils/utils';
import { nextTick } from 'vue';

/**
 * 功能：给标签添加id
 * 注册一个全局自定义指令 `v-addId`
 * 注意：
 * 1. v-addId 注意传一个字符串进去
 * 2. 添加类型修饰符，比如v-addId.select
 *    所有的修饰符：select（下拉框）、input(输入框)、checkbox（多选框）、radio（单选框）、 pagination（分页）、表格（table）
 * 3. 如果是下拉框，需要添加 popper-append-to-body 或者 teleported 为false(推荐使用teleported，不然控制台会报警告)
 *    这样自定义指令才能获取下拉列表的el-popper标签
 */
export function addIdDirective(app: any) {
  app.directive('addId', {
    mounted(el: any, binding: any) {
      // 新版本的element-plus会自己添加系统id，这个地方需要nextTick一下才能设置生效
      nextTick(() => {
        mountedDoing(el, binding);
      });
    },
    updated(el: any, binding: any) {
      const id = binding.value;
      Object.keys(binding.modifiers).forEach(key => {
        switch (key) {
          case 'pagination':
            nextTick(() => {
              paginationAction(id, el);
            });
            break;
          default:
            break;
        }
      });
    },
  });
}

function mountedDoing(el: any, binding: any) {
  const id = binding.value;
  Object.keys(binding.modifiers).forEach(key => {
    switch (key) {
      case 'select':
        selectAction(id, el);
        break;
      case 'input':
        inputAction(id, el);
        break;
      case 'checkbox':
        checkboxAction(id, el);
        break;
      case 'radio':
        radioAction(id, el);
        break;
      case 'table':
        tableAction(id, el);
        break;
      case 'pagination':
        paginationAction(id, el);
        break;
      default:
        break;
    }
  });
}

// 下拉框
function selectAction(id: string, el: any) {
  if (id) {
    // select根元素上加传入的id
    el.setAttribute('id', id);
    // 输入框加id
    findAimClassDom(el, 'el-input__inner')?.setAttribute('id', `${id}_dominator_input`);
    // 如果没有设置teleported 为false，无法找到要加id的ul标签
    const elPopper = findAimClassDom(el, 'el-popper');
    if (elPopper) {
      findAimClassDom(elPopper, 'el-select-dropdown__list')?.setAttribute(
        'id',
        `${id}_droplist_list_list`,
      );
    }
  }
}

// 输入框
function inputAction(id: string, el: any) {
  if (id) {
    el.setAttribute('id', `${id}Box`);
    // 输入框加id
    findAimClassDom(el, 'el-input__inner')?.setAttribute('id', id);
    // 前缀的搜索盒子
    findAimClassDom(el, 'el-input__prefix')?.setAttribute('id', `${id}_search`);
    // 后缀的清空和报错盒子
    findAimClassDom(el, 'el-input__suffix')?.setAttribute('id', `${id}_clear`);
  }
}

// 多选框
function checkboxAction(id: string, el: any) {
  if (id) {
    el.setAttribute('id', `${id}_checkbox`);
    findAimClassDom(el, 'el-checkbox__label')?.setAttribute('id', `${id}_label`);
    findAimClassDom(el, 'el-checkbox__original')?.setAttribute('id', id);
  }
}

// 单选框
function radioAction(id: string, el: any) {
  if (id) {
    el.setAttribute('id', `${id}_radio`);
    findAimClassDom(el, 'el-radio__label')?.setAttribute('id', `${id}_label`);
    findAimClassDom(el, 'el-radio__original')?.setAttribute('id', id);
  }
}

// 表格
function tableAction(id: string, el: any) {
  const bodyWrapper = findAimClassDom(el, 'el-table__body-wrapper');
  if (bodyWrapper) {
    findAimlabelingDom(bodyWrapper, 'tbody')[0]?.setAttribute('id', id);
  }
}

// 分页器
function paginationAction(id: string, el: any) {
  el?.setAttribute('id', id);
  findAimClassDom(el, 'el-pagination__total')?.setAttribute('id', `${id}Total`);
  paginationSelectAction(id, el);
  paginationBtnPrevAndNextAction(id, el);
  paginationPagerAction(id, el);
  paginationJumpAction(id, el);
}

// 分页器的下拉框
function paginationSelectAction(id: string, el: any) {
  const sizesDom = findAimClassDom(el, 'el-pagination__sizes');
  if (sizesDom) {
    sizesDom.setAttribute('id', `${id}Sizes`);
    findAimClassDom(sizesDom, 'el-select')?.setAttribute('id', `${id}Select`);
    findAimClassDom(sizesDom, 'select-trigger')?.setAttribute('id', `${id}Trigger`);
    findAimClassDom(sizesDom, 'el-input')?.setAttribute('id', `${id}Box`);
    findAimClassDom(sizesDom, 'el-input__inner')?.setAttribute('id', `${id}Input`);
    findAimClassDom(sizesDom, 'el-input__suffix')?.setAttribute('id', `${id}Suffix`);
    findAimClassDom(sizesDom, 'el-input__suffix-inner')?.setAttribute('id', `${id}SuffixInner`);
    findAimClassDom(sizesDom, 'el-select__caret')?.setAttribute('id', `${id}Caret`);
    findAimClassDom(sizesDom, 'el-select__caret')?.children[0]?.setAttribute('id', `${id}Svg`);
  }
}

// 分页器的前一页和后一页的箭头
function paginationBtnPrevAndNextAction(id: string, el: any) {
  findAimClassDom(el, 'btn-prev')?.setAttribute('id', `${id}Prev`);
  findAimClassDom(el, 'btn-next')?.setAttribute('id', `${id}Next`);
}

// 快速跳转的数字和“...”
function paginationPagerAction(id: string, el: any) {
  findAimClassDom(el, 'el-pager')?.setAttribute('id', `${id}Pager`);
}

// 分页器的跳转到指定页面的输入框
function paginationJumpAction(id: string, el: any) {
  const jumpDom = findAimClassDom(el, 'el-pagination__jump');
  if (!jumpDom) {
    return;
  }
  jumpDom.setAttribute('id', `${id}Jump`);
  findAimClassDom(jumpDom, 'el-pagination__editor')?.setAttribute('id', `${id}Editor`);
  findAimClassDom(jumpDom, 'el-input__inner')?.setAttribute('id', `${id}EditorInput`);
}
