-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local singleton = require 'mc.singleton'
local class = require 'mc.class'
local signal = require 'mc.signal'
local log = require 'mc.logging'
local context = require 'mc.context'
local initiator = require 'mc.initiator'
local common_def = require 'common_def'
local add_event = require 'add_event'
local MAX_TEMP <const> = 60
local Hysteresis <const> = 2
local drives_tmp_list = {}

local c_drives_object = class()

function c_drives_object:ctor(storage_app_service)
    self.storage_app_service = storage_app_service
    self.obj = nil
    self.db = storage_app_service.db
    self.on_update_drive_temperature = signal.new()
end

function c_drives_object:assert_max_temp(temp)
    if temp > MAX_TEMP then
        -- 如果已存在过温告警， 不产生新告警
        if not self.last_temp_alert_data.IsAlerted then
            self.last_temp_alert_data.TemperatureCelsius = temp
            self.last_temp_alert_data.IsAlerted = true
            return 'true', tostring(temp)
        end
    else
        -- 如果已存在告警并且温度变化大于迟滞量， 恢复告警并更新temperature
        if self.last_temp_alert_data.IsAlerted and temp < MAX_TEMP - Hysteresis then
            self.last_temp_alert_data.TemperatureCelsius = temp
            self.last_temp_alert_data.IsAlerted = false
            return 'false', tostring(self.last_temp_alert_data.TemperatureCelsius)
        end
    end
end

function c_drives_object:get_max_temp(drive_obj, node_id)
    if node_id then
        drives_tmp_list[node_id] = drive_obj.temperature
    end

    local max_temp = 0
    for _, v in pairs(drives_tmp_list) do
        max_temp = max_temp < v and v or max_temp
    end

    return max_temp
end

local function get_op_initiator()
    local m_initiator = initiator.new('N/A', 'N/A', 'localhost')
    local ctx = context.get_context()
    return (ctx and not ctx:is_empty()) and ctx:get_initiator() or m_initiator
end

local function drives_monitor_task(self)
    self.obj.property_changed:on(function(name, value)
        if name == 'LogAutoCollectEnable' then
            log:operation(get_op_initiator(), 'storage',
                'Set Drives log auto collect enable to %s successfully', value)
        end
        if name == 'LogAutoCollectInterval' then
            log:operation(get_op_initiator(), 'storage',
                'Set Drives log auto collect interval to %ss successfully', value)
        end
    end)
end

function c_drives_object:init()
    if not self.obj then
        self.obj = self.storage_app_service:CreateDrives(1)
        self.obj.MaxTemperatureCelsius = 0
    end

    drives_monitor_task(self)
    self.last_temp_alert_data = self.db.DrivesTableInfo({Id = 'DrivesMaxTemp'})
    self.on_update_drive_temperature:on(function (presence, drive_obj, node_id)
        if not drive_obj or not drive_obj.temperature or not self.obj or not presence then
            return
        end
        if presence == 1 and drive_obj.temperature ~= common_def.INVALID_U8 then
            self.obj.MaxTemperatureCelsius = self:get_max_temp(drive_obj, node_id)
            local alert_res, alert_temp = self:assert_max_temp(self.obj.MaxTemperatureCelsius)
            if alert_temp then
                add_event.get_instance().generate_drive_max_temp(alert_res, alert_temp)
            end
            self.last_temp_alert_data:save()
        end
    end)
end

return singleton(c_drives_object)