-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local bs = require 'mc.bitstring'
local ipmi = require 'ipmi'
local comp_code = ipmi.types.Cc
local error_engine = require 'error_engine'
local error_code = error_engine.SML_ERR_CODE_E
local ctrl_option = {
    option1_format = bs.new([[<<
    no_read_ahead_opt:1,
    read_ahead_opt:1,
    read_policy_opt_reserved:3,
    read_policy_default:2,
    read_policy_support:1,

    write_through_opt:1,
    write_back_opt:1,
    write_back_if_bad_bbu_opt:1,
    write_policy_opt_reserved:2,
    write_policy_default:2,
    write_policy_support:1,

    cached_io_opt:1,
    direct_io_opt:1,
    io_policy_opt_reserved:3,
    io_policy_default:2,
    io_policy_support:1,

    access_rw_opt:1,
    access_read_only_opt:1,
    access_blocked_opt:1,
    access_policy_opt_reserved:2,
    access_policy_default:2,
    access_policy_support:1
    >>]]),
    option2_format = bs.new([[<<
    pd_cache_unchanged_opt:1,
    pd_cache_enable_opt:1,
    pd_cache_disable_opt:1,
    pd_cache_opt_reserved:2,
    pd_cache_policy_default:2,
    pd_cache_policy_support:1,

    reserved0:8,

    ctrl_temp_src_internal:1,
    ctrl_support_crypto_erase:1,
    ctrl_support_epd:1,
    ctrl_support_jbod_state:1,
    reserved1:4,

    ctrl_support_mode_set:1,
    ctrl_support_raid:1,
    ctrl_support_jbod:1,
    ctrl_support_hba:1,
    ctrl_support_mixed:1,
    reserved2:3
    >>]]),
    option3_format = bs.new([[<<
    configured_drive_wcp:2,
    unconfigured_drive_wcp:2,
    hba_drive_wcp:2,
    reserved0:2,

    raid0_support_opt:1,
    raid1_support_opt:1,
    raid5_support_opt:1,
    raid6_support_opt:1,
    raid10_support_opt:1,
    raid50_support_opt:1,
    raid60_support_opt:1,
    raid1adm_support_opt:1,

    raid10adm_support_opt:1,
    raid1triple_support_opt:1,
    raid10triple_support_opt:1,
    reserved1:13
    >>]])
}

local pd_log_defs = {
    SML_PD_LOG_TYPE_E = {
        PD_LOG_SATA_SMART_ATTRIBUTE = 0,
        PD_LOG_SATA_SMART_ATTRIBUTE_THRESHOLD = 1,
        PD_LOG_SATA_ERROR = 2,
        PD_LOG_SATA_EXTENT_ERROR = 3,
        PD_LOG_SATA_EXTENDED_SELF_TEST = 4,
        PD_LOG_SATA_PHY_EVENT = 5,
        PD_LOG_SATA_SEAGATE_GLIST = 6,
        PD_LOG_SATA_SEAGATE_PLIST = 7,
        PD_LOG_SATA_SEAGATE_CRITICAL_EVENT = 8,
        PD_LOG_SATA_SEAGATE_FARM = 9,
        PD_LOG_SAS_SELF_TEST = 100,
        PD_LOG_SAS_WRITE_ERROR_CNT = 101,
        PD_LOG_SAS_READ_ERROR_CNT = 102,
        PD_LOG_SAS_VERIFY_ERROR_CNT = 103,
        PD_LOG_SAS_NONMEDIUM_ERROR_CNT = 104,
        PD_LOG_SAS_PHY_ERROR_CNT = 105,
        PD_LOG_SAS_INTERNAL_EXCEPTION = 106,
        PD_LOG_SAS_TEMPERATURE = 107,
        PD_LOG_SAS_BACKGORUN_MEDIA_SCAN = 108,
        PD_LOG_SAS_GLIST = 109,
        PD_LOG_SAS_GENERAL_STATISTICS = 110,
        PD_LOG_SAS_HUAWEI_SPECIFIC32H = 111,
        PD_LOG_SAS_HUAWEI_SPECIFIC34H = 112,
        PD_LOG_SAS_HUAWEI_SPECIFIC35H = 113,
        PD_LOG_SAS_HUAWEI_IO_INFO = 114,
        PD_LOG_SAS_KIOXIA_SSD_SPECIFIC = 115,
        PD_LOG_SAS_KIOXIA_EOPT = 116,
        PD_LOG_SAS_KIOXIA_SMART_DATA = 117
    },

    PD_LOG_MAX_RAW_SIZE = {
        PD_LOG_SAS_ERR_COUNT_LOG_MAX_BYTES = 512,
        PD_LOG_SAS_GLIST_MAX_BYTES = 2048,
        PD_LOG_SAS_IE_LOG_MAX_BYTES = 512,
        PD_LOG_SAS_SELF_TEST_MAX_BYTES = 404,

        PD_LOG_SATA_SMART_ATTR_MAX_SECTORS = 1,
        PD_LOG_SATA_EXTENT_ERROR_LOG_MAX_SECTORS = 5,
        PD_LOG_SATA_EXTENDED_SELF_TEST_MAX_SECTORS = 5,
        PD_LOG_SATA_ERROR_LOG_MAX_SECTORS = 1,
        PD_LOG_SATA_PHY_EVENT_LOG_MAX_SECTORS = 1,
        PD_LOG_SATA_SEAGATE_GLIST_MAX_SECTORS = 5,
        PD_LOG_SATA_SEAGATE_PLIST_MAX_SECTORS = 1,
        PD_LOG_SATA_SEAGATE_CR_EVENT_MAX_SECTORS = 20,
        PD_LOG_SATA_SEAGATE_FARM_MAX_SECTORS = 192,
    }
}

local pd_smart_defs = {
    PD_VENDOR_NAME_INTEL = 'INTEL',
    PD_VENDOR_NAME_SAMSUNG = 'SAMSUNG',
    PD_VENDOR_NAME_HUAWEI = 'HUAWEI',
    PD_VENDOR_NAME_AJ = 'UMIS',

    VENDOR_ATTR_ID = {
        ['INTEL'] = 233,
        ['SAMSUNG'] = 177,
        ['MICRON'] = 202,
        ['SANDISK'] = 245,
        ['SEAGATE'] = 0,
        ['FUJITSU'] = 0,
        ['WDC'] = 0,
        ['HGST'] = 0,
        ['SMI'] = 231,
        ['LITEON ER2'] = 231,
        ['LITEON CVZ'] = 231,
        ['LITEON EGT'] = 245,
        ['LITEON'] = 245,
        ['PHISON'] = 231,
        ['SSSTC'] = 231,
        ['HYNIX'] = 231,
        ['HUAWEI'] = 231,
        ['YANGTZE MEMORY'] = 245,
        ['RAMAXEL'] = 231,
        ['UMIS'] = 231,
        ['DERA'] = 248,
        ['DATSSD'] = 233,
        ['HUADIAN'] = 232
    },

    ONE_HUNDRED_PERCENT = 100,

    NAND_WRITTEN_ATTR_ID = {
        ['INTEL'] = 243,
        ['SAMSUNG'] = 251,
        ['HUAWEI'] = 245,
    },

    HOST_WRITTEN_ATTR_ID = {
        ['INTEL'] = 241,
        ['SAMSUNG'] = 241,
        ['HUAWEI'] = 241,
    },

    PERCENT_70 = 70,
    PERCENT_50 = 50,
    PERCENT_30 = 30,

    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_HOST_WRITTEN_L = 0x05,    -- 用户区写入量低位
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_HOST_WRITTEN_H = 0x06,    -- 用户区写入量高位
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_TOTAL_SPARE_BLOCK = 0x31, -- 用户区出厂时冗余块总数
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_VALID_SPARE_BLOCK = 0x33, -- 用户区当前有效冗余块个数
    HW_DEFINED_SMART_ATTRIBUTE_ID_SLC_TOTAL_SPARE_BLOCK = 0x3A, -- 非用户区出厂时冗余块总数
    HW_DEFINED_SMART_ATTRIBUTE_ID_SLC_VALID_SPARE_BLOCK = 0x3C, -- 非用户区当前有效冗余块个数
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_USED_LIFESPAN = 0x4E,     -- 用户区根据P/E情况评估的已使用寿命
    HW_DEFINED_SMART_ATTRIBUTE_ID_SLC_USED_LIFESPAN = 0x4F,     -- 非用户区根据P/E情况评估的已使用寿命
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_NAND_WRITTEN_L = 0x52,    -- 用户区nand flash写入量低位
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_NAND_WRITTEN_H = 0x53,    -- 用户区nand flash写入量高位
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_PE_CYCLE = 0x5A,          -- 用户区擦除次数规格值
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_ERASE_COUNT = 0x5B,       -- 用户区擦除次数(最大,最小,平均)
    HW_DEFINED_SMART_ATTRIBUTE_ID_TLC_WEAROUT = 0x5C,           -- 用户区剩余磨损
    HW_DEFINED_SMART_ATTRIBUTE_ID_SLC_PE_CYCLE = 0x67,          -- 非用户区擦除次数规格值
    HW_DEFINED_SMART_ATTRIBUTE_ID_SLC_ERASE_COUNT = 0x68,       -- 非用户区擦除次数(最大,最小,平均)
    HW_DEFINED_SMART_ATTRIBUTE_ID_SLC_WEAROUT = 0x69            -- 非用户区剩余磨损
}

local common_def = {
    CTRL_OPTION = ctrl_option,
    INVALID_STRING  = 'N/A',
    INVALID_U8      = 0xff,
    INVALID_U16     = 0xffff,
    INVALID_U32     = 0xffffffff,
    INVALID_BOOLEAN = 0,
    INVALID_NUM2STR = '255.0',
    RET_ERR = -1,
    RET_OK = 0,
    SML_ERR_INVALID_PARAMETER = 0x1003,
    SML_ERR_REBOOT_REQUIRED = 0x1074,
    SML_ERR_NULL_INFTERFACE = 0x10C2,
    SML_ERR_CTRL_STATUS_INVALID = 0x11D5,
    SML_ERR_CTRL_DUPLICATE_REGISTERED = 0x1102,
    SML_ERR_CTRL_OPERATION_NOT_SUPPORT = 0x11D7,
    SML_ERR_I2C_READ_WRITE_FAILED = 0x1001,
    SML_ERR_PD_STATE_UNSUPPORTED_TO_SET = 0x131f,
    SML_MAX_RAID_CONTROLLER = 8,
    SML_ASYNC_OPERATION_OFFSET = 0x80,

    DRIVE_TITLE = 'Disk',
    PHYSCIAL_DRIVE_LOCATION = 'HDDPlane',

    LSI_3004_WITH_IMR = 12,
    PMC_3152_8I_SMART_RAID = 64,
    PMC_2100_8I_SMART_HBA = 65,
    HI1880_SP186_M_16i = 96,
    HI1880_SP186_M_8i = 113,

    BOOT_PRIORITY_PRIMARY = 1,
    BOOT_PRIORITY_ALL = 3,
    READ_POLICY_READ_AHEAD_ADAPTIVE = 2,
    SM_CODE_SUCCESS = 0x0,
    SML_ERR_DATA_INVALID = 0x10C8,
    SML_ERR_PD_CLEAR_IN_PROGRESS = 0x1325,
    SML_ERR_CONFIG_BLOCK_SIZE_NOT_SAME = 0x1451,
    SM_CODE_INVALID_RAID_LEVEL = 0x2080,
    SM_CODE_INVALID_SPAN_DEPTH = 0x2081,
    SM_CODE_INVALID_PD_COUNT = 0x2082,
    SM_CODE_INVALID_ARRAY_ID = 0x2083,
    SM_CODE_CAPACITY_OUT_OF_RANGE = 0x2086,
    SM_CODE_STRIP_SIZE_OUT_OF_RANGE = 0x2087,
    SM_CODE_READ_POLCIY_OUT_OF_RANGE = 0x2088,
    SM_CODE_WRITE_POLICY_OUT_OF_RANGE = 0x2089,
    SM_CODE_IO_POLICY_OUT_OF_RANGE = 0x208A,
    SM_CODE_ACCESS_POLICY_OUT_OF_RANGE = 0x208B,
    SM_CODE_DISK_CAHCE_POLICY_OUT_OF_RANGE = 0x208C,
    SM_CODE_INIT_TYPE_OUT_OF_RANGE = 0x208D,
    SM_CODE_LD_NAME_INVALID_ASCII = 0x208E,
    SM_CODE_LD_NAME_EXCEED_MAX_LEN = 0x208F,
    SM_CODE_PARA_DATA_ILLEGAL = 0x20CC,
    SM_CODE_PD_NOT_IDENTIFIED = 0x20A1,

    SL_LD_NAME_LENGTH = 15,
    SC_LD_NAME_LENGTH = 64,
    HS_LD_NAME_LENGTH = 20,

    STORAGE_INFO_INVALID_BYTE = 0xFF,
    STORAGE_INFO_INVALID_DWORD = 0xFFFFFFFF,
    SML_LD_CAPACITY_UNIT_MB = 0,
    SML_LD_CAPACITY_UNIT_GB = 1,
    SML_LD_CAPACITY_UNIT_TB = 2,

    LD_CACHE_NO_READ_AHEAD = 0,
    LD_CACHE_READ_AHEAD = 1,
    LD_CACHE_WRITE_THROUGH = 0,
    LD_CACHE_WRITE_BACK = 1,
    LD_CACHE_WRITE_CACHE_IF_BAD_BBU = 2,
    LD_CACHE_CACHED_IO = 0,
    LD_CACHE_DIRECT_IO = 1,
    LD_ACCESS_RW = 0,
    LD_ACCESS_READ_ONLY = 1,
    LD_ACCESS_BLOCKED = 2,
    PD_CACHE_UNCHANGED = 0,
    PD_CACHE_ENABLE = 1,
    PD_CACHE_DISABLE = 2,
    LD_INIT_STATE_BUTT = 7,

    VENDER_LSI = 0,
    VENDER_PMC = 1,
    VENDER_HUAWEI = 2,

    RAID_LEVEL_0 = 0,
    RAID_LEVEL_1 = 1,
    RAID_LEVEL_5 = 5,
    RAID_LEVEL_6 = 6,
    RAID_LEVEL_10 = 10,
    RAID_LEVEL_1ADM = 11,
    RAID_LEVEL_10ADM = 12,
    RAID_LEVEL_1TRIPLE = 13,
    RAID_LEVEL_10TRIPLE = 14,
    RAID_LEVEL_50 = 50,
    RAID_LEVEL_60 = 60,

    LSI_3908_WITH_MR = 14,
    LSI_3916_WITH_MR = 15,

    OOB_TYPE_OVER_I2C = 0,
    OOB_TYPE_OVER_PCIE = 1,
    OOB_TYPE_BUTT = 2,

    OOB_NOT_SUPPORT_LOSE = 0,   -- 0 OOB不支持 掉电丢失
    OOB_SUPPORT_LOSE = 1,       -- 1 OOB支持 掉电丢失
    OOB_NO_SUPPORT_SAVE = 2,    -- 2 OOB不支持 掉电不丢失
    OOB_SUPPORT_SAVE = 3,       -- 3 OOB支持 掉电不丢失

    INTERFACE_TYPE = {
        ['SPI'] = 0,
        ['SAS 3G'] = 1,
        ['SATA 1.5G'] = 2,
        ['SATA 3G'] = 3,
        ['SAS 6G'] = 4,
        ['SAS 12G'] = 5,
        ['SAS 32G'] = 6,
        ['SAS 64G'] = 7,
        ['SATA 32G'] = 8,
        ['SATA 64G'] = 9,
        ['unknown'] = 255
    },

    STRIP_SIZE_TABLE = {
        [512] = 0,
        [1024] = 1,
        [2 * 1024] = 2,
        [4 * 1024] = 3,
        [8 * 1024] = 4,
        [16 * 1024] = 5,
        [32 * 1024] = 6,
        [64 * 1024] = 7,
        [128 * 1024] = 8,
        [256 * 1024] = 9,
        [512 * 1024] = 10,
        [1024 * 1024] = 11,
        [2 * 1024 * 1024] = 12,
        [0xFFFFFFFF] = 255
    },

    LD_ACCELERATION_METHOD_LIST = {
        [0] = 'Unknown',
        [1] = 'None',
        [2] = 'ControllerCache',
        [3] = 'IOBypass',
        [4] = 'maxCache'
    },

    LD_STRIP_SIZE_STR = {
        [5] = '16K',
        [6] = '32K',
        [7] = '64K',
        [8] = '128K',
        [9] = '256K',
        [10] = '512K',
        [11] = '1M',
    },

    LD_CACHE_LINE_SIZE_STR = {
        [1] = '64 KB',
        [4] = '256 KB'
    },

    LD_RP_STR = {
        [0] = 'No Read Ahead',
        [1] = 'Read Ahead',
        [2] = 'Read Ahead Adaptive'
    },

    LD_WP_STR = {
        [0] = 'Write Through',
        [1] = 'Write Back with BBU',
        [2] = 'Write Back'
    },

    LD_IOP_STR = {
        [0] = 'Cached IO',
        [1] = 'Direct IO'
    },

    LD_AP_STR = {
        [0] = "Read Write",
        [1] = "Read Only",
        [2] = "Blocked",
        [3] = "Hidden"
    },

    LD_DCP_STR = {
        [0] = "Disk's Default",
        [1] = "Enabled",
        [2] = "Disabled"
    },

    LD_INIT_TYPE_STR = {
        [0] = "No Init",
        [1] = "Quick Init",
        [2] = "Full Init"
    },

    LD_ACCELERATOR_STR = {
        [0] = "Unknown",
        [1] = "None",
        [2] = "Controller Cache",
        [3] = "SSD I/O Bypass"
    },

    PD_STATE_STR = {
        [0] = "UNCONFIGURED GOOD",
        [1] = "UNCONFIGURED BAD",
        [2] = "HOT SPARE",
        [3] = "OFFLINE",
        [4] = "FAILED",
        [5] = "REBUILD",
        [6] = "ONLINE",
        [7] = "COPYBACK",
        [8] = "JBOD",
        [9] = "UNCONFIGURED(Shielded)",
        [10] = "HOT SPARE(Shielded)",
        [11] = "CONFIGURED(Shielded)",
        [12] = "FOREIGN",
        [13] = "ACTIVE",
        [14] = "STAND-BY",
        [15] = "SLEEP",
        [16] = "DST executing in background",
        [17] = "SMART Off-line Data Collection executing in background",
        [18] = "SCT command executing in background",
        [19] = "ONLINE(EPD)",
        [20] = "RAW",
        [21] = "READY",
        [22] = "NOT SUPPORTED",
        [23] = "PREDICTIVE FAILURE",
        [24] = "DIAGNOSING",
        [25] = "INCOMPATIBLE",
        [26] = "ERASE IN PROGRESS",
        [255] = "N/A"
    },

    PD_HOT_SPARE_STR = {
        [0] = "None",
        [1] = "Global",
        [2] = "Dedicated",
        [3] = "Autoreplace",
    },

    PD_BOOT_PRIORITY = {
        [0] = "None",
        [1] = "Primary",
        [2] = "Secondary",
        [3] = "All"
    },

    PD_BOOTABLE_DESC = {
        [0] = "non-bootable",
        [1] = "bootable"
    },

    READ_POLICY_LIST = {
        [0] = 'NoReadAhead',
        [1] = 'ReadAhead',
        [2] = 'ReadAheadAdaptive'
    },

    AS_BOOT_DEVICE = 'as boot device',

    WRITE_PLOICY_LIST = {
        [0] = 'WriteThrough',
        [1] = 'WriteBackWithBBU',
        [2] = 'WriteBack'
    },

    IO_PLOICY_LIST = {
        [0] = 'CachedIO',
        [1] = 'DirectIO'
    },

    ACCESS_POLICY_LIST = {
        [0] = 'ReadWrite',
        [1] = 'ReadOnly',
        [2] = 'Blocked',
        [3] = 'Hidden'
    },
    DRIVE_CACHE_POLICY_LIST = {
        [0] = 'Unchanged',
        [1] = 'Enabled',
        [2] = 'Disabled'
    },

    LOCATE = 1,                            -- 点灯
    STOP_LOCATE = 0,                       -- 关灯
    TURN_ON_LOCATE = 1,                    -- 开定位灯 1表示闪烁 2表示常亮
    TURN_OFF_LOCATE = 0,                   -- 关定位灯
    FAULT = 1,                             -- 点故障灯
    STOP_FAULT = 0,                        -- 关故障灯
    FW_UG = 0,                             -- 固件状态为UG
    FW_ONLINE = 6,                         -- 固件状态为online
    FW_OFFLINE = 3,
    FW_SYSTEM = 8,
    FW_FOREIGN = 12,
    CANCEL_HOTSPARE = 0,                   -- 取消全局热备
    GLOBAL_HOTSPARE = 1,                   -- 全局热备状态
    DEDICATED_HOTSPARE = 2,                -- 局部热备状态
    AUTO_REPLACE_HOTSPARE = 3,             -- 自动替换热备盘
    MANUFACTURE_ID = 0x0007DB,             -- 小字节序厂商ID

    CONTROLLER_INTRERFACE = {
        'bmc.kepler.Systems.Storage.Controller.ConsistencyCheck',
        'bmc.kepler.Systems.Storage.Controller.VolumeManage',
        'bmc.kepler.Systems.Storage.Controller.ControllerStatus',
        'bmc.kepler.LifeCycle.Retirement'
    },

    STORAGE_MGNT_STATE = {
        STORAGE_MGNT_NOT_READY = 0,        -- 0-初始状态，上电未完成
        STORAGE_MGNT_READY = 1,            -- 1-上电已完成，且已经获取过一遍控制器信息、硬盘信息、逻辑盘信息
        STORAGE_MGNT_OOB_NOT_SUPPORT = 2,  -- 2-上电已完成，且已经判断当前不支持带外管理
        STORAGE_MGNT_CTRL_ABNOMAL = 3,     -- 3-上电已完成，且在获取控制器信息过程中出现异常（该异常不包括I2C通信失败）
        STORAGE_MGNT_NO_CTRLS = 4,         -- 4-上电已完成，且已经判断当前不存在RAID卡
        STORAGE_MGNT_BUTT = 5
    },

    DRIVE_PROTOCOL = {
        UNKNOWN = 0,
        SCSI = 1,
        SAS = 2,
        SATA = 3,
        FC = 4
    },
    DRIVE_PROTOCOL_STR = {
        [0] = "Unknown",
        [1] = "Parallel SCSI",
        [2] = "SAS",
        [3] = "SATA",
        [4] = "Fiber Channel",
        [5] = 'SATA/SAS',
        [6] = 'PCIe'
    },

    PD_LOG_DEF = pd_log_defs,

    PD_STATE = {
        UNCONFIGURED_GOOD   = 0,
        UNCONFIGURED_BAD    = 1,
        HOT_SPARE           = 2,
        OFFLINE             = 3,
        FAILED              = 4,
        REBUILD             = 5,
        ONLINE              = 6,
        COPYBACK            = 7,
        SYSTEM              = 8,
        SHIELD_UNCONFIGURED = 9,
        SHIELD_HOT_SPARE    = 10,
        SHIELD_CONFIGURED   = 11,
        FOREIGN             = 12,
        ACTIVE              = 13,
        STANDBY             = 14,
        SLEEP               = 15,
        DST                 = 16,
        SMART               = 17,
        SCT                 = 18,
        EPD                 = 19,
        RAW                 = 20,
        READY               = 21,
        NOT_SUPPORTED       = 22,
        PREDICTIVE_FAILURE  = 23,
        DIAGNOSING          = 24,
        INCOMPATIBLE        = 25,
        ERASING             = 26,
        UNKNOWN             = 255
    },

    LD_STATE = {
        OFFLINE = 0,
        PARTIALLY_DEGRADED = 1,
        DEGRADED = 2,
        OPTIMAL = 3,
        FAILED = 4,
        NOT_CONFIGURED = 5,
        INTERIM_RECOVERY = 6,
        READY_FOR_RECOVERY = 7,
        RECOVERYING = 8,
        WRONG_DRIVE_REPLACED = 9,
        DRVIE_IMPROPERLY_CONNECTED = 10,
        EXPANDING = 11,
        NOT_YET_AVAILABLE = 12,
        QUEUED_FOR_EXPANSION = 13,
        DISABLED_FROM_SCSIID_CONFLICT = 14,
        EJECTED = 15,
        ERASE_IN_PROGRESS = 16,
        UNUSED = 17,
        READY_TO_PERFORM_PREDICTIVE_SPARE_REBUILD = 18,
        RPI_IN_PROGRESS = 19,
        RPI_QUEUED = 20,
        ENCRYPTED_VOLUME_WITHOUT_KEY = 21,
        ENCRYPTION_MIGRATION = 22,
        ENCRYPTED_VOLUME_REKEYING = 23,
        ENCRYPTED_VOLUME_ENCRYPTION_OFF = 24,
        VOLUME_ENCODE_REQUESTED = 25,
        ENCRYPTED_VOLUME_REKEY_REQUESTED = 26,
        UNSUPPORTED_ON_THIS_CONTROLLER = 27,
        NOT_FORMATTED = 28,
        FORMATTING = 29,
        SANITIZING = 30,
        INITIALIZING = 31,
        INITIALIZEFAIL = 32,
        DELETING = 33,
        DELETEFAIL = 34,
        WRITE_PROTECT = 35,
        UNKNOWN = 0xFF,
    },

    LD_STATE_DESC = {
        [0] = 'Offline',
        'Partially degraded',
        'Degraded',
        'Optimal',
        'Failed',
        'Not Configured',
        'Interim recovery',
        'Ready for recovery',
        'Recovering',
        'Wrong drive replaced',
        'Drive improperly connected',
        'Expanding',
        'Not available',
        'Queued for expansion',
        'Disabled from SCSI conflict',
        'Ejected',
        'Erase in progress',
        'Unused',
        'Ready to perform predictive spare rebuild',
        'RPI in progress',
        'RPI queued',
        'Encrypted without key',
        'Encryption migration',
        'Encrypted & re-keying',
        'Encrypted with encryption off',
        'Volume encode requested',
        'Encrypted with re-key request',
        'Unsupported',
        'Not Formatted',
        'Formatting',
        'Sanitizing',
        'Initializing',
        'Initialize fail',
        'Deleting',
        'Delete fail',
        'Write protect'
    },

    RAID_LEVEL_DESC = {
        [0] = 'RAID0',
        [1] = 'RAID1',
        [5] = 'RAID5',
        [6] = 'RAID6',
        [10] = 'RAID10',
        [50] = 'RAID50',
        [60] = 'RAID60',
        [11] = 'RAID1(ADM)',
        [12] = 'RAID10(ADM)',
        [13] = 'RAID1Triple',
        [14] = 'RAID10Triple',
        [17] = 'RAID1(1E)'
    },

    FC__PUBLIC_OK = 'SVR-0000000',
    FC_STORAGE_NO_IDENTIFY = 'SVR-0080003',
    FC_STORAGE_PHYSICAL_FAULT = 'SVR-0080004',
    FC_STORAGE_LOGICAL_FAULT = 'SVR-0080005',
    FC_STORAGE_RAID_BBU = 'SVR-0080011',
    FC_STORAGE_RAID_BBU_REPLACE = 'SVR-0080012',
    FC_STORAGE_LINK_PHY_ERROR = 'SVR-0080021',

    NVME_VPD_VENDOR_ID_FOR_SAMSUNG_PM1733 = 0x144d0001,
    NVME_VPD_VENDOR_ID_FOR_SAMSUNG_PM983 = 0x144d0002,
    NVME_VPD_VENDOR_ID_FOR_MEMBLAZE_D520 = 0x1c5f0001,
    NVME_VPD_VENDOR_ID_FOR_MEMBLAZE_D920 = 0x1c5f0002,
    NVME_VPD_VENDOR_ID_FOR_MEMBLAZE_D526 = 0x1c5f0003,
    NVME_VPD_VENDOR_ID_FOR_MEMBLAZE_D926 = 0x1c5f0004,
    NVME_VPD_VENDOR_ID_FOR_MEMBLAZE_D6530_D6536 = 0x1c5f0005,
    NVME_VPD_VENDOR_ID_FOR_DERA_D5427_D5447 = 0x1d780001,
    NVME_VPD_VENDOR_ID_FOR_INTEL = 0x8086,
    NVME_VPD_VENDOR_ID_FOR_INTEL_P4600 = 0x80860001,
    NVME_VPD_VENDOR_ID_FOR_INTEL_P5800X = 0x80860002,
    NVME_VPD_VENDOR_ID_FOR_SAMSUNG_PM1735 = 0x144d0003,
    NVME_VPD_VENDOR_ID_FOR_SAMSUNG_PM1725 = 0x144d0004,
    NVME_VPD_VENDOR_ID_FOR_SAMSUNG_PM9A3 = 0x144d0005,
    NVME_VPD_VENDOR_ID_FOR_MEMBLAZE_7940 = 0x1c5f0007,
    NVME_VPD_VENDOR_ID_FOR_MEMBLAZE = 0x1c5f,
    NVME_VPD_VENDOR_ID_FOR_SHANNON = 0x1cb0,
    NVME_VPD_VENDOR_ID_FOR_MICRON = 0x1344,
    NVME_VPD_VENDOR_ID_FOR_DERA = 0x1d78,
    NVME_VPD_VENDOR_ID_FOR_SAMSUNG = 0x144d,
    NVME_VPD_VENDOR_ID_FOR_HUAWEI = 0x19e5,

    NVME_MANUFACTURER_INTEL = 'INTEL',
    NVME_MANUFACTURER_HUAWEI = 'HUAWEI',
    NVME_MANUFACTURER_SAMSUNG = 'SAMSUNG',
    NVME_MANUFACTURER_MEMBLAZE = 'MEMBLAZE',
    NVME_MANUFACTURER_SHANNON = 'SHANNON',
    NVME_MANUFACTURER_MICRON = 'MICRON',
    NVME_MANUFACTURER_WESTDIGI = 'WESTDIGI',
    NVME_MANUFACTURER_RAMAXEL = 'RAMAXEL',
    NVME_MANUFACTURER_DERA = 'DERA',
    NVME_MANUFACTURER_DAPUSTOR = 'DAPUSTOR',
    NVME_MANUFACTURER_UMIS = 'UMIS',
    NVME_MANUFACTURER_KIOXIA = 'KIOXIA',
    NVME_MANUFACTURER_STARBLZ = 'STARBLZ',

    MANUFACTURE_ID_MAP = {
        ['HUAWEI'] = 0x19e5,
        ['INTEL'] = 0x8086,
        ['WESTDIGI'] = 0x1b96,
        ['RAMAXEL'] = 0x1e81,
        ['DAPUSTOR'] = 0x1e3b,
        ['UMIS'] = 0x1cc4,
        ['KIOXIA'] = 0x1e0f,
        ['SAMSUNG'] = 0x144d,
        ['STARBLZ'] = 0x9d32,
        ['MEMBLAZE'] = 0x1c5f
    },

    -- Multi Record公共头长度
    MULTIRECORD_COMMON_HEADER_LEN = 5,
    MULTIRECORD_LAST_MASK = 0x80,

    -- Topology MultiRecord域中Element Descriptor内各信息相对偏移
    ELEMENT_DESCRIPTOR_LEN_OFFSET = 2,
    NVM_SUBSYSTEM_PORT_COUNT_OFFSET = 5,
    NVM_SUBSYSTEM_FIRST_PORT_LINK_SPEED_OFFSET = 8,

    -- 兼容MI1.0协议，1.0协议中RECORD LENGTH字段的取值有歧义，不同厂商写入值不一致，这里使用固定值
    NVME_MULTIRECORD_FIXED_LEN = 0x40,
    NVME_PCIE_PORT_MULTIRECORD_FIXED_LEN = 0x10,

    -- Topology MultiRecord域各信息相对偏移
    TOPOLOGY_MULTIRECORD_ELEMENT_COUNT_OFFSET = 7,
    TOPOLOGY_MULTIRECORD_FIRST_ELEMENT_OFFSET = 8,

    -- NVMe PCIe Port MultiRecord域数据偏移
    PCIE_PORT_MULTIRECORD_LINK_SPEED = 8,
    PCIE_PORT_MULTIRECORD_LINK_WIDTH = 9,
    MULTIRECORD_INFO_CAPACITY_OFFSET = 24,

    MULTIRECORD_TYPE_NVME = 0x0b,
    MULTIRECORD_TYPE_NVME_PCIE_PORT = 0x0c,
    MULTIRECORD_TYPE_TOPOLOGY = 0x0d,

    RECORD_ITEM_CAPACITY = 8,
    RECORD_ITEM_LINK_SPEED = 9,
    RECORD_ITEM_LINK_WIDTH = 10,

    RECORD_ITEM_MANUFACTURER_NAME = 0,
    RECORD_ITEM_PRODUCT_NAME = 1,
    RECORD_ITEM_MODEL_NUMBER = 2,
    RECORD_ITEM_PRODUCT_VERSION = 3,
    RECORD_ITEM_SERIAL_NUMBER = 4,

    MAX_MULTIRECORD_NUM = 10,
    NVME_VPD_PROTOCOL_NVME_MI = 0,
    NVME_VPD_PROTOCOL_SSD_FORM_FACTOR = 1,
    SAMSUNG_NVME_VPD_PROTOCOL_SSD_FORM_FACTOR = 2,
    NVME_VPD_STATE_ERROR = 1,
    PRODUCT_INFO_MANUFACTURER_LEN_OFFSET = 3,
    COMMON_HEADER_PRODUCT_AREA_OFFSET = 4,
    MAX_AREA_OFFSET_IN_HEADER = 31,
    NVME_MI_AREA_OFFSET_MULTIPLE = 8,
    NVME_COMMON_HEADER_LEN = 8,
    COMMON_HEADER_MULTI_RECORD_AREA_OFFSET = 5,

    MULTIRECORD_INFO_CAPACITY_LEN = 13,

    BIT_MEDIA_ERR = (1 << 0),
    BIT_PREDICTIVE_FAIL = (1 << 1),
    BIT_OTHER_ERR = (1 << 2),
    BIT_IO_DETERIORATION = (1 << 3),
    BIT_DRIVE_NOT_READY = (1 << 4),
    BIT_DRIVE_FUNCTIONAL = (1 << 5),
    BIT_RESET_NOT_REQUIRED = (1 << 6),
    BIT_CODE_PORT0_PCIE_LINK_ACTIVE = (1 << 7),

    NVME_MI_PCIE_GEN1_OFFSET = 0,
    NVME_MI_PCIE_GEN2_OFFSET = 1,
    NVME_MI_PCIE_GEN3_OFFSET = 2,
    NVME_MI_PCIE_GEN4_OFFSET = 3,
    NVME_MI_PCIE_GEN5_OFFSET = 4,
    NVME_MI_PCIE_GEN1_SPEED = 2.5,
    NVME_MI_PCIE_GEN2_SPEED = 5.0,
    NVME_MI_PCIE_GEN3_SPEED = 8.0,
    NVME_MI_PCIE_GEN4_SPEED = 16.0,
    NVME_MI_PCIE_GEN5_SPEED = 32.0,

    ELEMENT_TYPE_RESERVED = 0,
    ELEMENT_TYPE_EXTENDED = 1,
    ELEMENT_TYPE_UPSTREAM_CONNECTOR = 2,
    ELEMENT_TYPE_EXPANSION_CONNECTOR = 3,
    ELEMENT_TYPE_LABEL = 4,
    ELEMENT_TYPE_SMBUS_I2C_MUX = 5,
    ELEMENT_TYPE_PCIE_SWITCH = 6,
    ELEMENT_TYPE_NVM_SUBSYSTEM = 7,

    ELEMENT_TYPE_VENDOR_SPECIFIC_BEGIN = 240,
    ELEMENT_TYPE_VENDOR_SPECIFIC_END = 255,

    CTRL_AP_LOG_TYPE = 0,
    CTRL_IMU_LOG_TYPE = 1,
    CTRL_AP_LOG_INDEX = 2,
    CTRL_IMU_LOG_INDEX = 3,
    CTRL_AP_LASTWORD_TYPE = 4,
    CTRL_DUMP_FILE_TYPE = 5,
    CTRL_FLASH_DUMP_FILE_TYPE = 6,
    CTRL_NAND_LOG0_FILE_TYPE = 7,
    CTRL_NAND_LOG1_FILE_TYPE = 8,
    CTRL_LOG_LAST_CRASH_DUMP = 9,
    CTRL_LOG_SERIAL_OUTPUT = 10,
    CTRL_LOG_TYPE_BUTT = 0xFF,

    CTRL_LOG_BASE_PATH = "/data/var/log/storage/ctrllog",

    AP_FILE_BIN = "ap.bin",
    IMU_FILE_BIN = "imu.bin",
    AP_INDEX_FILE_GZ = "ap_index.gz",
    IMU_INDEX_FILE_GZ = "imu_index.gz",
    AP_LASTWORD_FILE_BIN = "lastword.bin",
    DUMP_FILE_BIN = "dump.bin",
    FLASH_DUMP_FILE_BIN = "flash_dump.bin",
    NAND_LOG0_FILE_BIN = "0nandlog.bin",
    NAND_LOG1_FILE_BIN = "1nandlog.bin",
    PD_PHYSICAL_SPEED = {
        [0] = "Undefined",
        [1] = "1.5 Gbps",
        [2] = "3.0 Gbps",
        [3] = "6.0 Gbps",
        [4] = "12.0 Gbps",
        [5] = "2.5 Gbps",
        [6] = "5.0 Gbps",
        [7] = "8.0 Gbps",
        [8] = "10.0 Gbps",
        [9] = "16.0 Gbps",
        [10] = "20.0 Gbps",
        [11] = "30.0 Gbps",
        [12] = "32.0 Gbps",
        [13] = "40.0 Gbps",
        [14] = "64.0 Gbps",
        [15] = "80.0 Gbps",
        [16] = "96.0 Gbps",
        [17] = "128.0 Gbps",
        [18] = "160.0 Gbps",
        [19] = "256.0 Gbps",
        [20] = "22.5 Gbps"
    },
    HEALTH_STATUS_STR = {
        [0] = 'Normal',
        [1] = 'Minor',
        [2] = 'Major',
        [3] = 'Critical'
    },
    POWER_STATE_STR = {
        [0] = 'Spun Up',
        [1] = 'Spun Down',
        [2] = 'Transition'
    },
    MEDIA_TYPE_STR = {
        [0] = 'HDD',
        [1] = 'SSD',
        [2] = 'SSM'
    },
    HOT_SPARE_TYPE_STR_NO_SPCAE = {
        [0] = "None",
        [1] = "Global",
        [2] = "Dedicated",
        [3] = "Auto replace"
    },
    PATROL_STATE_STR = {
        [0] = "No",
        [1] = "Yes"
    },
    IPMI_CODE = {
        [error_code.SML_SUCCESS] = comp_code.Success,
        [error_code.SML_ERR_REBOOT_REQUIRED] = comp_code.Success,
        [error_code.SML_ERR_CONFIG_INVALID_PD_SDD_HDD_MIXED] = comp_code.SensorInvalid,
        [error_code.SML_ERR_CONFIG_INVALID_PARAM_REPEATED_PD_ID] = comp_code.SensorInvalid,
        [error_code.SML_ERR_CONFIG_INVALID_PD_SDD_SAS_SATA_MIXED] = comp_code.SensorInvalid,
        [error_code.SML_ERR_CONFIG_INVALID_PARAM_ARRAY_REF] = comp_code.SensorInvalid,
        [error_code.SML_ERR_CONFIG_ARRAY_NUM_REACHED_LIMIT] = comp_code.ParmOutOfRange,
        [error_code.SML_ERR_CONFIG_ARRAY_SIZE_TOO_SMALL] = comp_code.ParmOutOfRange,
        [error_code.SML_ERR_CONFIG_INVALID_PARAM_SPAN_DEPTH] = comp_code.InvalidFieldRequest,
        [error_code.SML_ERR_CONFIG_INVALID_PARAM_NUM_DRIVE_PER_SPAN] = comp_code.InvalidFieldRequest,
        [error_code.SML_ERR_CONFIG_INVALID_PARAM_RAID_LEVEL] = comp_code.InvalidFieldRequest,
        [error_code.SML_ERR_LD_INVALID_TARGET_ID] = comp_code.InvalidFieldRequest,
        [error_code.SML_ERR_LD_DELETE_ON_SNAPSHOTS_ENABLED] = comp_code.CommandNotAvailable,
        [error_code.SML_ERR_CONFIG_INVALID_PD_IN_USE] = comp_code.CommandNotAvailable,
        [error_code.SML_ERR_CONFIG_TARGET_LD_ID_EXHAUSTED] = comp_code.CommandNotAvailable,
        [error_code.SML_ERR_CONFIG_INVALID_PD_NON_SUPPORTED] = comp_code.CommandNotAvailable,
        [error_code.SML_ERR_CONFIG_INVALID_PD_BOOT_SECTOR_FOUND] = comp_code.CommandNotAvailable,
        [error_code.SML_ERR_CONFIG_INVALID_PD_WITH_FOREIGN_CONFIG] = comp_code.CommandNotAvailable,
        [error_code.SML_ERR_LD_PROPERTY_SET_NOT_ALLOWED] = comp_code.CommandDisabled,
        [error_code.SML_ERR_LD_OPERATION_NOT_SUPPORT] = comp_code.CommandDisabled,
        [error_code.SML_ERR_CONFIG_INVALID_PD_SCSI_DEV_TYPE] = comp_code.CommandDisabled,
        [error_code.SML_ERR_CONFIG_OPERATION_NOT_SUPPORT] = comp_code.CommandDisabled,
        [error_code.SML_ERR_CTRL_OPERATION_NOT_SUPPORT] = comp_code.CommandDisabled
    },
    IPMI_CODE_DEFAULT = comp_code.Busy,

    -- SSD-FORM-FACTOR协议偏移量
    VPD_SSD_FORM_VENDOR_ID_OFFSET = 3,
    VPD_SSD_FORM_SN_OFFSET = 5,
    VPD_SSD_FORM_MODEL_NUMBER_OFFSET = 25,
    VPD_SSD_FORM_LINK_SPEED_OFFSET = 65,
    VPD_SSD_FORM_LINK_WIDTH_OFFSET = 66,

    -- SSD-FORM-FACTOR数据长度
    VPD_SSD_FORM_VENDOR_ID_LEN = 2,
    VPD_SSD_FORM_SN_LEN = 20,
    VPD_SSD_FORM_MODEL_NUMBER_LEN = 40,
    VPD_SSD_FORM_LINK_SPEED_LEN = 1,
    VPD_SSD_FORM_LINK_WIDTH_LEN = 1,

    PCI_EXPRESS_VERSION_GEN_ONE = 1,
    PCI_EXPRESS_VERSION_GEN_TWO = 2,
    PCI_EXPRESS_VERSION_GEN_THREE = 3,
    PCI_EXPRESS_VERSION_GEN_FOUR = 4,
    ORIGIN_MAX_SPEED_DEFAULT = 0,
    ORIGIN_MAX_SPEED_2P5G = 2.5,
    ORIGIN_MAX_SPEED_5G = 5.0,
    ORIGIN_MAX_SPEED_8G = 8.0,
    ORIGIN_MAX_SPEED_16G = 16.0,

    CAPABLE_SPEED_GBPS = {
        [5] = 2.5,
        [6] = 5.0,
        [7] = 8.0,
        [8] = 10.0,
        [9] = 16.0,
        [10] = 20.0,
        [11] = 30.0,
        [12] = 32.0,
        [13] = 40.0,
        [14] = 64.0,
        [15] = 80.0,
        [16] = 96.0,
        [17] = 128.0,
        [18] = 160.0,
        [19] = 256.0
    },

    SSD_FORM_CAPACITY_SAMSUNG_PM983_960 = 0xDF84758000,
    SSD_FORM_CAPACITY_SAMSUNG_PM983_1T9 = 0x1BF08EB0000,
    SSD_FORM_CAPACITY_SAMSUNG_PM983_3T8 = 0x37E11D60000,
    SSD_FORM_CAPACITY_SAMSUNG_PM983_7T6 = 0x6FC23AC0000,
    SSD_FORM_CAPACITY_MEMBLAZE_7T6 = 0x6FC23AC0000,
    NVME_VPD_PROTOCOL_SSD_FORM_VENDOR_ID_OFFSET_L = 4,
    NVME_VPD_PROTOCOL_SSD_FORM_VENDOR_ID_OFFSET_H = 5,

    ctrl_commu_normal = 0,
    ctrl_commu_loss = 1,

    UPDATE_NOT_SUPPORT = 0xfffffffe,

    SUBHEALTH_STR = {
        IO_DETERIORATION = 'IODeterioration',
        ESTIMATED_LIFESPAN = 'EstimatedLifespan'
    },

    RAID_CTRL_WCP_NAME = {
        [0] = 'Default',
        [1] = 'Enabled',
        [2] = 'Disabled'
    },

    RAID_CTRL_WCP_NUM = {
        ['Default'] = 0,
        ['Enabled'] = 1,
        ['Disabled'] = 2
    },
    ACTIVATION_MODE = {
        [0] = 'Failure',
        [1] = 'Predictive'
    },

    PERCENTAGE_UPPER_LIMIT = 100,

    PD_SMART_DEF = pd_smart_defs,

    -- 统一承载部分硬盘固件版本的特殊偏移
    SSD_FIRMWARE_OFFSET_TABLE = {
        [0x144d0005] = 33,
        [0x1c5f0007] = 97
    },

    U16_MAX = 0xffff,
    SIGNED_BIT = 0x8000
}

return common_def