-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local common_def = require 'common_def'
local class = require 'mc.class'
local singleton = require 'mc.singleton'

local array_dump = class()

function array_dump:ctor()
end

local function trans_capacity_size(capacity_size_mb)
    local capacity_basic_unit = 1024
    if capacity_size_mb == common_def.INVALID_U32 then
        return "N/A"
    elseif capacity_size_mb < capacity_basic_unit then
        return string.format("%s MB", capacity_size_mb)
    elseif capacity_size_mb < (capacity_basic_unit * capacity_basic_unit) then
        return string.format("%.3f GB", capacity_size_mb / capacity_basic_unit)
    elseif capacity_size_mb < (capacity_basic_unit * capacity_basic_unit * capacity_basic_unit) then
        return string.format("%.3f TB", capacity_size_mb / (capacity_basic_unit * capacity_basic_unit))
    else
        return string.format("%.3f PB",
            capacity_size_mb / (capacity_basic_unit * capacity_basic_unit * capacity_basic_unit))
    end
end

local function dump_list(fp_w, input_list)
    if not next(input_list) then
        fp_w:write("None")
        return
    end
    local drive_list = {}
    for _, drive_name in pairs(input_list) do
        if not drive_list[tostring(drive_name)] then
            drive_list[tostring(drive_name)] = true
            drive_list[#drive_list + 1] = string.match(drive_name, "%d+")
        end
    end
    for idx, drive_id in ipairs(drive_list) do
        fp_w:write(drive_id)
        if idx ~= #drive_list then
            fp_w:write(",")
        end
    end
end

function array_dump:dump_info(fp_w, array)
    if array.Id >= 0x8000 then -- 0x8000 1880卡span的起始Id
        return
    end

    fp_w:write("----------------------------------------------------------------------\n")
    fp_w:write(string.format("%-40s : %s\n", "Array ID", array.Id))
    fp_w:write(string.format("%-40s : %s\n", "Used Space", trans_capacity_size(array.UsedSpaceMiB)))
    fp_w:write(string.format("%-40s : %s\n", "Free Space", trans_capacity_size(array.TotalFreeSpaceMiB)))

    if array.RefVolumes then
        fp_w:write(string.format("%-40s : ", "Logcial Drive(s) ID"))
        dump_list(fp_w, array.RefVolumes)
        fp_w:write("\n")
    end

    if array.RefDrives then
        fp_w:write(string.format("%-40s : ", "Physical Drive(s) ID"))
        dump_list(fp_w, array.RefDrives)
        fp_w:write("\n")
    end
    fp_w:write("----------------------------------------------------------------------\n\n")
end

return singleton(array_dump)