-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: The unit test for sel common interfaces.
local lu = require 'luaunit'
local common = require 'sel.sel_common'

test_sel_common = {}

function test_sel_common.test_parse_sel_msg()
    local input = {0x04, 0x01, 0x30, 0x82, 0x09, 0xff, 0xff}
    for i = 1, #input do
        input[i] = string.char(input[i])
    end

    local msg = common.parse_sel_msg(table.concat(input, ''))
    lu.assertEquals(msg.version, 0x04)
    lu.assertEquals(msg.sensor_type, 0x01)
    lu.assertEquals(msg.event_type, 0x02)
    lu.assertEquals(msg.direction, 0x01)
end

function test_sel_common.test_pack_sel_msg()
    local msg = {
        EventMsgVersion = 0x04,
        SensorType = 0x01,
        SensorNumber = 0x30,
        EventType = 0x02,
        EventDir = 1,
        EventData1 = 0x09,
        EventData2 = 0xff,
        EventData3 = 0xff
    }

    local data = common.pack_sel_msg(msg)
    lu.assertEquals(string.byte(data, 1, 1), 0x04)
    lu.assertEquals(string.byte(data, 4, 4), 0x82)
    lu.assertEquals(string.byte(data, 5, 5), 0x09)
end

function test_sel_common.test_pack_sel_event_type()
    lu.assertEquals(common.pack_sel_event_type(3, 1), 0x83)
    lu.assertEquals(common.pack_sel_event_type(10, 0), 0x0A)
end

function test_sel_common.test_unpack_sel_event_type()
    local event_type, event_dir = common.unpack_sel_event_type(0x83)
    lu.assertEquals(event_type, 3)
    lu.assertEquals(event_dir, 1)
    event_type, event_dir = common.unpack_sel_event_type(0x0A)
    lu.assertEquals(event_type, 10)
    lu.assertEquals(event_dir, 0)
end

function test_sel_common.test_makeup_sel_datas()
    local record = {
        RecordId = 0x4142,
        SelType = 0x43,
        Timestamp = 0x44454647,
        GenerateId = 0x4849,
        SelMsgVersion = 0x4A,
        SensorType = 0x4B,
        SensorNumber = 0x4C,
        SelEventType = 0x4D,
        SelData1 = 0x4E,
        SelData2 = 0x4F,
        SelData3 = 0x50
    }
    local datas = common.makeup_sel_datas(record)
    local check = 'BACGFEDIHJKLMNOP'
    lu.assertEquals(datas, check)
end

local sel_base = {
    'RecordId',
    'Timestamp',
    'SensorName',
    'SensorType',
    'Level',
    'Status',
    'Description',
}
function test_sel_common.test_format_sel_list()
    local sel_1 = {
        RecordId = 0x0004,
        Timestamp = 0x66AC56CC,
        SensorName = 'DISK1',
        SensorType = 'Drive Slot / Bay',
        Level = 'Informational',
        Status = 'Asserted',
        Description = 'Drive Present',
    }
    local sel_2 = {
        RecordId = 0x0003,
        Timestamp = 0x66AC56CC,
        SensorName = 'DISK0',
        SensorType = 'Drive Slot / Bay',
        Level = 'Informational',
        Status = 'Asserted',
        Description = 'Drive Present',
    }
    local sel_3 = {
        RecordId = 0x0002,
        Timestamp = 0x66ABF1BD,
        SensorName = 'SysRestart',
        SensorType = 'System Boot Initiated',
        Level = 'Informational',
        Status = 'Asserted',
        Description = 'System Restart [Reset button][IPMB]',
    }
    local sel_4 = {
        RecordId = 0x0001,
        Timestamp = 0x66ABEF36,
        SensorName = 'Mngmnt Health',
        SensorType = 'Management Subsys Health',
        Level = 'Minor',
        Status = 'Asserted',
        Description = 'Sensor access degraded or unavailable',
    }
    local source = {sel_1, sel_2, sel_3, sel_4}
    local res
    -- 循环100次，判断首尾属性是否相同（事件是否乱序）
    for _ = 1, 100 do
        res = common.format_sel_list(sel_base, source)
        local base_start = res[1].MappingTable[1].Key
        local base_end = res[7].MappingTable[1].Key
        lu.assertEquals(res[8].MappingTable[1].Key, base_start)
        lu.assertEquals(res[15].MappingTable[1].Key, base_start)
        lu.assertEquals(res[22].MappingTable[1].Key, base_start)
        lu.assertEquals(res[7].MappingTable[1].Key, base_end)
        lu.assertEquals(res[14].MappingTable[1].Key, base_end)
        lu.assertEquals(res[21].MappingTable[1].Key, base_end)
    end
end

return test_sel_common