-- Copyright (c) 2025 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local json = require 'cjson'
local context = require 'mc.context'
local os = require 'os'

local FAN_POWER<const> = "FAN Power"
local INLET_TEMP<const> = "Inlet Temp"
local OUTLET_TEMP<const> = "Outlet Temp"
local SENSOR_TYPE_TEMPERATURE<const> = 1

local m = {}

function m.get_reset_time(time_stamp)
    return os.date('%Y-%m-%d %H:%M:%S', time_stamp)
end

function m.get_power_watts(id, sensor_list)
    local _, oid
    for _, obj in pairs(sensor_list) do
        if obj.SensorIdentifier == FAN_POWER then
            oid = '/redfish/v1/Chassis/' .. id .. '/Sensors/' ..
                string.format(string.gsub(obj.SensorName, " ", "") .. '.' .. obj.SensorNumber)
            return {oid, obj.SensorReading ~= 'na' and tonumber(obj.SensorReading) or json.null}
        end
    end
end

local function get_temperatures(redfish_sensor_path, obj)
    local rsp = cjson.json_object_new_object()
    local obj_sensor
    local obj_entity

    local sensor_path = bus:call('bmc.kepler.maca', '/bmc/kepler/MdbService', 'bmc.kepler.Mdb', 'GetPath',
    'a{ss}ssb', context.new(), "bmc.kepler.Systems.ThresholdSensor",
    json.encode({SensorName = obj.SensorName, SensorNumber = obj.SensorNumber}), false)
    obj_sensor = mdb.get_object(bus, sensor_path, "bmc.kepler.Systems.ThresholdSensor")

    local entity_path = bus:call('bmc.kepler.maca', '/bmc/kepler/MdbService', 'bmc.kepler.Mdb', 'GetPath',
    'a{ss}ssb', context.new(), "bmc.kepler.Systems.Entity",
    json.encode({Id = obj_sensor.EntityId, Instance = obj_sensor.EntityInstance}), false)

    if string.find(entity_path, "Entities") then
        obj_entity = mdb.get_object(bus, entity_path, "bmc.kepler.Systems.Entity")
    end

    rsp['DataSourceUri'] = redfish_sensor_path ..
        string.format(string.gsub(obj.SensorName, " ", "") .. '.' .. obj.SensorNumber)
    rsp['DeviceName'] = obj_entity and obj_entity.Name or cjson.null
    rsp['PhysicalContext'] = obj_entity and obj_entity.Name or cjson.null
    rsp['Reading'] = obj.SensorReading ~= 'na' and tonumber(obj.SensorReading) or json.null
    return rsp
end

function m.get_temperature_range_celsius(id, sensor_list)
    local res1 = cjson.json_object_new_array()
    local redfish_sensor_path = '/redfish/v1/Chassis/' .. id .. '/Sensors/'
    for _, obj in pairs(sensor_list) do
        if obj.SensorType == SENSOR_TYPE_TEMPERATURE then
            res1[#res1 + 1] = get_temperatures(redfish_sensor_path, obj)
        end
    end
    return res1
end

function m.get_exhaust(id, sensor_list)
    local _, oid
    for _, obj in pairs(sensor_list) do
        if obj.SensorIdentifier == OUTLET_TEMP then
            oid = '/redfish/v1/Chassis/' .. id .. '/Sensors/' ..
                string.format(string.gsub(obj.SensorName, " ", "") .. '.' .. obj.SensorNumber)
            return {oid, obj.SensorReading ~= 'na' and tonumber(obj.SensorReading) or json.null}
        end
    end
end

function m.get_intake(id, sensor_list)
    local _, oid
    for _, obj in pairs(sensor_list) do
        if obj.SensorIdentifier == INLET_TEMP then
            oid = '/redfish/v1/Chassis/' .. id .. '/Sensors/' ..
                string.format(string.gsub(obj.SensorName, " ", "") .. '.' .. obj.SensorNumber)
            return {oid, obj.SensorReading ~= 'na' and tonumber(obj.SensorReading) or json.null}
        end
    end
end

return m