-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local log = require 'mc.logging'
local cjson = require 'cjson'

local m = {}

function m.get_vlan_enable(req_vlan)
    -- 如果只传了vlan_id表示要开启vlan
    if req_vlan ~= nil and req_vlan.VLANEnable == nil and req_vlan.VLANId ~= nil then
        return true
    end
    if req_vlan == nil or req_vlan.VLANEnable == nil then
        local ethernet = mdb.get_object(bus, '/bmc/kepler/Managers/1/EthernetInterfaces',
            'bmc.kepler.Managers.EthernetInterfaces')
        return ethernet.VLANEnable
    else
        return req_vlan.VLANEnable
    end
end

function m.get_vlan_id(req_vlan, origin_vlan)
    -- 如果只传了vlan_enable为fasle没传vlan_id表示关闭vlan，则vlan_id返回0
    if req_vlan ~= nil and req_vlan.VLANEnable == false and req_vlan.VLANId == nil then
        return 0
    end
    if req_vlan ~= nil and req_vlan.VLANEnable == true and req_vlan.VLANId == nil and origin_vlan.VLANId == nil then
        local err = base_messages.PropertyMissing('VLAN/VLANId')
        err.RelatedProperties = {'#/VLAN'}
        error(err)
    end
    if req_vlan ~= nil and req_vlan.VLANEnable == true and (origin_vlan.VLANId < 1 or origin_vlan.VLANId > 4094) then
        local err = custom_messages.PropertyValueOutOfRange(origin_vlan.VLANId, 'VLAN/VLANId')
        err.RelatedProperties = {'#/VLAN'}
        error(err)
    end
    if req_vlan ~= nil and req_vlan.VLANEnable == nil and (req_vlan.VLANId < 1 or req_vlan.VLANId > 4094) then
        local err = custom_messages.PropertyValueOutOfRange(req_vlan.VLANId, 'VLAN/VLANId')
        err.RelatedProperties = {'#/VLAN'}
        error(err)
    end
    if req_vlan == nil or req_vlan.VLANId == nil then
        local ethernet = mdb.get_object(bus, '/bmc/kepler/Managers/1/EthernetInterfaces',
            'bmc.kepler.Managers.EthernetInterfaces')
        return ethernet.VLANId
    else
        return req_vlan.VLANId
    end
end

function m.get_dedicated_vlan_enable(req_vlan)
    -- 如果只传了vlan_id表示要开启vlan
    if req_vlan ~= nil and req_vlan.VLANEnable == nil and req_vlan.VLANId ~= nil then
        return true
    end
    if req_vlan ~= nil and req_vlan.VLANEnable == false and req_vlan.VLANId ~= nil and req_vlan.VLANId ~= 0 then
        local err = custom_messages.VLANInfoConflict()
        err.RelatedProperties = {'#/VLAN/VLANEnable', '#/VLAN/VLANId'}
        error(err)
    end
    return req_vlan.VLANEnable
end

function m.get_dedicated_vlan_id(req_vlan, origin_vlan)
    -- 如果只传了vlan_enable为fasle没传vlan_id表示关闭vlan，则vlan_id返回0
    if req_vlan ~= nil and req_vlan.VLANEnable == false and req_vlan.VLANId == nil then
        return 0
    end
    if req_vlan ~= nil and req_vlan.VLANEnable == true and req_vlan.VLANId == nil and origin_vlan.VLANId == nil then
        local err = base_messages.PropertyMissing('VLAN/VLANId')
        err.RelatedProperties = {'#/VLAN'}
        error(err)
    end
    if req_vlan ~= nil and req_vlan.VLANEnable == true and (origin_vlan.VLANId < 1 or origin_vlan.VLANId > 4094) then
        local err = custom_messages.PropertyValueOutOfRange(origin_vlan.VLANId, 'VLAN/VLANId')
        err.RelatedProperties = {'#/VLAN'}
        error(err)
    end
    if req_vlan ~= nil and req_vlan.VLANEnable == nil and (req_vlan.VLANId < 1 or req_vlan.VLANId > 4094) then
        local err = custom_messages.PropertyValueOutOfRange(req_vlan.VLANId, 'VLAN/VLANId')
        err.RelatedProperties = {'#/VLAN'}
        error(err)
    end
    return req_vlan.VLANId
end

function m.set_address()
    local err = base_messages.PropertyNotWritable('IPv4Addresses')
    err.RelatedProperties = {'#/IPv4Addresses'}
    error(err)
end

function m.check_management_network_port(object)
    if object.Type == nil and object.PortNumber ~= nil then
        local err = custom_messages.PropertyValueConflict(
            'Oem/{{OemIdentifier}}/ManagementNetworkPort/PortNumber',
            'Oem/{{OemIdentifier}}/ManagementNetworkPort/Type')
        err.RelatedProperties = {
            '#/Oem/{{OemIdentifier}}/ManagementNetworkPort/PortNumber',
            '#/Oem/{{OemIdentifier}}/ManagementNetworkPort/Type'}
        error(err)
    end

    if object.PortNumber == nil and object.Type ~= nil then
        local err = custom_messages.PropertyValueConflict(
            'Oem/{{OemIdentifier}}/ManagementNetworkPort/Type',
            'Oem/{{OemIdentifier}}/ManagementNetworkPort/PortNumber')
        err.RelatedProperties = {
            '#/Oem/{{OemIdentifier}}/ManagementNetworkPort/Type',
            '#/Oem/{{OemIdentifier}}/ManagementNetworkPort/PortNumber'}
        error(err)
    end

    return true
end

function m.check_port(path, type, port_number)
    log:notice('path = %s, type = %s, port_number = %s', path, type, port_number)
    if path == '' then
        local err = custom_messages.PortNotExist(type, tostring(port_number))
        err.RelatedProperties = {
            '#/Oem/{{OemIdentifier}}/ManagementNetworkPort/Type',
            '#/Oem/{{OemIdentifier}}/ManagementNetworkPort/PortNumber'}
        error(err)
    end
    return true
end

function m.check_vlan_setting(vlan, origin_vlan)
    if vlan.VLANEnable == true then
        if vlan.VLANId == nil and origin_vlan.VLANId == nil then
            local err = base_messages.PropertyMissing('VLAN/VLANId')
            err.RelatedProperties = {'#/VLAN'}
            error(err)
        end

        if vlan.VLANId == nil and origin_vlan.VLANId ~= nil then
            local err = custom_messages.PropertyValueConflict('VLAN/VLANEnable', 'VLAN/VLANId')
            err.RelatedProperties = {'#/VLAN/VLANEnable', '#/VLAN/VLANId'}
            error(err)
        end
    end

    if vlan.VLANEnable == false and vlan.VLANId ~= nil then
        local err = custom_messages.VLANInfoConflict()
        err.RelatedProperties = {'#/VLAN'}
        error(err)
    end

    return true
end

function m.get_switch_connections(conns)
    local objs = cjson.decode(conns)
    local switch_conns = cjson.json_object_new_array()
    for _, v in pairs(objs) do
        local conn = cjson.json_object_new_object()
        conn.SwitchManagementIP = v.SwitchManagementIP
        conn.SwitchConnectionPortIDs = cjson.json_object_new_array()
        for _, vv in pairs(v.SwitchConnectionPortIDs) do
            conn.SwitchConnectionPortIDs[#conn.SwitchConnectionPortIDs + 1] = vv
        end
        switch_conns[#switch_conns + 1] = conn
    end
    return switch_conns
end

function m.check_ipv6_enabled(enable, ipversion)
    -- 仅在IPv4下才需要进行使能,否则不需要更新
    if enable then
        return ipversion == 'IPv4'
    end

    -- 仅开启IPv6模式下不允许关闭
    if ipversion == 'IPv6' then
        local err = custom_messages.PropertyModificationNotSupported()
        err.RelatedProperties = {'#/IPv6Enabled'}
        error(err)
    end

    return ipversion == 'IPv4AndIPv6'
end

function m.check_user_ntp_servers(enable, ipversion, ipv4mode, current_ntp_mode)
    if not enable then
        return current_ntp_mode == 'IPv4'
    end

    -- 开启使能时需要IPv4开启DHCP
    if ipversion == 'IPv6' or ipv4mode == 'Static' then
        local err = custom_messages.PropertyModificationNotSupported('UseNTPServers')
        err.RelatedProperties = {'#/DHCPv4/UseNTPServers'}
        error(err)
    end

    return true
end

return m
