-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


-- Description: 通用工具
local file_utils = require 'utils.file'
local core_utils = require 'utils.core'
local mdb_service = require 'mc.mdb.mdb_service'
local log = require 'mc.logging'

local m = {}

function m.is_import_permitted(type, content, file_type, property_name, result)
    if type ~= 'URI' then
        return true
    end

    local pattern_collection = {
        ['pub'] = "^((https|sftp|nfs|cifs|scp)://.{1,1000}|/tmp/.{1,246})\\.pub$",
        ['cert'] = "^((https|sftp|nfs|cifs|scp)://.{1,1000}|/tmp/.{1,246})\\.(crt|cer|cert|pem|p12|pfx|crl|der)$",
        ['tab'] = "^((https|sftp|nfs|cifs|scp)://.{1,1000}|/tmp/.{1,246})\\.tab$",
        ['weakpwddic'] = "^((https|sftp|nfs|cifs|scp)://.{1,1000}|/tmp/.{1,251})$"
    }
    local error_collection = {
        ['pub'] = custom_messages.PublicKeyImportFailed(),
        ['cert'] = custom_messages.CertImportFailed(),
        ['tab'] = custom_messages.InvalidPath("******", property_name),
        ['weakpwddic'] = custom_messages.WeakPWDDictImportFailed()
    }

    if not core_utils.g_regex_match(pattern_collection[file_type], content) then
        error(base_messages.PropertyValueFormatError("******", property_name))
    end

    if content:sub(1,1) ~= '/' then
        return true
    end

    if not core_utils.is_file(content) then
        error(error_collection[file_type])
    end

    if file_utils.check_real_path_s(content, "/tmp") ~= 0 then
        error(error_collection[file_type])
    end

    if result then
        return true
    end
    error(custom_messages.NoPrivilegeToOperateSpecifiedFile())
end

function m.is_valid_path(path)
    local ok, rsp = pcall(mdb_service.is_valid_path, bus, path)
    if not ok then
        log:info('Invalid Path, err(%s)', rsp.message)
        error(rsp)
    end
    return rsp.Result
end

function m.get_all_system_id()
    -- 装备测试L1板环境上当前路径下没有深度为1的子路径，所以直接找一个可以覆盖的路径深度，深度为20
    local ok, rsp = pcall(mdb_service.get_sub_paths, bus, '/bmc/kepler/Systems', 20, {})
    if not ok then
        log:error('Invalid path, err(%s)', rsp)
    end

    local pre_suffix_len = string.len('/bmc/kepler/Systems/')
    local result = {}
    local hash = {}
    local sub_str
    local start
    for _, v in pairs(rsp.SubPaths) do
        start = string.find(v, '/', pre_suffix_len + 1)
        if start == nil then
            sub_str = v:sub(pre_suffix_len + 1)
        elseif start > pre_suffix_len + 1 then
            sub_str = v:sub(pre_suffix_len + 1, start - 1)
        end
        if sub_str ~= nil and sub_str ~= '' and hash[sub_str] == nil then
            table.insert(result, sub_str)
            hash[sub_str] = true
        end
    end

    return next(result) and result or {'1'}
end

return m
