-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local bs = require 'mc.bitstring'
local ipmi = require 'ipmi'
local mdb = require 'mc.mdb'
require 'ipmi'
local context = require 'mc.context'
local lu = require 'luaunit'
local custom_msg = require 'messages.custom'
local comp_code = ipmi.types.Cc
local log = require 'mc.logging'
local utils = require 'mc.utils'
local json = require 'cjson'
local enums = require 'ipmi.enums'
local skynet = require 'skynet'
local ctx = require 'mc.context'
local channel_type = enums.ChannelType

POWER_PATH = '/bmc/kepler/Systems/1/PowerMgmt/'
POWER_INTF_STATUS = 'bmc.kepler.Systems.PowerMgmt.OnePower.Status'
local test_power_ipmi = {}

-- 测试发送ipmi命令
local function send_ipmi_cmd(bus, ipmi_req)
    ipmi_req.Payload = ipmi_req.Payload .. '\x00'
    local ipmi_package = bs.new('<<_,_:2,DestNetFn:6,_:3/unit:8,Cmd,Payload/string>>')
    local req = ipmi_package:pack(ipmi_req)
    local ctx = json.encode({ ChanType = channel_type.CT_ME:value(), Instance = 0 })
    local service = 'bmc.kepler.ipmi_core'
    local path = '/bmc/kepler/IpmiCore'
    local intf = 'bmc.kepler.IpmiCore'
    local rsp = bus:call(service, path, intf, 'Route', 'a{ss}ayay', require 'mc.context'.new(), req, ctx)
    return rsp
end

local function ipmi_test_tool(bus, ipmi_req, chan_type)
    local ipmi_package = bs.new('<<_,_:2,DestNetFn:6,_:3/unit:8,Cmd,Payload/string>>')
    local req = ipmi_package:pack(ipmi_req)
    if not chan_type then
        chan_type = channel_type.CT_ME:value()
    end
    local ctx = json.encode({ChanType = chan_type, Instance = 0})
    local ok, rsp = pcall(bus.call, bus, 'bmc.kepler.ipmi_core', '/bmc/kepler/IpmiCore',
        'bmc.kepler.IpmiCore', 'Route', 'a{ss}ayay', require'mc.context'.new(), req, ctx)
    return ok, rsp
end

local function set_sleep_mode(bus)
    local ok, status_obj, rsp
    ok, status_obj = pcall(
        mdb.get_object, bus, POWER_PATH .. 'OnePower_0_010109', POWER_INTF_STATUS
    )
    lu.assertTrue(ok, true)
    lu.assertEquals(status_obj['SleepMode'], 'Normal')
    ok, rsp = status_obj.pcall:SetSleepMode(ctx.new(), 'DeepSleep')
    lu.assertTrue(ok)
    lu.assertEquals(rsp, 0)
    skynet.sleep(500)
    lu.assertEquals(status_obj['SleepMode'], 'DeepSleep')
end

function test_power_ipmi.test_set_power_status(bus)
    -- 设置psu1 StandBy模式
    local req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x2D\x01\x10\x00\x00\x04\x01\x00\x00\x00'}
    send_ipmi_cmd(bus, req)
    skynet.sleep(1500)
    -- 查询psu1 预期StandBy模式
    req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1E\x01\x12\x00\x08'}
    local rsp = send_ipmi_cmd(bus, req)
    print(utils.to_hex(rsp))
    assert(rsp == '\00\xdb\x07\x00\x00\x01\x00\x00\x00')

    -- 设置Active模式
    assert(rsp:byte(1) == 0x00)
    req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x2D\x01\x10\x00\x00\x04\x00\x00\x00\x00'}
    rsp = send_ipmi_cmd(bus, req)
    assert(rsp:byte(1) == 0x00)
    -- 设置异常，长度小于4
    req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x2D\x01\x10\x00\x00\x04\x00\x00'}
    rsp = send_ipmi_cmd(bus, req)
    -- 第一字节为完成码
    assert(rsp:byte(1) == 0xc7)
    -- 设置不存在的电源
    req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x2D\x03\x10\x00\x00\x04\x00\x00\x00\x00'}
    rsp = send_ipmi_cmd(bus, req)
    assert(rsp:byte(1) == 0xc9)
    skynet.sleep(500)
    -- 电源深度休眠模式，设置电源备用成功
    set_sleep_mode(bus)
    req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x2D\x01\x10\x00\x00\x04\x01\x00\x00\x00'}
    send_ipmi_cmd(bus, req)
    skynet.sleep(1500)
    req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1E\x01\x12\x00\x08'}
    rsp = send_ipmi_cmd(bus, req)
    assert(rsp == '\00\xdb\x07\x00\x00\x01\x00\x00\x00')
    -- 电源深度休眠模式，设置电源主用成功
    set_sleep_mode(bus)
    req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x2D\x01\x10\x00\x00\x04\x00\x00\x00\x00'}
    send_ipmi_cmd(bus, req)
    skynet.sleep(1500)
    req = {DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1E\x01\x12\x00\x08'}
    rsp = send_ipmi_cmd(bus, req)
    assert(rsp == '\00\xdb\x07\x00\x00\x00\x00\x00\x00')
end

function test_power_ipmi.test_get_voltage(bus)
    log:notice('================ test_get_voltage start ================')
    -- 获取第一个电源的电压输入
    local req1 = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x31\x02\x01\x01' }
    local rsp1 = send_ipmi_cmd(bus, req1)

    local rsp_info1 = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        Reading:4/unit:8,
        0x68:1/unit:8>>]=])
    log:notice(utils.to_hex(rsp1))
    local rsp_data1 = rsp_info1:unpack(rsp1)
    lu.assertEquals(rsp_data1['CompletionCode'], 0)
    lu.assertEquals(rsp_data1['Reading'], 229000)

    -- 获取第二个电源的电压输出
    local req2 = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x31\x02\x02\x02' }
    local rsp2 = send_ipmi_cmd(bus, req2)
    local rsp_info2 = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        Reading:4/unit:8,
        0x68:1/unit:8>>]=])
    local rsp_data2 = rsp_info2:unpack(rsp2)
    lu.assertEquals(rsp_data2['CompletionCode'], 0)
    lu.assertEquals(rsp_data2['Reading'], 0)
    log:notice('================ test_get_voltage complete ================')
end

function test_power_ipmi.test_get_current(bus)
    log:notice('================ test_get_current start ================')
    -- 获取第一个电源的电流输入
    local req1 = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x31\x03\x01\x01' }
    local rsp1 = send_ipmi_cmd(bus, req1)
    local rsp_info1 = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        Reading:4/unit:8,
        0x69:1/unit:8>>]=])
    local rsp_data1 = rsp_info1:unpack(rsp1)
    lu.assertEquals(rsp_data1['CompletionCode'], 0)
    lu.assertEquals(rsp_data1['Reading'], 610)

    -- 获取第二个电源的电流输出
    local req2 = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x31\x03\x02\x02' }
    local rsp2 = send_ipmi_cmd(bus, req2)
    local rsp_info2 = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        Reading:4/unit:8,
        0x69:1/unit:8>>]=])
    local rsp_data2 = rsp_info2:unpack(rsp2)
    lu.assertEquals(rsp_data2['CompletionCode'], 0)
    lu.assertEquals(rsp_data2['Reading'], 10000)
    log:notice('================ test_get_current complete ================')
end

function test_power_ipmi.test_get_power(bus)
    log:notice('================ test_get_power start ================')
    -- 获取第一个电源的功率输入
    local req1 = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x31\x08\x01\x01' }
    local rsp1 = send_ipmi_cmd(bus, req1)

    local rsp_info1 = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        Reading:4/unit:8,
        0x6:1/unit:8>>]=])
    local rsp_data1 = rsp_info1:unpack(rsp1)
    lu.assertEquals(rsp_data1['CompletionCode'], 0)
    lu.assertEquals(rsp_data1['Reading'], 140)

    -- 获取第一个电源的功率输出
    local req2 = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x31\x08\x02\x01' }
    local rsp2 = send_ipmi_cmd(bus, req2)

    local rsp_info2 = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        Reading:4/unit:8,
        0x6:1/unit:8>>]=])
    local rsp_data2 = rsp_info2:unpack(rsp2)
    lu.assertEquals(rsp_data2['CompletionCode'], 0)
    lu.assertEquals(rsp_data2['Reading'], 129)

    -- 获取第二个电源的功率输出
    local req3 = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x31\x08\x02\x02' }
    local rsp3 = send_ipmi_cmd(bus, req3)
    local rsp_info3 = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        Reading:4/unit:8,
        0x6:1/unit:8>>]=])
    local rsp_data3 = rsp_info3:unpack(rsp3)
    lu.assertEquals(rsp_data3['CompletionCode'], 0)
    lu.assertEquals(rsp_data3['Reading'], 0)
    log:notice('================ test_get_power complete ================')
end

function test_power_ipmi.test_get_power_supply_info(bus)
    log:notice('================ test_get_power_supply_info start ================')
    local resp_info = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        EndOfList:1/unit:8,
        Data/string>>]=])
    -- Manufacturer
    local req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x01\x00\x20' }
    local rsp = send_ipmi_cmd(bus, req)
    local data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 'HUAWEI')

    -- PowerSupplyType
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x02\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, '\1')

    -- Model
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x03\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 'PAC900S12-B2')

    -- FirmwareVersion
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x04\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 'DC:108 PFC:108')

    -- Protocol
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x05\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, '\1')

    --Mfr Status
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x14\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)

    -- PSON Status
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x15\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, '\1')

    -- SourceType
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x11\x00\x01' }
    rsp = send_ipmi_cmd(bus, req)
    -- 第一字节为完成码
    lu.assertEquals(rsp: byte(1), 0)
    -- 第六字节为Data
    lu.assertEquals(rsp: byte(6), 0)
    log:notice('================ test_get_power_supply_info complete ================')
end

function test_power_ipmi.test_get_power_supply_info_2(bus)
    log:notice('================ test_get_power_supply_info_2 start ================')
    local resp_info = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        EndOfList:1/unit:8,
        Data/string>>]=])

    --Rated Power
    local req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x06\x00\x20' }
    local rsp = send_ipmi_cmd(bus, req)
    local resp_info_number1 = bs.new([=[<<
    CompletionCode:1/unit:8,
    0x0007DB:3/unit:8,
    EndOfList:1/unit:8,
    Data:2/unit:8>>]=])
    local data = resp_info_number1:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 900)

    --Vin
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x07\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    local resp_info_number = bs.new([=[<<
    CompletionCode:1/unit:8,
    0x0007DB:3/unit:8,
    EndOfList:1/unit:8,
    Data:8/unit:8>>]=])
    data = resp_info_number:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 4642261237772582912)

    --Vout
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x08\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info_number:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 4622945017495814144)

    --Iin
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x09\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info_number:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 4603669611090668421)

    --Iout
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x0a\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info_number:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 4622100592565682176)

    --Ps OK Status
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x13\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, '\1')

    log:notice('================ test_get_power_supply_info_2 complete ================')
end


function test_power_ipmi.test_get_power_supply_info_3(bus)
    log:notice('================ test_get_power_supply_info_3 start ================')
    local resp_info = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        EndOfList:1/unit:8,
        Data/string>>]=])

    local resp_info_number = bs.new([=[<<
        CompletionCode:1/unit:8,
        0x0007DB:3/unit:8,
        EndOfList:1/unit:8,
        Data:8/unit:8>>]=])

    --Switch Status
    local req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x0b\x00\x20' }
    local rsp = send_ipmi_cmd(bus, req)
    local data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, '\1')

    --SN
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x0c\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, '2102312XWK10N8112866')

    --Env Temp
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x0d\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info_number:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 0)

    --Input Power
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x0e\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info_number:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 4639129828656676864)

    --Output Power
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x0f\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info_number:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)
    lu.assertEquals(data.Data, 4638742800563699712)

    --Vout Work mode
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x10\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)

    --Actual Vout Work Mode
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x1e\x01\x12\x00\x20' }
    rsp = send_ipmi_cmd(bus, req)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.CompletionCode, 0)

    log:notice('================ test_get_power_supply_info_3 complete ================')
end

function test_power_ipmi.test_set_psu_fru_config(bus)
    log:notice('================ test_set_psu_fru_config start ================')
    local resp_info = bs.new([=[<<
    CompletionCode:1/unit:8,
    0x0007DB:3/unit:8,
    ConfigValue:1/unit:8>>]=])

    local ok, req, rsp, data
    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x5a\x1d\x00\x02\x00\x00\x00\x00' }
    ok, rsp = ipmi_test_tool(bus, req)
    assert(ok, 'err_msg=%s', rsp)
    lu.assertEquals(rsp:byte(1), 0)


    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x5b\x1d\x00\x01\x00\x00\x00' }
    ok, rsp = ipmi_test_tool(bus, req)
    assert(ok, 'err_msg=%s', rsp)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.ConfigValue, 0)

    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x5a\x1d\x00\x02\x00\x00\x01\x00' }
    ok, rsp = ipmi_test_tool(bus, req)
    assert(ok, 'err_msg=%s', rsp)
    lu.assertEquals(rsp:byte(1), 0)


    req = { DestNetFn = 0x30, Cmd = 0x93, Payload = '\xdb\x07\x00\x5b\x1d\x00\x01\x00\x00\x00' }
    ok, rsp = ipmi_test_tool(bus, req)
    assert(ok, 'err_msg=%s', rsp)
    data = resp_info:unpack(rsp)
    lu.assertEquals(data.ConfigValue, 1)

    log:notice('================ test_set_psu_fru_config complete ================')
end

return test_power_ipmi
