-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


local class = require 'mc.class'
local log = require 'mc.logging'
local initiator = require 'mc.initiator'
local enums = require 'macros.power_mgmt_enums'
local singleton = require 'mc.singleton'
local utils = require 'mc.utils'
local parser_cfg = require 'parser_cfg'
local custom_msg = require 'messages.custom'
local client = require 'power_mgmt.client'

local power_mgmt_utils = class(utils)

function power_mgmt_utils:ctor()
    self.last_time_map = {}
    self.is_upgrading = false
    self.power_upgrade_slot_numbers = {}
    self.is_perform_upgrade = true -- 默认值为true，即每次升级时默认升级所有电源
    self.power_upgrade_success = false
    self.update_cfg_firmware_version = 'null' -- 默认值为null，即升级包中的update.cfg文件中不存在固件版本号
    self.upgrade_psu_objs = nil
end

function power_mgmt_utils.operation_log(ctx, fmt, ...)
    local intf = (ctx and ctx.Interface) and ctx.Interface or 'N/A'
    local user_name = (ctx and ctx.UserName) and ctx.UserName or 'N/A'
    local client_addr = (ctx and ctx.ClientAddr)  and ctx.ClientAddr or '127.0.0.1'
    log:operation(initiator.new(intf, user_name, client_addr), 'power_mgmt', fmt, ...)
end

-- 限频日志
---@param level number params: 1,error 2,info 3,debug
---@param interval_s number params: interval seconds
function power_mgmt_utils:frequency_limit_log(level, interval_s, fmt, ...)
    local last_time_key = debug.getinfo(2, 'S').short_src .. ':' .. debug.getinfo(2, "l").currentline
    local cur_time = os.time()
    local log_callback = {
        [enums.LOG_LEVEL.ERROR] = log.error,
        [enums.LOG_LEVEL.INFO] = log.info,
        [enums.LOG_LEVEL.DEBUG] = log.debug,
        [enums.LOG_LEVEL.NOTICE] = log.notice
    }

    local last_time = self.last_time_map[last_time_key]
    if not last_time or cur_time - last_time >= interval_s then
        self.last_time_map[last_time_key] = cur_time
        log_callback[level](log, fmt, ...)
    end
end

local cfgs = {}

function cfgs:get_cfgs(file_path)
    local ok, version, cfg = pcall(parser_cfg.get_cfgs, file_path)
    if ok then
        self.version = version
        self.cfgs = cfg
    end
end

function cfgs:get_version()
    if not self.cfgs then
        return nil
    end
    if self.cfgs['Firmware1'] and self.cfgs['Firmware1'].version then
        if self.cfgs['Firmware1'].version == 'null' then
            log:notice('The firm version does not exist')
        else
            log:notice('The firm version in file is %s', self.cfgs['Firmware1'].version)
        end
        return self.cfgs['Firmware1'].version
    end
    return nil
end

function cfgs:check_idex(idex)
    if not self.cfgs then
        return false
    end
    for _, cfg in pairs(self.cfgs) do
        if cfg then
            log:notice('power upgrade, component_idex %s', cfg.component_idex)
            if not cfg.component_idex then
                goto continue
            end
            if cfg.component_idex == idex then
                return true
            end
        end
        ::continue::
    end
    return false
end

function cfgs:get_component_idex()
    if not self.cfgs then
        log:notice('get_component_idex cfg nil')
        return 0
    end
    for _, cfg in pairs(self.cfgs) do
        if cfg then
            log:notice('power upgrade, component_idex %s', cfg.component_idex)
            if not cfg.component_idex then
                goto continue
            end
            if cfg.component_idex == 2 then  -- 2代表canbus电源
                log:notice('canbus power upgrade')
                return cfg.component_idex
            end
        end
        ::continue::
    end
    return 0  -- 默认按照pmbus电源处理
end

function power_mgmt_utils:set_upgrade_flag(value)
    self.is_upgrading = value
end

function power_mgmt_utils:get_upgrade_flag()
    return self.is_upgrading
end

local function validate_unique_natural_sequence(input_str)
    -- 不能存在空格，小数和字符
    if input_str:find(' ') or input_str:find('.', 1, true) or input_str:find('%a') then
        return false
    end
    -- 验证基本格式，不重复的自然数序列
    if not string.match(input_str, '^%d+') and
        not string.match(input_str, '^%d+(,%d+)*') then
        return false
    end
    -- 检查前导零和逗号格式
    -- 包括：前导0，含前导0的数字，连续逗号，起始逗号，尾随逗号
    if string.match(input_str, '^0%d') or
        string.match(input_str, ',0%d') or
        string.match(input_str, ',,') or
        string.match(input_str, '^,') or
        string.match(input_str, ',$') then
        return false
    end
    -- 获取所有槽位号，若有重复则失败
    local numbers = {}
    for num in string.gmatch(input_str, '%d+') do
        if numbers[num] then
            return false
        end
        numbers[num] = true
    end
    return true
end

function power_mgmt_utils:get_power_upgrade_flag_and_slot_number(slot_number)
    if string.len(slot_number) == 0 then
        self.is_perform_upgrade = true
        self.power_upgrade_slot_numbers = {}
        return
    end
    self.power_upgrade_slot_numbers = {}
    if validate_unique_natural_sequence(slot_number) == false then
        self.is_perform_upgrade = false
        self.power_upgrade_slot_numbers = {}
        return
    end
    for num in slot_number:gmatch('%d+') do
        table.insert(self.power_upgrade_slot_numbers, tonumber(num))
    end
    self.is_perform_upgrade = true
end

function power_mgmt_utils:init_power_upgrade_slot_number()
    self.power_upgrade_slot_numbers = {}
end

function power_mgmt_utils:get_power_upgrade_slot_number()
    return self.power_upgrade_slot_numbers
end

function power_mgmt_utils:get_perform_upgrade_flag()
    return self.is_perform_upgrade
end

function power_mgmt_utils:set_perform_upgrade_flag(value)
    self.is_perform_upgrade = value
end

function power_mgmt_utils:get_power_upgrade_success_flag()
    return self.power_upgrade_success
end

function power_mgmt_utils:set_power_upgrade_success_flag(value)
    self.power_upgrade_success = value
end

function power_mgmt_utils:set_update_cfg_firmware_version(version)
    self.update_cfg_firmware_version = version
end

function power_mgmt_utils:get_update_cfg_firmware_version()
    return self.update_cfg_firmware_version
end

function power_mgmt_utils:set_upgrade_psu_objs(psu_objs)
    self.upgrade_psu_objs = psu_objs
end

function power_mgmt_utils:get_upgrade_psu_objs()
    return self.upgrade_psu_objs
end

function power_mgmt_utils:get_psu_fan_rpm(c_psu_object, obj)
    local psu_obj = c_psu_object.collection:find(function(object)
        return object.path == obj.path
    end)
    if not psu_obj then
        error(custom_msg.OperationFailed())
    end
    return psu_obj:get_psu_fan_rpm()
end

function power_mgmt_utils:get_psu_register_info(c_psu_object, obj, cmd, length)
    local psu_obj = c_psu_object.collection:find(function(object)
        return object.path == obj.path
    end)
    if not psu_obj then
        error(custom_msg.OperationFailed())
    end
    return psu_obj:get_psu_register_info(cmd, length)
end

function power_mgmt_utils:dump_psu_blackbox(c_psu_object, obj, power_mgmt_mock)
    local psu_obj = c_psu_object.collection:find(function(object)
        local object_paths = self.split(object.path, '/')
        if not object_paths then
            return
        end
        local object_name = object_paths[#object_paths]
        local object_names = self.split(object_name, '_')
        if not object_names then
            return
        end
        object_names[1] = object_names[1] .. 'Debug'
        object_name = table.concat(object_names, '_')
        local obj_paths = self.split(obj.path, '/')
        if not obj_paths then
            return
        end
        return object_name == obj_paths[#obj_paths]
    end)
    return power_mgmt_mock.get_instance():dump_psu_blackbox(psu_obj.ps_id)
end

-- input字符串按照delimiter进行分隔, 返回table
function power_mgmt_utils.split(input, delimiter)
    if type(input) ~= 'string' or #input <= 0 then
        return nil
    end
    if type(delimiter) ~= 'string' or #delimiter <= 0 then
        return nil
    end
    local res = {}
    -- 使用 Lua 的 string.gmatch 函数进行字符串的拆分
    for match in string.gmatch(input, "([^" .. delimiter .. "]+)") do
        table.insert(res, match)
    end
    return res
end

function power_mgmt_utils:get_bmc_time()
    local date_time
    client:ForeachTimeObjects(function (obj)
        date_time = obj['DateTime']
    end)
    return date_time
end

power_mgmt_utils.cfgs = cfgs

return singleton(power_mgmt_utils)