local lu = require 'luaunit'
local metrics = require 'otel.metrics'
local utils = require 'mc.utils'
local paser = require 'parser.metric'
local dump_file = "./metrics.txt"

TestMetric = {}

local function read_file(filename)
    local file = io.open(filename, "r")
    lu.assertNotIsNil(file)
    local data = file:read("*all")
    file:close()
    return data
end

function TestMetric:setupClass()
    metrics.init_meter_provider(100, 50)
    self.meter = metrics.get_meter("test_meter")
end

function TestMetric:teardownClass()
    utils.remove_file(dump_file)
end

function TestMetric:test_metric()
    local counter = self.meter:create_counter("test_counter")
    counter:add(1)

    local updowncounter = self.meter:create_updowncounter("test_updowncounter", "a test updowncounter")
    updowncounter:add(-1)

    local observable_counter =
        self.meter:create_observable_counter("test_observable_counter", "a test observable counter")
    os.execute("sleep 0.2")
    observable_counter:add_callback(function() return 10 end)

    local observable_updowncounter =
        self.meter:create_observable_updowncounter("test_observable_updowncounter", "a test observable updowncounter")
    os.execute("sleep 0.2")
    observable_updowncounter:add_callback(function() return 20 end)

    local observable_gauge = self.meter:create_observable_gauge("test_observable_gauge", "a test observable gauge")
    os.execute("sleep 0.2")
    observable_gauge:add_callback(function() return 30 end)
    os.execute("sleep 0.2")

    local metrics = paser:parse_metric_file(dump_file)
    for _, metric in ipairs(metrics) do
        lu.assertEquals(metric.scope["scope name"], "test_meter")
        lu.assertEquals(metric.scope["version"], "")
        lu.assertEquals(metric.scope["schema url"], "")
        for _, instrument in ipairs(metric.instruments) do
            if instrument.name == "test_counter" then
                lu.assertEquals(instrument.value, "1")
            elseif instrument.name == "test_updowncounter" then
                lu.assertEquals(instrument.value, "-1")
                lu.assertEquals(instrument.description, "a test updowncounter")
            elseif instrument.name == "test_observable_counter" then
                lu.assertEquals(instrument.value, "10")
                lu.assertEquals(instrument.description, "a test observable counter")
            elseif instrument.name == "test_observable_updowncounter" then
                lu.assertEquals(instrument.value, "20")
                lu.assertEquals(instrument.description, "a test observable updowncounter")
            end
        end
    end
end

function TestMetric:test_metric_error()
    local meter = metrics.get_meter("test_meter_error")
    lu.assertErrorMsgContains("name can not be empty", function()
        meter:create_counter("")
    end)
    lu.assertErrorMsgContains("name must start with an alphabetic character", function()
        meter:create_counter("1_aa")
    end)
    lu.assertErrorMsgContains("name contains invalid character", function()
        meter:create_counter("bc@de")
    end)

    local counter = meter:create_counter("test_counter_error")
    lu.assertErrorMsgContains("value must be non-negative", function()
        counter:add(-1)
    end)
end

function TestMetric:test_metric_finish()
    local content = read_file(dump_file)
    if not content then
        print("failed to read file: " .. dump_file)
        return
    end
    lu.assertStrContains(content, "scope name	: test_meter")

    lu.assertStrContains(content, "instrument name	: test_counter")
    lu.assertStrContains(content, "value		: 1")
    
    lu.assertStrContains(content, "instrument name	: test_updowncounter")
    lu.assertStrContains(content, "description	: a test updowncounter")
    lu.assertStrContains(content, "value		: -1")

    lu.assertStrContains(content, "instrument name	: test_observable_counter")
    lu.assertStrContains(content, "description	: a test observable counter")
    lu.assertStrContains(content, "value		: 10")

    lu.assertStrContains(content, "instrument name	: test_observable_updowncounter")
    lu.assertStrContains(content, "description	: a test observable updowncounter")
    lu.assertStrContains(content, "value		: 20")

    lu.assertStrContains(content, "instrument name	: test_observable_gauge")
    lu.assertStrContains(content, "description	: a test observable gauge")
    lu.assertStrContains(content, "value     : 30")
end