-- Copyright (c) 2025 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local log = require 'mc.logging'
local cjson = require 'cjson'
local curl = require 'lcurl.core'
local defs = require 'public.defs'

log:set_log_module_name('observability')

local HEADER_CONTENT_TYPE = 'Content-Type'
local CONTENT_TYPE_JSON = 'application/json'

local function curl_https_get_header_call_back(title, context)
    log:debug('get https rsp header, %s:%s', title, context)
end

local function curl_https_get_body_call_back(str)
    log:debug('get https rsp body, %s', str)
end

-- 测试数据格式如下：
-- {
--     "resourceLogs": [
--         {
--             "resource": {},
--             "scopeLogs": [
--                 {
--                     "scope": {},
--                     "logRecords": [
--                         {
--                             "timeUnixNano": "1660296023390371588",
--                             "body": {
--                                 "stringValue": "{\"message\":\"Test log record\"}"
--                             },
--                             "traceId": "",
--                             "spanId": ""
--                         }
--                     ]
--                 }
--             ]
--         }
--     ]
-- }

local function generate_req_body()
    local body_json = cjson.json_object_new_object()
    body_json.resourceLogs = cjson.json_object_new_array()
    local resource_logs = cjson.json_object_new_object()
    body_json.resourceLogs[1] = resource_logs

    resource_logs.resource = cjson.json_object_new_object()
    resource_logs.scopeLogs = cjson.json_object_new_array()
    local scope_log = cjson.json_object_new_object()
    resource_logs.scopeLogs[1] = scope_log

    scope_log.scope = cjson.json_object_new_object()
    scope_log.logRecords = cjson.json_object_new_array()
    local log_record = cjson.json_object_new_object()
    scope_log.logRecords[1] = log_record

    local time = os.time()
    -- 此处获取的系统时间以秒为单位，需要转换为以纳秒为单位的时间戳
    log_record.timeUnixNano = tostring(time) .. '000000000'

    local str = cjson.json_object_new_object()
    str.stringValue = string.format("{\"message\":\"Test log record\"}")
    log_record.body = str
    log_record.traceId = ""
    log_record.spanId = ""

    return body_json
end

local function adjust_header(headers)
    local request_headers = curl.slist_init()
    for title, content in pairs(headers) do
        request_headers = curl.slist_append(request_headers, title .. ':' .. content)
    end

    return request_headers
end

return function(worker)
    local curl_handle = curl.easy_init()
    if curl_handle == nil then
        log:error("get curl handle failed")
        return '{}'
    end

    curl.easy_setopt_headerfunction(curl_handle, curl_https_get_header_call_back)
    curl.easy_setopt_writefunction(curl_handle, curl_https_get_body_call_back)

    local url = "http://" .. defs.FLUENT_BIT_LISTEN_ADDRESS .. ":" .. defs.FLUENT_BIT_LISTEN_PORT .. "/v1/logs"
    curl.easy_setopt_url(curl_handle, url)
    curl.easy_setopt_connecttimeout(curl_handle, 10)
    curl.easy_setopt_timeout(curl_handle, 60)

    local http_header = cjson.json_object_new_object()
    http_header[HEADER_CONTENT_TYPE] = CONTENT_TYPE_JSON
    curl.easy_setopt_httpheader(curl_handle, adjust_header(http_header))
    curl.easy_setopt_forbid_reuse(curl_handle, 1)
    curl.easy_setopt_nosignal(curl_handle, 1)
    curl.easy_setopt_customerquest(curl_handle, "POST")
    curl.easy_setopt_ssl_verifyhost(curl_handle, 0)
    curl.easy_setopt_ssl_verifypeer(curl_handle, 0)

    local req_body = generate_req_body()
    local json_data = cjson.json_object_ordered_encode(req_body)
    curl.easy_setopt_postfields(curl_handle, json_data)

    local ret_code = curl.easy_perform(curl_handle)
    if ret_code ~= 0 then
        log:notice_easy('send http msg failed, err: %s', curl.easy_strerror(ret_code))
        log:net_stream_print("[Observability] report event failed, Destination is: %s, error: %s",
            url, curl.easy_strerror(ret_code))
    end

    local curl_code, response_code = curl.easy_getinfo_response_code(curl_handle)
    g_result = cjson.encode({
        RetCode = ret_code,
        CurlCode = curl_code,
        ResponseCode = response_code
    })

    curl.easy_cleanup(curl_handle)
end