/* Copyright (c) 2024 Huawei Technologies Co., Ltd.
 * openUBMC is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */
#include <lua.hpp>
#include <iomanip>
#include <vector>
#include <memory>
#include <stdlib.h>
#include <mutex>
#include <opentelemetry/trace/provider.h>
#include <opentelemetry/sdk/trace/simple_processor.h>
#include <opentelemetry/sdk/trace/simple_processor_factory.h>
#include <opentelemetry/sdk/trace/tracer_provider_factory.h>
#include <opentelemetry/sdk/trace/batch_span_processor_factory.h>
#include <opentelemetry/sdk/trace/batch_span_processor_options.h>
#include <opentelemetry/exporters/otlp/otlp_grpc_exporter.h>
#include "opentelemetry/exporters/otlp/otlp_grpc_exporter_factory.h"

#define BUILD_TYPE_DT  (0x0a)
#if defined(BUILD_TYPE) && defined(BUILD_TYPE_DT) && BUILD_TYPE == BUILD_TYPE_DT
#include <opentelemetry/exporters/ostream/span_exporter_factory.h>
#include <opentelemetry/exporters/ostream/span_exporter.h>
#include <fstream>
#endif
#include "utils/utils.h"

static bool hasProvider = false;
static std::mutex initMutex;

namespace Otel {

namespace common = opentelemetry::common;
using namespace opentelemetry::trace;
using opentelemetry::nostd::string_view;
using Link = std::vector<std::pair<SpanContext, std::map<nostd::string_view, common::AttributeValue>>>;

struct LuaTracer {
    nostd::shared_ptr<opentelemetry::trace::Tracer> tracer;
};

struct LuaSpan {
    nostd::shared_ptr<opentelemetry::trace::Span> span;
};

namespace LTracer {

static std::string ToHex(const uint8_t* data, size_t size) {
    static const char hex[] = "0123456789abcdef";
    std::string output;
    output.reserve(size * 2);
    for (size_t i = 0; i < size; ++i) {
        output.push_back(hex[data[i] >> 4]);
        output.push_back(hex[data[i] & 0x0F]);
    }
    return output;
}

static bool isHexChar(char c) {
    return (c >= '0' && c <= '9') || (c >= 'a' && c <= 'f') || (c >= 'A' && c <= 'F');
}

static uint8_t toHexValue(char c) {
    if (c >= '0' && c <= '9') {
        return c - '0';
    } else if (c >= 'a' && c <= 'f') {
        return c - 'a' + 10;
    } else if (c >= 'A' && c <= 'F') {
        return c - 'A' + 10;
    } else {
        return 0;
    }
}

static std::vector<uint8_t> HexToBytes(const std::string& hex) {
    std::vector<uint8_t> bytes;
    size_t len = hex.length();
    if (len % 2 != 0) {
        // 长度非偶
        return {};
    }
    bytes.reserve(len / 2);
    for (size_t i = 0; i < len; i += 2) {
        char ch1 = hex[i];
        char ch2 = hex[i + 1];

        if (!isHexChar(ch1) || !isHexChar(ch2)) {
            // 非法字符
            return {};
        }
        uint8_t high = toHexValue(ch1);
        uint8_t low = toHexValue(ch2);
        uint8_t byte = (high << 4) | low;

        bytes.push_back(byte);
    }
    return bytes;
}

static int push_context(lua_State* L, const SpanContext& ctx)
{
    lua_newtable(L); // 创建一个新表

    // trace_id
    auto trace_id = ctx.trace_id();
    std::string trace_id_str = ToHex(trace_id.Id().data(), trace_id.Id().size());
    lua_pushstring(L, "trace_id");
    lua_pushstring(L, trace_id_str.c_str());
    lua_settable(L, -3);

    // span_id
    auto span_id = ctx.span_id();
    std::string span_id_str = ToHex(span_id.Id().data(), span_id.Id().size());
    lua_pushstring(L, "span_id");
    lua_pushstring(L, span_id_str.c_str());
    lua_settable(L, -3);

    // trace_flags
    lua_pushstring(L, "trace_flags");
    lua_pushinteger(L, static_cast<int>(ctx.trace_flags().flags()));
    lua_settable(L, -3);

    // is_remote
    lua_pushstring(L, "is_remote");
    lua_pushboolean(L, ctx.IsRemote());
    lua_settable(L, -3);

    // trace_state
    std::string trace_state_str = ctx.trace_state()->ToHeader();
    lua_pushstring(L, "trace_state");
    lua_pushstring(L, trace_state_str.c_str());
    lua_settable(L, -3);

    return 1;
}

static opentelemetry::trace::TraceId GetTraceId(lua_State* L) {
    const char* trace_id_str = lua_tostring(L, -1);
    if (trace_id_str == nullptr) {
        luaL_error(L, "%s: invalid trace id", __FUNCTION__);
    }
    lua_pop(L, 1);
    auto trace_id_bytes = HexToBytes(trace_id_str);
    if (trace_id_bytes.empty() || trace_id_bytes.size() != 16) {
        luaL_error(L, "%s: invalid trace id", __FUNCTION__);
    }
    return opentelemetry::trace::TraceId(trace_id_bytes);
}

static opentelemetry::trace::SpanId GetSpanId(lua_State* L, int index) {
    lua_getfield(L, index, "span_id");
    const char* span_id_str = lua_tostring(L, -1);
    if (span_id_str == nullptr) {
        luaL_error(L, "%s: invalid span id", __FUNCTION__);
    }
    lua_pop(L, 1);
    auto span_id_bytes = HexToBytes(span_id_str);
    if (span_id_bytes.empty() || span_id_bytes.size() != 8) {
        luaL_error(L, "%s: invalid span id", __FUNCTION__);
    }

    return opentelemetry::trace::SpanId(span_id_bytes);
}

static uint8_t GetTraceFlags(lua_State* L, int index) {
    lua_getfield(L, index, "trace_flags");
    uint8_t trace_flags = 0;
    if (lua_isinteger(L, -1)) {
        trace_flags = static_cast<uint8_t>(lua_tointeger(L, -1));
    } else {
        luaL_error(L, "%s: invalid trace flags", __FUNCTION__);
    }
    lua_pop(L, 1);
    return trace_flags;
}

static bool GetIsRemote(lua_State* L, int index) {
    lua_getfield(L, index, "is_remote");
    bool is_remote = lua_isboolean(L, -1) ? lua_toboolean(L, -1) : false;
    lua_pop(L, 1);
    return is_remote;
}

static nostd::shared_ptr<TraceState> GetTraceState(lua_State* L, int index) {
    lua_getfield(L, index, "trace_state");
    const char* trace_state_str = lua_tostring(L, -1);
    if (trace_state_str == nullptr) {
        luaL_error(L, "%s: invalid trace state", __FUNCTION__);
    }
    lua_pop(L, 1);
    auto trace_state = nostd::shared_ptr<TraceState>(TraceState::FromHeader(trace_state_str));
    if (!trace_state) {
        luaL_error(L, "%s: invalid trace state", __FUNCTION__);
    }

    return trace_state;
}

static SpanContext restore_context(lua_State* L, int index) {
    if (index < 0) {
        index = lua_gettop(L) + index + 1;
    }

    lua_getfield(L, index, "trace_id");
    if (lua_isnil(L, -1)) {
        lua_pop(L, 1);
        return SpanContext::GetInvalid();
    }
    TraceId trace_id = GetTraceId(L);
    SpanId span_id = GetSpanId(L, index);
    uint8_t trace_flags = GetTraceFlags(L, index);
    bool is_remote = GetIsRemote(L, index);
    nostd::shared_ptr<TraceState> trace_state = GetTraceState(L, index);

    opentelemetry::trace::SpanContext ctx(
        trace_id,
        span_id,
        opentelemetry::trace::TraceFlags(trace_flags),
        is_remote,
        trace_state
    );
    return ctx;
}

#if defined(BUILD_TYPE) && defined(BUILD_TYPE_DT) && BUILD_TYPE == BUILD_TYPE_DT
// 创建输出流导出器
std::string file_path = "./span_test.txt";
static std::ofstream file(file_path);
std::ostream& sout = file;

static void init_tracer_provider() 
{
    std::lock_guard<std::mutex> guard(initMutex);
    if (hasProvider) {
        return;
    }
    Utils::initialize_telemetry();
    auto exporter = opentelemetry::exporter::trace::OStreamSpanExporterFactory::Create(sout);

    // 创建处理器
    auto processor = std::unique_ptr<opentelemetry::sdk::trace::SpanProcessor>(
        new opentelemetry::sdk::trace::SimpleSpanProcessor(std::move(exporter)));

    // 创建TracerProvider
    auto provider = opentelemetry::sdk::trace::TracerProviderFactory::Create(std::move(processor));

    // 设置为全局Provider
    opentelemetry::trace::Provider::SetTracerProvider(std::move(provider));
    hasProvider = true;
}

static int flush(lua_State *L)
{
    sout.flush();
    return 0;
}

#else

static void init_tracer_provider(int max_size=2048, int delay_millies=5000, int export_size=512) 
{
    std::lock_guard<std::mutex> guard(initMutex);
    if (hasProvider) {
        return;
    }
    Utils::initialize_telemetry();
    // 创建grpc导出器
    opentelemetry::exporter::otlp::OtlpGrpcExporterOptions opts;
    opts.endpoint = "localhost:44318";
    auto exporter = opentelemetry::exporter::otlp::OtlpGrpcExporterFactory::Create(opts);

    opentelemetry::sdk::trace::BatchSpanProcessorOptions options{};
    options.max_queue_size = max_size;
    options.schedule_delay_millis = std::chrono::milliseconds(delay_millies);
    options.max_export_batch_size = export_size;
    
    // 创建处理器
    auto processor = opentelemetry::sdk::trace::BatchSpanProcessorFactory::Create(std::move(exporter), options);

    // 创建TracerProvider
    auto provider = opentelemetry::sdk::trace::TracerProviderFactory::Create(std::move(processor));

    // 设置为全局Provider
    opentelemetry::trace::Provider::SetTracerProvider(std::move(provider));
    hasProvider = true;
}

static int l_init_tracer_provider(lua_State *L)
{
     // 最大缓冲span数量,未指定使用默认值2048
     int max_queue_size = lua_isinteger(L, 1) ? static_cast<int32_t>(lua_tointeger(L, 1)) : 2048;

     // 发送间隔,未指定使用默认值5000ms
     int schedule_delay_millis = lua_isinteger(L, 2) ? static_cast<int32_t>(lua_tointeger(L, 2)) : 5000;

     // 设置每次发送最大Span数量,未指定使用默认值512
     int max_export_batch_size = lua_isinteger(L, 3) ? static_cast<int32_t>(lua_tointeger(L, 3)) : 512;

    init_tracer_provider(max_queue_size, schedule_delay_millis, max_export_batch_size);

    return 0;
}

#endif

static int get_tracer(lua_State *L)
{
    if (!hasProvider) {
        init_tracer_provider();
    }
    const char *name = lua_tostring(L, 1);
    if (name == nullptr) {
        luaL_error(L, "%s: invalid parameter, get tracer failed", __FUNCTION__);
    }

    auto provider = opentelemetry::trace::Provider::GetTracerProvider();
    auto tracer = provider->GetTracer(name, string_view(""), string_view(""));
    void* mem = lua_newuserdata(L, sizeof(LuaTracer));
    LuaTracer* udata = new (mem) LuaTracer();
    udata->tracer = tracer;
    luaL_getmetatable(L, "Tracer");
    lua_setmetatable(L, -2);

    return 1;
}

static int start_span(lua_State *L)
{
    LuaTracer* lua_tracer = (LuaTracer*)luaL_checkudata(L, 1, "Tracer");
    const char *name = lua_tostring(L, 2);
    if (name == nullptr) {
        luaL_error(L, "%s: invalid parameter, start span failed", __FUNCTION__);
    }

    AttributeMap attributes;
    Utils::get_attributes(L, attributes, 3);

    opentelemetry::trace::StartSpanOptions options;
    if (lua_istable(L, 4)) {
        SpanContext ctx = restore_context(L, 4);
        if (ctx.IsValid()) {
            options.parent = ctx;
        }
    }

    auto span = lua_tracer->tracer->StartSpan(name, attributes, options);
    void* mem = lua_newuserdata(L, sizeof(LuaSpan));
    LuaSpan* udata = new (mem) LuaSpan();
    udata->span = span;
    luaL_getmetatable(L, "Span");
    lua_setmetatable(L, -2);

    return 1;
}

} // namespace LTracer

namespace LSpan {

static int set_attribute(lua_State *L)
{
    LuaSpan* lua_span = (LuaSpan*)luaL_checkudata(L, 1, "Span");

    const char *key = lua_tostring(L, 2);
    if (key == nullptr) {
        luaL_error(L, "%s: invalid parameter, set attribute failed", __FUNCTION__);
    }

    AttributeValue value = Utils::get_attribute_value(L, 3);
    lua_span->span->SetAttribute(key, value);
    return 0;
}

static int add_event(lua_State *L)
{
    LuaSpan* lua_span = (LuaSpan*)luaL_checkudata(L, 1, "Span");

    const char *name = lua_tostring(L, 2);
    if (name == nullptr) {
        luaL_error(L, "%s: invalid parameter, add event failed", __FUNCTION__);
    }

    AttributeMap attributes;
    if (lua_gettop(L) >= 3) {
        Utils::get_attributes(L, attributes, 3);
    }

    lua_span->span->AddEvent(name, attributes);
    return 0;
}

static int add_link(lua_State *L)
{
    LuaSpan* lua_span = (LuaSpan*)luaL_checkudata(L, 1, "Span");

    SpanContext ctx = LTracer::restore_context(L, 2);

    AttributeMap attributes;
    if (lua_gettop(L) >= 3) {
        Utils::get_attributes(L, attributes, 3);
    }

    lua_span->span->AddLink(ctx, attributes);
    return 0;
}

static const std::unordered_map<std::string, StatusCode> StatusCodeMap = {
    {"unset", StatusCode::kUnset},
    {"ok", StatusCode::kOk},
    {"error", StatusCode::kError}
};

static int set_status(lua_State *L)
{
    LuaSpan* lua_span = (LuaSpan*)luaL_checkudata(L, 1, "Span");

    StatusCode code = StatusCode::kUnset;
    const char *status = lua_tostring(L, 2);
    const char *desc = lua_tostring(L, 3);
    if (status == nullptr || desc == nullptr) {
        luaL_error(L, "%s: invalid parameter, set status failed", __FUNCTION__);
    }

    auto target = StatusCodeMap.find(status);
    if (target == StatusCodeMap.end()) {
        luaL_error(L, "%s: invalid parameter, set status failed", __FUNCTION__);
    } else {
        code = target->second;
    }

    lua_span->span->SetStatus(code, desc);
    return 0;
}

static int get_context(lua_State *L)
{
    LuaSpan* lua_span = (LuaSpan*)luaL_checkudata(L, 1, "Span");

    SpanContext spanctx = lua_span->span->GetContext();
    return LTracer::push_context(L, spanctx);
}

static int is_recording(lua_State *L)
{
    LuaSpan* lua_span = (LuaSpan*)luaL_checkudata(L, 1, "Span");

    if (lua_span->span->IsRecording()) {
        lua_pushboolean(L, 1);
    } else {
        lua_pushboolean(L, 0);
    }

    return 1;
}

static int end(lua_State *L)
{
    LuaSpan* lua_span = (LuaSpan*)luaL_checkudata(L, 1, "Span");

    lua_span->span->End();
    return 0;
}

} // namespace LSpan

#if defined(BUILD_TYPE) && defined(BUILD_TYPE_DT) && BUILD_TYPE == BUILD_TYPE_DT
static const luaL_Reg lmetry[] = {
    {"get_tracer", LTracer::get_tracer},
    {"flush", LTracer::flush},
    {NULL, NULL}
};

#else

static const luaL_Reg lmetry[] = {
    {"get_tracer", LTracer::get_tracer},
    {"init", LTracer::l_init_tracer_provider},
    {NULL, NULL}
};

#endif

static const luaL_Reg ltracer[] = {
    {"start_span", LTracer::start_span},
    {NULL, NULL}
};

static const luaL_Reg lspan[] = {
    {"set_attribute", LSpan::set_attribute},
    {"add_event", LSpan::add_event},
    {"add_link", LSpan::add_link},
    {"set_status", LSpan::set_status},
    {"get_context", LSpan::get_context},
    {"is_recording", LSpan::is_recording},
    {"finish", LSpan::end},
    {NULL, NULL}
};

static void set_index_metatable(lua_State *L)
{
    luaL_newmetatable(L, "Tracer");
    luaL_setfuncs(L, ltracer, 0);
    lua_pushvalue(L, -1);
    lua_setfield(L, -2, "__index");

    luaL_newmetatable(L, "Span");
    luaL_setfuncs(L, lspan, 0);
    lua_pushvalue(L, -1);
    lua_setfield(L, -2, "__index");
}

extern "C" int luaopen_otel_trace(lua_State *L)
{
    luaL_checkversion(L);
    set_index_metatable(L);
    luaL_newlibtable(L, lmetry);
    luaL_setfuncs(L, lmetry, 0);

    return 1;
}

} // namespace Otel