-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local bs = require 'mc.bitstring'
local log = require 'mc.logging'
local core = require 'network.core'
local ncsi_protocol_intf = require 'ncsi_protocol_intf'
local ncsi_utils = require 'ncsi.ncsi_protocol.ncsi_utils'
local ncsi_def = require 'ncsi.ncsi_protocol.ncsi_def'
local ncsi_packet = require 'ncsi.ncsi_protocol.ncsi_packet'
local ncsi_parameter = require 'ncsi.ncsi_protocol.ncsi_parameter'

local ncsi_broadcast_filter = {}

-- 命令类型常量定义
local ENABLE_BROADCAST_FILTER = 0x10
local ENABLE_BROADCAST_FILTER_RSP = 0x90

-- 请求长度
local ENABLE_BROADFILTER_REQ_LEN = 4

-- 广播过滤器标志
local SUPPORT = 1
local NOT_SUPPORT = 0

-- 填充长度
local ENABLE_BROADFILTER_PAD_LEN = 22

-- 报文大小为payload + checksum + pad + FCS
local ENABLE_BROADFILTER_PKT_SIZE = ENABLE_BROADFILTER_REQ_LEN + 4 + ENABLE_BROADFILTER_PAD_LEN + 4

-- 启用广播过滤请求报文结构
local enable_broadcast_filter_req_bs = bs.new([[<<
    reserved1:3/string,
    arp_packet:1,
    dhcp_client_packet:1,
    dhcp_server_packet:1,
    netbios_packet:1,
    reserved2:4,
    check_sum:32,
    pad_data:22/string,
    fcs:32
>>]])

-- 填充启用广播过滤请求的payload
local function fill_enable_brdcast_filter_payload(req_packet, support_flag)
    -- 定义公共的payload数据结构
    local payload_data = {
        reserved1 = '',
        arp_packet = support_flag,
        dhcp_client_packet = support_flag,
        dhcp_server_packet = support_flag,
        netbios_packet = 0,
        reserved2 = 0,
        check_sum = 0, -- 初始值
        pad_data = string.rep('\0', ENABLE_BROADFILTER_PAD_LEN),
        fcs = 0        -- 初始值
    }

    -- 首先创建一个check_sum为0的初始payload
    req_packet.payload = enable_broadcast_filter_req_bs:pack(payload_data)

    -- 计算校验和和CRC32
    local check_sum = ncsi_utils.get_checksum(req_packet, ncsi_def.PACKET_HEAD_LEN + ENABLE_BROADFILTER_REQ_LEN)
    local crc32 = ncsi_utils.get_crc32(req_packet, ncsi_def.PACKET_HEAD_LEN + ENABLE_BROADFILTER_PKT_SIZE - 4)

    -- 更新checksum和FCS字段
    payload_data.check_sum = core.htonl(check_sum)
    payload_data.fcs = core.htonl(crc32)

    -- 返回最终的payload
    return enable_broadcast_filter_req_bs:pack(payload_data)
end

-- 写入启用广播过滤请求
local function write_enable_brdcast_filter_req(req_packet, eth_name, support_flag)
    ncsi_utils.ncsi_cmd_common_config(req_packet)
    req_packet.packet_head.payload_len_hi = (ENABLE_BROADFILTER_REQ_LEN >> 8) & 0x0f
    req_packet.packet_head.payload_len_lo = ENABLE_BROADFILTER_REQ_LEN & 0xff
    req_packet.payload = fill_enable_brdcast_filter_payload(req_packet, support_flag)

    local req_data = ncsi_utils.ncsi_packet_bs:pack(req_packet)
    return ncsi_protocol_intf.send_ncsi_cmd(req_data,
        ENABLE_BROADFILTER_PKT_SIZE + ncsi_def.ETHERNET_HEAD_LEN + ncsi_def.PACKET_HEAD_LEN, eth_name)
end

-- 读取启用广播过滤响应
local function read_enable_brdcast_filter_rsp(rsp)
    return ncsi_packet.read_common_rsp(rsp, 'broadcast filter')
end

-- 命令处理表
local broadcast_filter_table = {
    [ENABLE_BROADCAST_FILTER] = write_enable_brdcast_filter_req,
    [ENABLE_BROADCAST_FILTER_RSP] = read_enable_brdcast_filter_rsp
}

-- 通用广播过滤处理函数
local function handle_broadcast_filter(package_id, channel_id, eth_name, support_flag, operation_name)
    -- 填充广播过滤请求包
    local req_packet = ncsi_packet.create_request_packet(package_id, channel_id, ENABLE_BROADCAST_FILTER)

    local custom_cmd_table = ncsi_utils.create_custom_cmd_table(
        broadcast_filter_table, ENABLE_BROADCAST_FILTER, write_enable_brdcast_filter_req, support_flag
    )

    local ret = ncsi_utils.ncsi_cmd_ctrl(package_id, channel_id, req_packet, eth_name, custom_cmd_table)
    if ret ~= ncsi_def.NCSI_SUCCESS then
        log:error('NCSI %s broadcast filter failed, package_id = %s, channel_id = %s, eth_name = %s',
            operation_name, package_id, channel_id, eth_name)
    end
    return ret
end

-- 启用广播过滤
function ncsi_broadcast_filter.ncsi_enable_brdcast_filter(package_id, channel_id, eth_name)
    ncsi_parameter.get_instance():get_ncsi_parameter().current_channel = channel_id
    return handle_broadcast_filter(package_id, channel_id, eth_name, NOT_SUPPORT, "enable")
end

-- 禁用广播过滤
function ncsi_broadcast_filter.ncsi_disable_brdcast_filter(package_id, channel_id, eth_name)
    return handle_broadcast_filter(package_id, channel_id, eth_name, SUPPORT, "disable")
end

return ncsi_broadcast_filter
