-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local debounce = require 'mc.debounce.debounce'

-- 判断光模块8个通道是否符合告警条件，适用发送功率、接收功率、发送偏置电流告警。
local function get_channel_event_status(arr_val, channel_idx, upper, lower)
    if not arr_val or not arr_val[channel_idx] or not upper or not lower then
        return false
    end
    if arr_val[channel_idx] >= 0 and arr_val[channel_idx] < 255 and
        upper >= 0 and upper < 255 and
        lower >= 0 and lower < 255 and
        (arr_val[channel_idx] < lower or arr_val[channel_idx] > upper) then
        return true
    end
    return false
end

local function get_npu_name(npu_id)
    local board_id = (npu_id - 1) // 2 + 1
    return string.format("NPU%s", board_id)
end
local function get_arg_rx_power(npu_id, channel_idx)
    return string.format("Port %s(Channel %s Rx Power)", npu_id, channel_idx)
end
local function get_arg_tx_power(npu_id, channel_idx)
    return string.format("Port %s(Channel %s Tx Power)", npu_id, channel_idx)
end
local function get_arg_bias(npu_id, channel_idx)
    return string.format("Port %s(Channel %s)", npu_id, channel_idx)
end
local function get_arg_bias_val(arr_val, channel_idx)
    if not arr_val or not next(arr_val) then
        return 0
    end
    return string.format("%0.4f", arr_val[channel_idx])
end

-- 每个npu光模块的告警，36个
-- 防抖类型默认持续防抖，次数默认15次
local npu_om = {
    Events = {
        Event_OpticalModule1Die0HostInterfaceFault = {
            EventKeyId = "Port.PortOpticalModuleHostInterfaceFault",
            EventCode = '0x29000019',
            -- 告警主体类型，默认0x29。可不配置。
            SubjectType = 0x29,
            -- 相关的属性，用来打印定位。可不配置。
            prop_names = {"HostInterfaceFaultStatus"},
            -- 防抖类型，默认持续防抖。可不配置。
            type = debounce.DEBOUNCED_CONT,
            -- 防抖次数，默认15次。可不配置。
            cont_num = 15,
            -- 是否符合告警触发条件
            trigger_fun = function(om_obj)
                if om_obj.HostInterfaceFaultStatus == 65535 then
                    return false
                end
                return om_obj.HostInterfaceFaultStatus >= 1
            end,
            -- 告警参数
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID))
            end,
            -- 在同Component同EventCode下区分告警事件的若干个参数索引。可不配置。
            identify_arg_index = {}
        },
        Event_OpticalModule1Die0MediaInterfaceFault = {
            EventKeyId = "Port.PortOpticalModuleMediaInterfaceFault",
            EventCode = '0x2900001B',
            prop_names = {"MediaInterfaceFaultStatus"},
            trigger_fun = function(om_obj)
                if om_obj.MediaInterfaceFaultStatus == 65535 then
                    return false
                end
                return om_obj.MediaInterfaceFaultStatus >= 1
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID))
            end
        },
        Event_OpticalModule1Die0SelfTestFault = {
            EventKeyId = "Port.PortOpticalModuleSelfTestFault",
            EventCode = '0x2900001D',
            prop_names = {"SelfTestStatus"},
            trigger_fun = function(om_obj)
                return om_obj.SelfTestStatus >= 1
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID))
            end
        },
        Event_OpticalModule1Die0AccessFault = {
            EventKeyId = "Port.PortOpticalModuleAccessFault",
            EventCode = '0x29000023',
            prop_names = {"Accessible"},
            trigger_fun = function(om_obj)
                return om_obj.Accessible >= 1
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID))
            end
        },
        Event_OpticalModule1Die0TxFault = {
            EventKeyId = "Port.PortOpticalModuleTxFault",
            EventCode = '0x29000021',
            prop_names = {"TxFaultState"},
            trigger_fun = function(om_obj)
                return om_obj.TxFaultState
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID))
            end
        },
        Event_OpticalModule1Die0LossOfLock = {
            EventKeyId = "Port.PortOpticalModuleLossOfLock",
            EventCode = '0x29000025',
            prop_names = {"TxLossOfLock", "RxLossOfLock"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return (om_obj.TxLossOfLock + om_obj.RxLossOfLock) > 0
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID))
            end
        },
        Event_OpticalModule1Die0_SFPOverTempInfo = {
            EventKeyId = "NICCard.NICCardOMOverTemp",
            EventCode = '0x0D000017',
            prop_names = {"ReadingCelsius", "TemperatureUpperThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj.ReadingCelsius > 127 or om_obj.ReadingCelsius == 0 or
                    om_obj.TemperatureUpperThresholdCritical == 0 then
                    return false
                end
                return om_obj.ReadingCelsius >= om_obj.TemperatureUpperThresholdCritical
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID),
                    om_obj.ReadingCelsius, om_obj.TemperatureUpperThresholdCritical)
            end
        },
        Event_OpticalModule1Die0_SFPLowerTempInfo = {
            EventKeyId = "Port.PortOpticalModuleLowTempAlarm",
            EventCode = '0x29000015',
            prop_names = {"ReadingCelsius", "TemperatureLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj.ReadingCelsius > 127 or om_obj.ReadingCelsius == 0 or
                    om_obj.TemperatureLowerThresholdCritical > 127 then
                    return false
                end
                return om_obj.ReadingCelsius <= om_obj.TemperatureLowerThresholdCritical
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID),
                    om_obj.ReadingCelsius, om_obj.TemperatureLowerThresholdCritical)
            end
        },
        Event_OpticalModule1Die0_SFPTempFail = {
            EventKeyId = "NICCard.OMAccessTempFailure",
            EventCode = '0x0D000019',
            prop_names = {"ReadingCelsius"},
            trigger_fun = function(om_obj)
                return om_obj.ReadingCelsius == 32768
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, get_npu_name(om_obj:get_parent().NpuID))
            end
        },
        Event_OpticalModule1Die0_VoltageAlarm = {
            EventKeyId = "Port.PortOpticalModuleVoltageAlarm",
            EventCode = '0x2900000F',
            prop_names = {"SupplyVoltage", "VoltageLowerThresholdCritical", "VoltageUpperThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj.SupplyVoltage >= 255 or om_obj.SupplyVoltage == 0 then
                    return false
                end
                if om_obj.VoltageLowerThresholdCritical ~= 65535 and om_obj.VoltageLowerThresholdCritical ~= 0 and
                    om_obj.SupplyVoltage < om_obj.VoltageLowerThresholdCritical then
                    return true
                end
                if om_obj.VoltageUpperThresholdCritical ~= 65535 and om_obj.VoltageUpperThresholdCritical ~= 0 and
                    om_obj.SupplyVoltage > om_obj.VoltageUpperThresholdCritical then
                    return true
                end
                return false
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '', get_npu_name(om_obj:get_parent().NpuID),
                    om_obj.SupplyVoltage)
            end
        },
        Event_OpticalModule1Die0_RxLoss = {
            EventKeyId = "Port.PortOpticalModuleSignalAlarm",
            EventCode = '0x29000013',
            prop_names = {"RxLoS", "LinkStatusNumeric"},
            trigger_fun = function(om_obj)
                if om_obj.RxLoS ~= 0 and om_obj:get_parent().LinkStatusNumeric == 1 then
                    return true
                end
                return false
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_npu_name(om_obj:get_parent().NpuID), 'Rx')
            end,
            identify_arg_index = {4} --第4个参数作为区分
        },
        Event_OpticalModule1Die0_TxLoss = {
            EventKeyId = "Port.PortOpticalModuleSignalAlarm",
            EventCode = '0x29000013',
            prop_names = {"Presence", "TxLoS"},
            trigger_fun = function(om_obj)
                if om_obj.Presence == 0 then
                    return false
                end
                if om_obj.TxLoS ~= 0 then
                    return true
                end
                return false
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_npu_name(om_obj:get_parent().NpuID), 'Tx')
            end,
            identify_arg_index = {4} --第4个参数作为区分
        },
        Event_OpticalModule1Die0_1_RxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"RXInputPowerMilliWatts", "RXUpperThresholdCritical", "RXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return get_channel_event_status(om_obj.RXInputPowerMilliWatts, 1, om_obj.RXUpperThresholdCritical,
                    om_obj.RXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_rx_power(om_obj:get_parent().NpuID, 1))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_2_RxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"RXInputPowerMilliWatts", "RXUpperThresholdCritical", "RXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return get_channel_event_status(om_obj.RXInputPowerMilliWatts, 2, om_obj.RXUpperThresholdCritical,
                    om_obj.RXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_rx_power(om_obj:get_parent().NpuID, 2))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_3_RxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"RXInputPowerMilliWatts", "RXUpperThresholdCritical", "RXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return get_channel_event_status(om_obj.RXInputPowerMilliWatts, 3, om_obj.RXUpperThresholdCritical,
                    om_obj.RXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_rx_power(om_obj:get_parent().NpuID, 3))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_4_RxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"RXInputPowerMilliWatts", "RXUpperThresholdCritical", "RXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return get_channel_event_status(om_obj.RXInputPowerMilliWatts, 4, om_obj.RXUpperThresholdCritical,
                    om_obj.RXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_rx_power(om_obj:get_parent().NpuID, 4))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_5_RxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"RXInputPowerMilliWatts", "RXUpperThresholdCritical", "RXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return get_channel_event_status(om_obj.RXInputPowerMilliWatts, 5, om_obj.RXUpperThresholdCritical,
                    om_obj.RXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_rx_power(om_obj:get_parent().NpuID, 5))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_6_RxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"RXInputPowerMilliWatts", "RXUpperThresholdCritical", "RXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return get_channel_event_status(om_obj.RXInputPowerMilliWatts, 6, om_obj.RXUpperThresholdCritical,
                    om_obj.RXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_rx_power(om_obj:get_parent().NpuID, 6))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_7_RxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"RXInputPowerMilliWatts", "RXUpperThresholdCritical", "RXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return get_channel_event_status(om_obj.RXInputPowerMilliWatts, 7, om_obj.RXUpperThresholdCritical,
                    om_obj.RXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_rx_power(om_obj:get_parent().NpuID, 7))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_8_RxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"RXInputPowerMilliWatts", "RXUpperThresholdCritical", "RXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                if om_obj:get_parent().LinkStatus == 'Disconnected' then
                    return false
                end
                return get_channel_event_status(om_obj.RXInputPowerMilliWatts, 8, om_obj.RXUpperThresholdCritical,
                    om_obj.RXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_rx_power(om_obj:get_parent().NpuID, 8))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },

        Event_OpticalModule1Die0_1_TxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"TXOutputPowerMilliWatts", "Power_TXUpperThresholdCritical",
                "Power_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXOutputPowerMilliWatts, 1,
                    om_obj.Power_TXUpperThresholdCritical, om_obj.Power_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_tx_power(om_obj:get_parent().NpuID, 1))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_2_TxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"TXOutputPowerMilliWatts", "Power_TXUpperThresholdCritical",
                "Power_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXOutputPowerMilliWatts, 2,
                    om_obj.Power_TXUpperThresholdCritical, om_obj.Power_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_tx_power(om_obj:get_parent().NpuID, 2))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_3_TxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"TXOutputPowerMilliWatts", "Power_TXUpperThresholdCritical",
                "Power_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXOutputPowerMilliWatts, 3,
                    om_obj.Power_TXUpperThresholdCritical, om_obj.Power_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_tx_power(om_obj:get_parent().NpuID, 3))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_4_TxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"TXOutputPowerMilliWatts", "Power_TXUpperThresholdCritical",
                "Power_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXOutputPowerMilliWatts, 4,
                    om_obj.Power_TXUpperThresholdCritical, om_obj.Power_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_tx_power(om_obj:get_parent().NpuID, 4))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_5_TxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"TXOutputPowerMilliWatts", "Power_TXUpperThresholdCritical",
                "Power_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXOutputPowerMilliWatts, 5,
                    om_obj.Power_TXUpperThresholdCritical, om_obj.Power_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_tx_power(om_obj:get_parent().NpuID, 5))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_6_TxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"TXOutputPowerMilliWatts", "Power_TXUpperThresholdCritical",
                "Power_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXOutputPowerMilliWatts, 6,
                    om_obj.Power_TXUpperThresholdCritical, om_obj.Power_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_tx_power(om_obj:get_parent().NpuID, 6))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_7_TxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"TXOutputPowerMilliWatts", "Power_TXUpperThresholdCritical",
                "Power_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXOutputPowerMilliWatts, 7,
                    om_obj.Power_TXUpperThresholdCritical, om_obj.Power_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_tx_power(om_obj:get_parent().NpuID, 7))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_8_TxPowerAlarm = {
            EventKeyId = "Port.PortOpticalModulePowerAlarm",
            EventCode = '0x29000017',
            prop_names = {"TXOutputPowerMilliWatts", "Power_TXUpperThresholdCritical",
                "Power_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXOutputPowerMilliWatts, 8,
                    om_obj.Power_TXUpperThresholdCritical, om_obj.Power_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_tx_power(om_obj:get_parent().NpuID, 8))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },

        Event_OpticalModule1Die0_1_TxBiasAlarm = {
            EventKeyId = "Port.PortOpticalModuleCurrentAlarm",
            EventCode = '0x29000011',
            prop_names = {"TXBiasCurrentMilliAmps", "BC_TXUpperThresholdCritical", "BC_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXBiasCurrentMilliAmps, 1, om_obj.BC_TXUpperThresholdCritical,
                    om_obj.BC_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name,
                    '', get_arg_bias(om_obj:get_parent().NpuID, 1),
                    get_arg_bias_val(om_obj.TXBiasCurrentMilliAmps, 1))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_2_TxBiasAlarm = {
            EventKeyId = "Port.PortOpticalModuleCurrentAlarm",
            EventCode = '0x29000011',
            prop_names = {"TXBiasCurrentMilliAmps", "BC_TXUpperThresholdCritical", "BC_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXBiasCurrentMilliAmps, 2, om_obj.BC_TXUpperThresholdCritical,
                    om_obj.BC_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_arg_bias(om_obj:get_parent().NpuID, 2),
                    get_arg_bias_val(om_obj.TXBiasCurrentMilliAmps, 2))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_3_TxBiasAlarm = {
            EventKeyId = "Port.PortOpticalModuleCurrentAlarm",
            EventCode = '0x29000011',
            prop_names = {"TXBiasCurrentMilliAmps", "BC_TXUpperThresholdCritical", "BC_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXBiasCurrentMilliAmps, 3, om_obj.BC_TXUpperThresholdCritical,
                    om_obj.BC_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_arg_bias(om_obj:get_parent().NpuID, 3),
                    get_arg_bias_val(om_obj.TXBiasCurrentMilliAmps, 3))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_4_TxBiasAlarm = {
            EventKeyId = "Port.PortOpticalModuleCurrentAlarm",
            EventCode = '0x29000011',
            prop_names = {"TXBiasCurrentMilliAmps", "BC_TXUpperThresholdCritical", "BC_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXBiasCurrentMilliAmps, 4, om_obj.BC_TXUpperThresholdCritical,
                    om_obj.BC_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_arg_bias(om_obj:get_parent().NpuID, 4),
                    get_arg_bias_val(om_obj.TXBiasCurrentMilliAmps, 4))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_5_TxBiasAlarm = {
            EventKeyId = "Port.PortOpticalModuleCurrentAlarm",
            EventCode = '0x29000011',
            prop_names = {"TXBiasCurrentMilliAmps", "BC_TXUpperThresholdCritical", "BC_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXBiasCurrentMilliAmps, 5, om_obj.BC_TXUpperThresholdCritical,
                    om_obj.BC_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_arg_bias(om_obj:get_parent().NpuID, 5),
                    get_arg_bias_val(om_obj.TXBiasCurrentMilliAmps, 5))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_6_TxBiasAlarm = {
            EventKeyId = "Port.PortOpticalModuleCurrentAlarm",
            EventCode = '0x29000011',
            prop_names = {"TXBiasCurrentMilliAmps", "BC_TXUpperThresholdCritical", "BC_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXBiasCurrentMilliAmps, 6, om_obj.BC_TXUpperThresholdCritical,
                    om_obj.BC_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_arg_bias(om_obj:get_parent().NpuID, 6),
                    get_arg_bias_val(om_obj.TXBiasCurrentMilliAmps, 6))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_7_TxBiasAlarm = {
            EventKeyId = "Port.PortOpticalModuleCurrentAlarm",
            EventCode = '0x29000011',
            prop_names = {"TXBiasCurrentMilliAmps", "BC_TXUpperThresholdCritical", "BC_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXBiasCurrentMilliAmps, 7, om_obj.BC_TXUpperThresholdCritical,
                    om_obj.BC_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_arg_bias(om_obj:get_parent().NpuID, 7),
                    get_arg_bias_val(om_obj.TXBiasCurrentMilliAmps, 7))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        },
        Event_OpticalModule1Die0_8_TxBiasAlarm = {
            EventKeyId = "Port.PortOpticalModuleCurrentAlarm",
            EventCode = '0x29000011',
            prop_names = {"TXBiasCurrentMilliAmps", "BC_TXUpperThresholdCritical", "BC_TXLowerThresholdCritical"},
            trigger_fun = function(om_obj)
                return get_channel_event_status(om_obj.TXBiasCurrentMilliAmps, 8, om_obj.BC_TXUpperThresholdCritical,
                    om_obj.BC_TXLowerThresholdCritical)
            end,
            get_args = function(om_obj)
                return table.pack(om_obj:get_parent():get_parent().Name, '',
                    get_arg_bias(om_obj:get_parent().NpuID, 8),
                    get_arg_bias_val(om_obj.TXBiasCurrentMilliAmps, 8))
            end,
            identify_arg_index = {3} --第3个参数作为区分
        }
    }
}

return npu_om