-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local validate = require 'mc.validate'
local utils = require 'mc.utils'
local create_enum_type = require 'mc.enum'

local msg = {}
local defs = {}

---@class defs.OccupationMode: Enum
local EOccupationMode = create_enum_type('OccupationMode')
EOccupationMode.default = EOccupationMode.new(2147483647)
EOccupationMode.struct = nil
EOccupationMode.Shared = EOccupationMode.new(0)
EOccupationMode.Exclusive = EOccupationMode.new(1)

defs.OccupationMode = EOccupationMode

---@class defs.SessionType: Enum
local ESessionType = create_enum_type('SessionType')
ESessionType.default = ESessionType.new(2147483647)
ESessionType.struct = nil
ESessionType.GUI = ESessionType.new(0)
ESessionType.Redfish = ESessionType.new(1)
ESessionType.CLI = ESessionType.new(2)
ESessionType.GUI_SSO = ESessionType.new(3)
ESessionType.KVM = ESessionType.new(4)
ESessionType.VNC = ESessionType.new(5)
ESessionType.KVM_VMM = ESessionType.new(6)
ESessionType.Video = ESessionType.new(7)

defs.SessionType = ESessionType

---@class defs.CLISession
---@field SessionId string
---@field Username string
---@field Ip string
---@field LoginTimeStr string
---@field Privilege integer
---@field SessionType integer
local TCLISession = {}
TCLISession.__index = TCLISession
TCLISession.group = {}

local function TCLISession_from_obj(obj)
    return setmetatable(obj, TCLISession)
end

function TCLISession.new(SessionId, Username, Ip, LoginTimeStr, Privilege, SessionType)
    return TCLISession_from_obj({
        SessionId = SessionId,
        Username = Username,
        Ip = Ip,
        LoginTimeStr = LoginTimeStr,
        Privilege = Privilege,
        SessionType = SessionType
    })
end
---@param obj defs.CLISession
function TCLISession:init_from_obj(obj)
    self.SessionId = obj.SessionId
    self.Username = obj.Username
    self.Ip = obj.Ip
    self.LoginTimeStr = obj.LoginTimeStr
    self.Privilege = obj.Privilege
    self.SessionType = obj.SessionType
end

function TCLISession:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TCLISession.group)
end

TCLISession.from_obj = TCLISession_from_obj

TCLISession.proto_property = {
    'SessionId', 'Username', 'Ip', 'LoginTimeStr', 'Privilege', 'SessionType'
}

TCLISession.default = {'', '', '', '', 0, 0}

TCLISession.struct = {
    {name = 'SessionId', is_array = false, struct = nil},
    {name = 'Username', is_array = false, struct = nil},
    {name = 'Ip', is_array = false, struct = nil},
    {name = 'LoginTimeStr', is_array = false, struct = nil},
    {name = 'Privilege', is_array = false, struct = nil},
    {name = 'SessionType', is_array = false, struct = nil}
}

function TCLISession:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'SessionId', self.SessionId, 'string', false, errs)
    validate.Required(prefix .. 'Username', self.Username, 'string', false, errs)
    validate.Required(prefix .. 'Ip', self.Ip, 'string', false, errs)
    validate.Required(prefix .. 'LoginTimeStr', self.LoginTimeStr, 'string', false, errs)
    validate.Required(prefix .. 'Privilege', self.Privilege, 'uint8', false, errs)
    validate.Required(prefix .. 'SessionType', self.SessionType, 'uint8', false, errs)

    TCLISession:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TCLISession.proto_property, errs)
    return self
end

function TCLISession:unpack(_)
    return self.SessionId, self.Username, self.Ip, self.LoginTimeStr, self.Privilege,
        self.SessionType
end

defs.CLISession = TCLISession

---@class Sessions.GetCLISessionListRsp
---@field CLISessions defs.CLISession[]
local TGetCLISessionListRsp = {}
TGetCLISessionListRsp.__index = TGetCLISessionListRsp
TGetCLISessionListRsp.group = {}

local function TGetCLISessionListRsp_from_obj(obj)
    obj.CLISessions = utils.from_obj(defs.CLISession, obj.CLISessions, true)
    return setmetatable(obj, TGetCLISessionListRsp)
end

function TGetCLISessionListRsp.new(CLISessions)
    return TGetCLISessionListRsp_from_obj({CLISessions = CLISessions})
end
---@param obj Sessions.GetCLISessionListRsp
function TGetCLISessionListRsp:init_from_obj(obj)
    self.CLISessions = obj.CLISessions
end

function TGetCLISessionListRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetCLISessionListRsp.group)
end

TGetCLISessionListRsp.from_obj = TGetCLISessionListRsp_from_obj

TGetCLISessionListRsp.proto_property = {'CLISessions'}

TGetCLISessionListRsp.default = {{}}

TGetCLISessionListRsp.struct = {
    {name = 'CLISessions', is_array = true, struct = defs.CLISession.struct}
}

function TGetCLISessionListRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.RequiredArray(prefix .. 'CLISessions', self.CLISessions, 'defs.CLISession', false, errs)

    TGetCLISessionListRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetCLISessionListRsp.proto_property, errs)
    return self
end

function TGetCLISessionListRsp:unpack(raw)
    return utils.unpack(raw, self.CLISessions, true)
end

msg.GetCLISessionListRsp = TGetCLISessionListRsp

---@class Sessions.GetCLISessionListReq
local TGetCLISessionListReq = {}
TGetCLISessionListReq.__index = TGetCLISessionListReq
TGetCLISessionListReq.group = {}

local function TGetCLISessionListReq_from_obj(obj)
    return setmetatable(obj, TGetCLISessionListReq)
end

function TGetCLISessionListReq.new()
    return TGetCLISessionListReq_from_obj({})
end
---@param obj Sessions.GetCLISessionListReq
function TGetCLISessionListReq:init_from_obj(obj)

end

function TGetCLISessionListReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetCLISessionListReq.group)
end

TGetCLISessionListReq.from_obj = TGetCLISessionListReq_from_obj

TGetCLISessionListReq.proto_property = {}

TGetCLISessionListReq.default = {}

TGetCLISessionListReq.struct = {}

function TGetCLISessionListReq:validate(prefix, errs)
    prefix = prefix or ''

    TGetCLISessionListReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetCLISessionListReq.proto_property, errs)
    return self
end

function TGetCLISessionListReq:unpack(_)
end

msg.GetCLISessionListReq = TGetCLISessionListReq

---@class Sessions.SessionHeartBeatRsp
local TSessionHeartBeatRsp = {}
TSessionHeartBeatRsp.__index = TSessionHeartBeatRsp
TSessionHeartBeatRsp.group = {}

local function TSessionHeartBeatRsp_from_obj(obj)
    return setmetatable(obj, TSessionHeartBeatRsp)
end

function TSessionHeartBeatRsp.new()
    return TSessionHeartBeatRsp_from_obj({})
end
---@param obj Sessions.SessionHeartBeatRsp
function TSessionHeartBeatRsp:init_from_obj(obj)

end

function TSessionHeartBeatRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TSessionHeartBeatRsp.group)
end

TSessionHeartBeatRsp.from_obj = TSessionHeartBeatRsp_from_obj

TSessionHeartBeatRsp.proto_property = {}

TSessionHeartBeatRsp.default = {}

TSessionHeartBeatRsp.struct = {}

function TSessionHeartBeatRsp:validate(prefix, errs)
    prefix = prefix or ''

    TSessionHeartBeatRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TSessionHeartBeatRsp.proto_property, errs)
    return self
end

function TSessionHeartBeatRsp:unpack(_)
end

msg.SessionHeartBeatRsp = TSessionHeartBeatRsp

---@class Sessions.SessionHeartBeatReq
---@field SessionId string
---@field RemainActiveSeconds integer
local TSessionHeartBeatReq = {}
TSessionHeartBeatReq.__index = TSessionHeartBeatReq
TSessionHeartBeatReq.group = {}

local function TSessionHeartBeatReq_from_obj(obj)
    return setmetatable(obj, TSessionHeartBeatReq)
end

function TSessionHeartBeatReq.new(SessionId, RemainActiveSeconds)
    return TSessionHeartBeatReq_from_obj({
        SessionId = SessionId,
        RemainActiveSeconds = RemainActiveSeconds
    })
end
---@param obj Sessions.SessionHeartBeatReq
function TSessionHeartBeatReq:init_from_obj(obj)
    self.SessionId = obj.SessionId
    self.RemainActiveSeconds = obj.RemainActiveSeconds
end

function TSessionHeartBeatReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TSessionHeartBeatReq.group)
end

TSessionHeartBeatReq.from_obj = TSessionHeartBeatReq_from_obj

TSessionHeartBeatReq.proto_property = {'SessionId', 'RemainActiveSeconds'}

TSessionHeartBeatReq.default = {'', 0}

TSessionHeartBeatReq.struct = {
    {name = 'SessionId', is_array = false, struct = nil},
    {name = 'RemainActiveSeconds', is_array = false, struct = nil}
}

function TSessionHeartBeatReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'SessionId', self.SessionId, 'string', false, errs)
    validate.Required(prefix .. 'RemainActiveSeconds', self.RemainActiveSeconds, 'uint8', false,
        errs)

    TSessionHeartBeatReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TSessionHeartBeatReq.proto_property, errs)
    return self
end

function TSessionHeartBeatReq:unpack(_)
    return self.SessionId, self.RemainActiveSeconds
end

msg.SessionHeartBeatReq = TSessionHeartBeatReq

---@class Sessions.GetSessionLogoutTypeRsp
---@field LogoutType integer
local TGetSessionLogoutTypeRsp = {}
TGetSessionLogoutTypeRsp.__index = TGetSessionLogoutTypeRsp
TGetSessionLogoutTypeRsp.group = {}

local function TGetSessionLogoutTypeRsp_from_obj(obj)
    return setmetatable(obj, TGetSessionLogoutTypeRsp)
end

function TGetSessionLogoutTypeRsp.new(LogoutType)
    return TGetSessionLogoutTypeRsp_from_obj({LogoutType = LogoutType})
end
---@param obj Sessions.GetSessionLogoutTypeRsp
function TGetSessionLogoutTypeRsp:init_from_obj(obj)
    self.LogoutType = obj.LogoutType
end

function TGetSessionLogoutTypeRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetSessionLogoutTypeRsp.group)
end

TGetSessionLogoutTypeRsp.from_obj = TGetSessionLogoutTypeRsp_from_obj

TGetSessionLogoutTypeRsp.proto_property = {'LogoutType'}

TGetSessionLogoutTypeRsp.default = {0}

TGetSessionLogoutTypeRsp.struct = {{name = 'LogoutType', is_array = false, struct = nil}}

function TGetSessionLogoutTypeRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'LogoutType', self.LogoutType, 'uint8', false, errs)

    TGetSessionLogoutTypeRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetSessionLogoutTypeRsp.proto_property, errs)
    return self
end

function TGetSessionLogoutTypeRsp:unpack(_)
    return self.LogoutType
end

msg.GetSessionLogoutTypeRsp = TGetSessionLogoutTypeRsp

---@class Sessions.GetSessionLogoutTypeReq
---@field SessionId string
local TGetSessionLogoutTypeReq = {}
TGetSessionLogoutTypeReq.__index = TGetSessionLogoutTypeReq
TGetSessionLogoutTypeReq.group = {}

local function TGetSessionLogoutTypeReq_from_obj(obj)
    return setmetatable(obj, TGetSessionLogoutTypeReq)
end

function TGetSessionLogoutTypeReq.new(SessionId)
    return TGetSessionLogoutTypeReq_from_obj({SessionId = SessionId})
end
---@param obj Sessions.GetSessionLogoutTypeReq
function TGetSessionLogoutTypeReq:init_from_obj(obj)
    self.SessionId = obj.SessionId
end

function TGetSessionLogoutTypeReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetSessionLogoutTypeReq.group)
end

TGetSessionLogoutTypeReq.from_obj = TGetSessionLogoutTypeReq_from_obj

TGetSessionLogoutTypeReq.proto_property = {'SessionId'}

TGetSessionLogoutTypeReq.default = {''}

TGetSessionLogoutTypeReq.struct = {{name = 'SessionId', is_array = false, struct = nil}}

function TGetSessionLogoutTypeReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'SessionId', self.SessionId, 'string', false, errs)

    TGetSessionLogoutTypeReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetSessionLogoutTypeReq.proto_property, errs)
    return self
end

function TGetSessionLogoutTypeReq:unpack(_)
    return self.SessionId
end

msg.GetSessionLogoutTypeReq = TGetSessionLogoutTypeReq

---@class Sessions.ValidateSessionRsp
---@field SessionId string
local TValidateSessionRsp = {}
TValidateSessionRsp.__index = TValidateSessionRsp
TValidateSessionRsp.group = {}

local function TValidateSessionRsp_from_obj(obj)
    return setmetatable(obj, TValidateSessionRsp)
end

function TValidateSessionRsp.new(SessionId)
    return TValidateSessionRsp_from_obj({SessionId = SessionId})
end
---@param obj Sessions.ValidateSessionRsp
function TValidateSessionRsp:init_from_obj(obj)
    self.SessionId = obj.SessionId
end

function TValidateSessionRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TValidateSessionRsp.group)
end

TValidateSessionRsp.from_obj = TValidateSessionRsp_from_obj

TValidateSessionRsp.proto_property = {'SessionId'}

TValidateSessionRsp.default = {''}

TValidateSessionRsp.struct = {{name = 'SessionId', is_array = false, struct = nil}}

function TValidateSessionRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'SessionId', self.SessionId, 'string', false, errs)

    TValidateSessionRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TValidateSessionRsp.proto_property, errs)
    return self
end

function TValidateSessionRsp:unpack(_)
    return self.SessionId
end

msg.ValidateSessionRsp = TValidateSessionRsp

---@class Sessions.ValidateSessionReq
---@field SessionType integer
---@field Token string
local TValidateSessionReq = {}
TValidateSessionReq.__index = TValidateSessionReq
TValidateSessionReq.group = {}

local function TValidateSessionReq_from_obj(obj)
    return setmetatable(obj, TValidateSessionReq)
end

function TValidateSessionReq.new(SessionType, Token)
    return TValidateSessionReq_from_obj({SessionType = SessionType, Token = Token})
end
---@param obj Sessions.ValidateSessionReq
function TValidateSessionReq:init_from_obj(obj)
    self.SessionType = obj.SessionType
    self.Token = obj.Token
end

function TValidateSessionReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TValidateSessionReq.group)
end

TValidateSessionReq.from_obj = TValidateSessionReq_from_obj

TValidateSessionReq.proto_property = {'SessionType', 'Token'}

TValidateSessionReq.default = {0, ''}

TValidateSessionReq.struct = {
    {name = 'SessionType', is_array = false, struct = nil},
    {name = 'Token', is_array = false, struct = nil}
}

function TValidateSessionReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'SessionType', self.SessionType, 'uint8', false, errs)
    validate.Required(prefix .. 'Token', self.Token, 'string', false, errs)

    TValidateSessionReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TValidateSessionReq.proto_property, errs)
    return self
end

function TValidateSessionReq:unpack(_)
    return self.SessionType, self.Token
end

msg.ValidateSessionReq = TValidateSessionReq

---@class Sessions.ValidateSessionWithCsrfRsp
---@field SessionId string
local TValidateSessionWithCsrfRsp = {}
TValidateSessionWithCsrfRsp.__index = TValidateSessionWithCsrfRsp
TValidateSessionWithCsrfRsp.group = {}

local function TValidateSessionWithCsrfRsp_from_obj(obj)
    return setmetatable(obj, TValidateSessionWithCsrfRsp)
end

function TValidateSessionWithCsrfRsp.new(SessionId)
    return TValidateSessionWithCsrfRsp_from_obj({SessionId = SessionId})
end
---@param obj Sessions.ValidateSessionWithCsrfRsp
function TValidateSessionWithCsrfRsp:init_from_obj(obj)
    self.SessionId = obj.SessionId
end

function TValidateSessionWithCsrfRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TValidateSessionWithCsrfRsp.group)
end

TValidateSessionWithCsrfRsp.from_obj = TValidateSessionWithCsrfRsp_from_obj

TValidateSessionWithCsrfRsp.proto_property = {'SessionId'}

TValidateSessionWithCsrfRsp.default = {''}

TValidateSessionWithCsrfRsp.struct = {{name = 'SessionId', is_array = false, struct = nil}}

function TValidateSessionWithCsrfRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'SessionId', self.SessionId, 'string', false, errs)

    TValidateSessionWithCsrfRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TValidateSessionWithCsrfRsp.proto_property, errs)
    return self
end

function TValidateSessionWithCsrfRsp:unpack(_)
    return self.SessionId
end

msg.ValidateSessionWithCsrfRsp = TValidateSessionWithCsrfRsp

---@class Sessions.ValidateSessionWithCsrfReq
---@field SessionType integer
---@field Token string
---@field CsrfToken string
local TValidateSessionWithCsrfReq = {}
TValidateSessionWithCsrfReq.__index = TValidateSessionWithCsrfReq
TValidateSessionWithCsrfReq.group = {}

local function TValidateSessionWithCsrfReq_from_obj(obj)
    return setmetatable(obj, TValidateSessionWithCsrfReq)
end

function TValidateSessionWithCsrfReq.new(SessionType, Token, CsrfToken)
    return TValidateSessionWithCsrfReq_from_obj({
        SessionType = SessionType,
        Token = Token,
        CsrfToken = CsrfToken
    })
end
---@param obj Sessions.ValidateSessionWithCsrfReq
function TValidateSessionWithCsrfReq:init_from_obj(obj)
    self.SessionType = obj.SessionType
    self.Token = obj.Token
    self.CsrfToken = obj.CsrfToken
end

function TValidateSessionWithCsrfReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TValidateSessionWithCsrfReq.group)
end

TValidateSessionWithCsrfReq.from_obj = TValidateSessionWithCsrfReq_from_obj

TValidateSessionWithCsrfReq.proto_property = {'SessionType', 'Token', 'CsrfToken'}

TValidateSessionWithCsrfReq.default = {0, '', ''}

TValidateSessionWithCsrfReq.struct = {
    {name = 'SessionType', is_array = false, struct = nil},
    {name = 'Token', is_array = false, struct = nil},
    {name = 'CsrfToken', is_array = false, struct = nil}
}

function TValidateSessionWithCsrfReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'SessionType', self.SessionType, 'uint8', false, errs)
    validate.Required(prefix .. 'Token', self.Token, 'string', false, errs)
    validate.Required(prefix .. 'CsrfToken', self.CsrfToken, 'string', false, errs)

    TValidateSessionWithCsrfReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TValidateSessionWithCsrfReq.proto_property, errs)
    return self
end

function TValidateSessionWithCsrfReq:unpack(_)
    return self.SessionType, self.Token, self.CsrfToken
end

msg.ValidateSessionWithCsrfReq = TValidateSessionWithCsrfReq

---@class Sessions.NewVNCSessionRsp
---@field VNCSessionId string
local TNewVNCSessionRsp = {}
TNewVNCSessionRsp.__index = TNewVNCSessionRsp
TNewVNCSessionRsp.group = {}

local function TNewVNCSessionRsp_from_obj(obj)
    return setmetatable(obj, TNewVNCSessionRsp)
end

function TNewVNCSessionRsp.new(VNCSessionId)
    return TNewVNCSessionRsp_from_obj({VNCSessionId = VNCSessionId})
end
---@param obj Sessions.NewVNCSessionRsp
function TNewVNCSessionRsp:init_from_obj(obj)
    self.VNCSessionId = obj.VNCSessionId
end

function TNewVNCSessionRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TNewVNCSessionRsp.group)
end

TNewVNCSessionRsp.from_obj = TNewVNCSessionRsp_from_obj

TNewVNCSessionRsp.proto_property = {'VNCSessionId'}

TNewVNCSessionRsp.default = {''}

TNewVNCSessionRsp.struct = {{name = 'VNCSessionId', is_array = false, struct = nil}}

function TNewVNCSessionRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'VNCSessionId', self.VNCSessionId, 'string', false, errs)

    TNewVNCSessionRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TNewVNCSessionRsp.proto_property, errs)
    return self
end

function TNewVNCSessionRsp:unpack(_)
    return self.VNCSessionId
end

msg.NewVNCSessionRsp = TNewVNCSessionRsp

---@class Sessions.NewVNCSessionReq
---@field CipherText string
---@field AuthChallenge string
---@field SessionMode integer
local TNewVNCSessionReq = {}
TNewVNCSessionReq.__index = TNewVNCSessionReq
TNewVNCSessionReq.group = {}

local function TNewVNCSessionReq_from_obj(obj)
    return setmetatable(obj, TNewVNCSessionReq)
end

function TNewVNCSessionReq.new(CipherText, AuthChallenge, SessionMode)
    return TNewVNCSessionReq_from_obj({
        CipherText = CipherText,
        AuthChallenge = AuthChallenge,
        SessionMode = SessionMode
    })
end
---@param obj Sessions.NewVNCSessionReq
function TNewVNCSessionReq:init_from_obj(obj)
    self.CipherText = obj.CipherText
    self.AuthChallenge = obj.AuthChallenge
    self.SessionMode = obj.SessionMode
end

function TNewVNCSessionReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TNewVNCSessionReq.group)
end

TNewVNCSessionReq.from_obj = TNewVNCSessionReq_from_obj

TNewVNCSessionReq.proto_property = {'CipherText', 'AuthChallenge', 'SessionMode'}

TNewVNCSessionReq.default = {'', '', 0}

TNewVNCSessionReq.struct = {
    {name = 'CipherText', is_array = false, struct = nil},
    {name = 'AuthChallenge', is_array = false, struct = nil},
    {name = 'SessionMode', is_array = false, struct = nil}
}

function TNewVNCSessionReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CipherText', self.CipherText, 'string', false, errs)
    validate.Required(prefix .. 'AuthChallenge', self.AuthChallenge, 'string', false, errs)
    validate.Required(prefix .. 'SessionMode', self.SessionMode, 'uint8', false, errs)

    TNewVNCSessionReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TNewVNCSessionReq.proto_property, errs)
    return self
end

function TNewVNCSessionReq:unpack(_)
    return self.CipherText, self.AuthChallenge, self.SessionMode
end

msg.NewVNCSessionReq = TNewVNCSessionReq

---@class Sessions.NewRemoteConsoleSessionRsp
---@field RemoteConsoleToken string
---@field RemoteConsoleSessionId string
local TNewRemoteConsoleSessionRsp = {}
TNewRemoteConsoleSessionRsp.__index = TNewRemoteConsoleSessionRsp
TNewRemoteConsoleSessionRsp.group = {}

local function TNewRemoteConsoleSessionRsp_from_obj(obj)
    return setmetatable(obj, TNewRemoteConsoleSessionRsp)
end

function TNewRemoteConsoleSessionRsp.new(RemoteConsoleToken, RemoteConsoleSessionId)
    return TNewRemoteConsoleSessionRsp_from_obj({
        RemoteConsoleToken = RemoteConsoleToken,
        RemoteConsoleSessionId = RemoteConsoleSessionId
    })
end
---@param obj Sessions.NewRemoteConsoleSessionRsp
function TNewRemoteConsoleSessionRsp:init_from_obj(obj)
    self.RemoteConsoleToken = obj.RemoteConsoleToken
    self.RemoteConsoleSessionId = obj.RemoteConsoleSessionId
end

function TNewRemoteConsoleSessionRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TNewRemoteConsoleSessionRsp.group)
end

TNewRemoteConsoleSessionRsp.from_obj = TNewRemoteConsoleSessionRsp_from_obj

TNewRemoteConsoleSessionRsp.proto_property = {'RemoteConsoleToken', 'RemoteConsoleSessionId'}

TNewRemoteConsoleSessionRsp.default = {'', ''}

TNewRemoteConsoleSessionRsp.struct = {
    {name = 'RemoteConsoleToken', is_array = false, struct = nil},
    {name = 'RemoteConsoleSessionId', is_array = false, struct = nil}
}

function TNewRemoteConsoleSessionRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'RemoteConsoleToken', self.RemoteConsoleToken, 'string', false, errs)
    validate.Required(prefix .. 'RemoteConsoleSessionId', self.RemoteConsoleSessionId, 'string',
        false, errs)

    TNewRemoteConsoleSessionRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TNewRemoteConsoleSessionRsp.proto_property, errs)
    return self
end

function TNewRemoteConsoleSessionRsp:unpack(_)
    return self.RemoteConsoleToken, self.RemoteConsoleSessionId
end

msg.NewRemoteConsoleSessionRsp = TNewRemoteConsoleSessionRsp

---@class Sessions.NewRemoteConsoleSessionReq
---@field Token string
---@field SessionType integer
---@field SessionMode integer
local TNewRemoteConsoleSessionReq = {}
TNewRemoteConsoleSessionReq.__index = TNewRemoteConsoleSessionReq
TNewRemoteConsoleSessionReq.group = {}

local function TNewRemoteConsoleSessionReq_from_obj(obj)
    return setmetatable(obj, TNewRemoteConsoleSessionReq)
end

function TNewRemoteConsoleSessionReq.new(Token, SessionType, SessionMode)
    return TNewRemoteConsoleSessionReq_from_obj({
        Token = Token,
        SessionType = SessionType,
        SessionMode = SessionMode
    })
end
---@param obj Sessions.NewRemoteConsoleSessionReq
function TNewRemoteConsoleSessionReq:init_from_obj(obj)
    self.Token = obj.Token
    self.SessionType = obj.SessionType
    self.SessionMode = obj.SessionMode
end

function TNewRemoteConsoleSessionReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TNewRemoteConsoleSessionReq.group)
end

TNewRemoteConsoleSessionReq.from_obj = TNewRemoteConsoleSessionReq_from_obj

TNewRemoteConsoleSessionReq.proto_property = {'Token', 'SessionType', 'SessionMode'}

TNewRemoteConsoleSessionReq.default = {'', 0, 0}

TNewRemoteConsoleSessionReq.struct = {
    {name = 'Token', is_array = false, struct = nil},
    {name = 'SessionType', is_array = false, struct = nil},
    {name = 'SessionMode', is_array = false, struct = nil}
}

function TNewRemoteConsoleSessionReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Token', self.Token, 'string', false, errs)
    validate.Required(prefix .. 'SessionType', self.SessionType, 'uint8', false, errs)
    validate.Required(prefix .. 'SessionMode', self.SessionMode, 'uint8', false, errs)

    TNewRemoteConsoleSessionReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TNewRemoteConsoleSessionReq.proto_property, errs)
    return self
end

function TNewRemoteConsoleSessionReq:unpack(_)
    return self.Token, self.SessionType, self.SessionMode
end

msg.NewRemoteConsoleSessionReq = TNewRemoteConsoleSessionReq

---@class Sessions.NewSessionRsp
---@field Token string
---@field CsrfToken string
---@field SessionId string
local TNewSessionRsp = {}
TNewSessionRsp.__index = TNewSessionRsp
TNewSessionRsp.group = {}

local function TNewSessionRsp_from_obj(obj)
    return setmetatable(obj, TNewSessionRsp)
end

function TNewSessionRsp.new(Token, CsrfToken, SessionId)
    return TNewSessionRsp_from_obj({Token = Token, CsrfToken = CsrfToken, SessionId = SessionId})
end
---@param obj Sessions.NewSessionRsp
function TNewSessionRsp:init_from_obj(obj)
    self.Token = obj.Token
    self.CsrfToken = obj.CsrfToken
    self.SessionId = obj.SessionId
end

function TNewSessionRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TNewSessionRsp.group)
end

TNewSessionRsp.from_obj = TNewSessionRsp_from_obj

TNewSessionRsp.proto_property = {'Token', 'CsrfToken', 'SessionId'}

TNewSessionRsp.default = {'', '', ''}

TNewSessionRsp.struct = {
    {name = 'Token', is_array = false, struct = nil},
    {name = 'CsrfToken', is_array = false, struct = nil},
    {name = 'SessionId', is_array = false, struct = nil}
}

function TNewSessionRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Token', self.Token, 'string', false, errs)
    validate.Required(prefix .. 'CsrfToken', self.CsrfToken, 'string', false, errs)
    validate.Required(prefix .. 'SessionId', self.SessionId, 'string', false, errs)

    TNewSessionRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TNewSessionRsp.proto_property, errs)
    return self
end

function TNewSessionRsp:unpack(_)
    return self.Token, self.CsrfToken, self.SessionId
end

msg.NewSessionRsp = TNewSessionRsp

---@class Sessions.NewSessionReq
---@field UserName string
---@field Password integer[]
---@field SessionType integer
---@field AuthType integer
---@field Ip string
---@field BrowserType integer
local TNewSessionReq = {}
TNewSessionReq.__index = TNewSessionReq
TNewSessionReq.group = {}

local function TNewSessionReq_from_obj(obj)
    return setmetatable(obj, TNewSessionReq)
end

function TNewSessionReq.new(UserName, Password, SessionType, AuthType, Ip, BrowserType)
    return TNewSessionReq_from_obj({
        UserName = UserName,
        Password = Password,
        SessionType = SessionType,
        AuthType = AuthType,
        Ip = Ip,
        BrowserType = BrowserType
    })
end
---@param obj Sessions.NewSessionReq
function TNewSessionReq:init_from_obj(obj)
    self.UserName = obj.UserName
    self.Password = obj.Password
    self.SessionType = obj.SessionType
    self.AuthType = obj.AuthType
    self.Ip = obj.Ip
    self.BrowserType = obj.BrowserType
end

function TNewSessionReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TNewSessionReq.group)
end

TNewSessionReq.from_obj = TNewSessionReq_from_obj

TNewSessionReq.proto_property = {
    'UserName', 'Password', 'SessionType', 'AuthType', 'Ip', 'BrowserType'
}

TNewSessionReq.default = {'', {}, 0, 0, '', 0}

TNewSessionReq.struct = {
    {name = 'UserName', is_array = false, struct = nil},
    {name = 'Password', is_array = true, struct = nil},
    {name = 'SessionType', is_array = false, struct = nil},
    {name = 'AuthType', is_array = false, struct = nil},
    {name = 'Ip', is_array = false, struct = nil},
    {name = 'BrowserType', is_array = false, struct = nil}
}

function TNewSessionReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'UserName', self.UserName, 'string', false, errs)
    validate.RequiredArray(prefix .. 'Password', self.Password, 'uint8', false, errs)
    validate.Required(prefix .. 'SessionType', self.SessionType, 'uint8', false, errs)
    validate.Required(prefix .. 'AuthType', self.AuthType, 'uint8', false, errs)
    validate.Required(prefix .. 'Ip', self.Ip, 'string', false, errs)
    validate.Required(prefix .. 'BrowserType', self.BrowserType, 'uint8', false, errs)

    validate.lens(prefix .. 'UserName', self.UserName, 1, 32, errs)
    validate.lens(prefix .. 'Password', self.Password, 1, 20, errs)

    TNewSessionReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TNewSessionReq.proto_property, errs)
    return self
end

function TNewSessionReq:unpack(_)
    return self.UserName, self.Password, self.SessionType, self.AuthType, self.Ip, self.BrowserType
end

msg.NewSessionReq = TNewSessionReq

local Sessions = {}

local Mdb = require 'mc.mdb'

---@class Sessions: Table
Sessions.Sessions = Mdb.register_interface('bmc.kepler.SessionService.Sessions', {}, {
    NewSession = {'a{ss}sayyysy', 'sss', msg.NewSessionReq, msg.NewSessionRsp},
    NewRemoteConsoleSession = {
        'a{ss}syy', 'ss', msg.NewRemoteConsoleSessionReq, msg.NewRemoteConsoleSessionRsp
    },
    NewVNCSession = {'a{ss}ssy', 's', msg.NewVNCSessionReq, msg.NewVNCSessionRsp},
    ValidateSessionWithCsrf = {
        'a{ss}yss', 's', msg.ValidateSessionWithCsrfReq, msg.ValidateSessionWithCsrfRsp
    },
    ValidateSession = {'a{ss}ys', 's', msg.ValidateSessionReq, msg.ValidateSessionRsp},
    GetSessionLogoutType = {'a{ss}s', 'y', msg.GetSessionLogoutTypeReq, msg.GetSessionLogoutTypeRsp},
    SessionHeartBeat = {'a{ss}sy', '', msg.SessionHeartBeatReq, msg.SessionHeartBeatRsp},
    GetCLISessionList = {'a{ss}', 'a(ssssyy)', msg.GetCLISessionListReq, msg.GetCLISessionListRsp}
}, {})
return Sessions
