-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local validate = require 'mc.validate'
local utils = require 'mc.utils'

local msg = {}
local defs = {}

---@class defs.ObjectInfo
---@field ClassName string
---@field ObjectName string
---@field ObjectProps string
---@field ObjectExtends string
local TObjectInfo = {}
TObjectInfo.__index = TObjectInfo
TObjectInfo.group = {}

local function TObjectInfo_from_obj(obj)
    return setmetatable(obj, TObjectInfo)
end

function TObjectInfo.new(ClassName, ObjectName, ObjectProps, ObjectExtends)
    return TObjectInfo_from_obj({
        ClassName = ClassName,
        ObjectName = ObjectName,
        ObjectProps = ObjectProps,
        ObjectExtends = ObjectExtends
    })
end
---@param obj defs.ObjectInfo
function TObjectInfo:init_from_obj(obj)
    self.ClassName = obj.ClassName
    self.ObjectName = obj.ObjectName
    self.ObjectProps = obj.ObjectProps
    self.ObjectExtends = obj.ObjectExtends
end

function TObjectInfo:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TObjectInfo.group)
end

TObjectInfo.from_obj = TObjectInfo_from_obj

TObjectInfo.proto_property = {'ClassName', 'ObjectName', 'ObjectProps', 'ObjectExtends'}

TObjectInfo.default = {'', '', '', ''}

TObjectInfo.struct = {
    {name = 'ClassName', is_array = false, struct = nil},
    {name = 'ObjectName', is_array = false, struct = nil},
    {name = 'ObjectProps', is_array = false, struct = nil},
    {name = 'ObjectExtends', is_array = false, struct = nil}
}

function TObjectInfo:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'ClassName', self.ClassName, 'string', false, errs)
    validate.Required(prefix .. 'ObjectName', self.ObjectName, 'string', false, errs)
    validate.Required(prefix .. 'ObjectProps', self.ObjectProps, 'string', false, errs)
    validate.Required(prefix .. 'ObjectExtends', self.ObjectExtends, 'string', false, errs)

    TObjectInfo:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TObjectInfo.proto_property, errs)
    return self
end

function TObjectInfo:unpack(_)
    return self.ClassName, self.ObjectName, self.ObjectProps, self.ObjectExtends
end

defs.ObjectInfo = TObjectInfo

---@class ObjectGroup.GetObjectsRsp
---@field Position string
---@field Objects defs.ObjectInfo[]
---@field LifeCycleId integer
local TGetObjectsRsp = {}
TGetObjectsRsp.__index = TGetObjectsRsp
TGetObjectsRsp.group = {}

local function TGetObjectsRsp_from_obj(obj)
    obj.Objects = utils.from_obj(defs.ObjectInfo, obj.Objects, true)
    return setmetatable(obj, TGetObjectsRsp)
end

function TGetObjectsRsp.new(Position, Objects, LifeCycleId)
    return TGetObjectsRsp_from_obj({
        Position = Position,
        Objects = Objects,
        LifeCycleId = LifeCycleId
    })
end
---@param obj ObjectGroup.GetObjectsRsp
function TGetObjectsRsp:init_from_obj(obj)
    self.Position = obj.Position
    self.Objects = obj.Objects
    self.LifeCycleId = obj.LifeCycleId
end

function TGetObjectsRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetObjectsRsp.group)
end

TGetObjectsRsp.from_obj = TGetObjectsRsp_from_obj

TGetObjectsRsp.proto_property = {'Position', 'Objects', 'LifeCycleId'}

TGetObjectsRsp.default = {'', {}, 0}

TGetObjectsRsp.struct = {
    {name = 'Position', is_array = false, struct = nil},
    {name = 'Objects', is_array = true, struct = defs.ObjectInfo.struct},
    {name = 'LifeCycleId', is_array = false, struct = nil}
}

function TGetObjectsRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Position', self.Position, 'string', false, errs)
    validate.RequiredArray(prefix .. 'Objects', self.Objects, 'defs.ObjectInfo', false, errs)
    validate.Required(prefix .. 'LifeCycleId', self.LifeCycleId, 'uint32', false, errs)

    TGetObjectsRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetObjectsRsp.proto_property, errs)
    return self
end

function TGetObjectsRsp:unpack(raw)
    return self.Position, utils.unpack(raw, self.Objects, true), self.LifeCycleId
end

msg.GetObjectsRsp = TGetObjectsRsp

---@class ObjectGroup.GetObjectsReq
---@field Owner string
local TGetObjectsReq = {}
TGetObjectsReq.__index = TGetObjectsReq
TGetObjectsReq.group = {}

local function TGetObjectsReq_from_obj(obj)
    return setmetatable(obj, TGetObjectsReq)
end

function TGetObjectsReq.new(Owner)
    return TGetObjectsReq_from_obj({Owner = Owner})
end
---@param obj ObjectGroup.GetObjectsReq
function TGetObjectsReq:init_from_obj(obj)
    self.Owner = obj.Owner
end

function TGetObjectsReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetObjectsReq.group)
end

TGetObjectsReq.from_obj = TGetObjectsReq_from_obj

TGetObjectsReq.proto_property = {'Owner'}

TGetObjectsReq.default = {''}

TGetObjectsReq.struct = {{name = 'Owner', is_array = false, struct = nil}}

function TGetObjectsReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Owner', self.Owner, 'string', false, errs)

    TGetObjectsReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetObjectsReq.proto_property, errs)
    return self
end

function TGetObjectsReq:unpack(_)
    return self.Owner
end

msg.GetObjectsReq = TGetObjectsReq

local ObjectGroup = {}

local Mdb = require 'mc.mdb'

---@class ObjectGroup: Table
---@field Position string
ObjectGroup.ObjectGroup = Mdb.register_interface('bmc.kepler.ObjectGroup',
    {Position = {'s', nil, true, nil}}, {
        GetObjects = {'a{ss}s', 'sa(ssss)u', msg.GetObjectsReq, msg.GetObjectsRsp}
    }, {})
return ObjectGroup
