-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local validate = require 'mc.validate'
local utils = require 'mc.utils'
local create_enum_type = require 'mc.enum'

local msg = {}
local defs = {}

---@class defs.CertInChainType: Enum
local ECertInChainType = create_enum_type('CertInChainType')
ECertInChainType.default = ECertInChainType.new(2147483647)
ECertInChainType.struct = nil
ECertInChainType.Root = ECertInChainType.new(0)
ECertInChainType.Intermediate = ECertInChainType.new(1)
ECertInChainType.Last = ECertInChainType.new(2)

defs.CertInChainType = ECertInChainType

---@class defs.CertTaskStatus: Enum
local ECertTaskStatus = create_enum_type('CertTaskStatus')
ECertTaskStatus.default = ECertTaskStatus.new(2147483647)
ECertTaskStatus.struct = nil
ECertTaskStatus.NotStart = ECertTaskStatus.new(0)
ECertTaskStatus.Running = ECertTaskStatus.new(1)
ECertTaskStatus.Failed = ECertTaskStatus.new(2)
ECertTaskStatus.Success = ECertTaskStatus.new(3)

defs.CertTaskStatus = ECertTaskStatus

---@class defs.CertAlgorithm: Enum
local ECertAlgorithm = create_enum_type('CertAlgorithm')
ECertAlgorithm.default = ECertAlgorithm.new(2147483647)
ECertAlgorithm.struct = nil
ECertAlgorithm.RSA = ECertAlgorithm.new(0)
ECertAlgorithm.ECC = ECertAlgorithm.new(1)

defs.CertAlgorithm = ECertAlgorithm

---@class defs.CertificateUsageType: Enum
local ECertificateUsageType = create_enum_type('CertificateUsageType')
ECertificateUsageType.default = ECertificateUsageType.new(2147483647)
ECertificateUsageType.struct = nil
ECertificateUsageType.ManagerCACertificate = ECertificateUsageType.new(0)
ECertificateUsageType.ManagerSSLCertificate = ECertificateUsageType.new(1)
ECertificateUsageType.ManagerAccountCertificate = ECertificateUsageType.new(2)

defs.CertificateUsageType = ECertificateUsageType

---@class defs.KeyUsage: Enum
local EKeyUsage = create_enum_type('KeyUsage')
EKeyUsage.default = EKeyUsage.new(2147483647)
EKeyUsage.struct = nil
EKeyUsage.DigitalSignature = EKeyUsage.new(0)
EKeyUsage.NonRepudiation = EKeyUsage.new(1)
EKeyUsage.KeyEncipherment = EKeyUsage.new(2)
EKeyUsage.DataEncipherment = EKeyUsage.new(3)
EKeyUsage.KeyAgreement = EKeyUsage.new(4)
EKeyUsage.KeyCertSign = EKeyUsage.new(5)
EKeyUsage.CRLSigning = EKeyUsage.new(6)
EKeyUsage.EncipherOnly = EKeyUsage.new(7)
EKeyUsage.DecipherOnly = EKeyUsage.new(8)
EKeyUsage.ServerAuthentication = EKeyUsage.new(9)
EKeyUsage.ClientAuthentication = EKeyUsage.new(10)
EKeyUsage.CodeSigning = EKeyUsage.new(11)
EKeyUsage.EmailProtection = EKeyUsage.new(12)
EKeyUsage.Timestamping = EKeyUsage.new(13)
EKeyUsage.OCSPSigning = EKeyUsage.new(14)

defs.KeyUsage = EKeyUsage

---@class defs.CertificateType: Enum
local ECertificateType = create_enum_type('CertificateType')
ECertificateType.default = ECertificateType.new(2147483647)
ECertificateType.struct = nil
ECertificateType.PEM = ECertificateType.new(0)
ECertificateType.PEMchain = ECertificateType.new(1)
ECertificateType.PKCS7 = ECertificateType.new(2)

defs.CertificateType = ECertificateType

---@class defs.CertificateInfo
---@field CertificateType defs.CertificateType
---@field CertificateUsageType defs.CertificateUsageType
---@field Fingerprint string
---@field FingerprintHashAlgorithm string
---@field Issuer string
---@field KeyUsage defs.KeyUsage[]
---@field SerialNumber string
---@field SignatureAlgorithm string
---@field Subject string
---@field ValidNotAfter string
---@field ValidNotBefore string
---@field FilePath string
---@field CommonName string
---@field CRLStartTime string
---@field CRLExpireTime string
---@field CertCount integer
---@field KeyLength integer
---@field CertInChainType defs.CertInChainType
local TCertificateInfo = {}
TCertificateInfo.__index = TCertificateInfo
TCertificateInfo.group = {}

local function TCertificateInfo_from_obj(obj)
    obj.CertificateType = obj.CertificateType and defs.CertificateType.new(obj.CertificateType)
    obj.CertificateUsageType = obj.CertificateUsageType and
                                   defs.CertificateUsageType.new(obj.CertificateUsageType)
    obj.KeyUsage = utils.from_obj(defs.KeyUsage, obj.KeyUsage, true)
    obj.CertInChainType = obj.CertInChainType and defs.CertInChainType.new(obj.CertInChainType)
    return setmetatable(obj, TCertificateInfo)
end

function TCertificateInfo.new(CertificateType, CertificateUsageType, Fingerprint,
    FingerprintHashAlgorithm, Issuer, KeyUsage, SerialNumber, SignatureAlgorithm, Subject,
    ValidNotAfter, ValidNotBefore, FilePath, CommonName, CRLStartTime, CRLExpireTime, CertCount,
    KeyLength, CertInChainType)
    return TCertificateInfo_from_obj({
        CertificateType = CertificateType,
        CertificateUsageType = CertificateUsageType,
        Fingerprint = Fingerprint,
        FingerprintHashAlgorithm = FingerprintHashAlgorithm,
        Issuer = Issuer,
        KeyUsage = KeyUsage,
        SerialNumber = SerialNumber,
        SignatureAlgorithm = SignatureAlgorithm,
        Subject = Subject,
        ValidNotAfter = ValidNotAfter,
        ValidNotBefore = ValidNotBefore,
        FilePath = FilePath,
        CommonName = CommonName,
        CRLStartTime = CRLStartTime,
        CRLExpireTime = CRLExpireTime,
        CertCount = CertCount,
        KeyLength = KeyLength,
        CertInChainType = CertInChainType
    })
end
---@param obj defs.CertificateInfo
function TCertificateInfo:init_from_obj(obj)
    self.CertificateType = obj.CertificateType
    self.CertificateUsageType = obj.CertificateUsageType
    self.Fingerprint = obj.Fingerprint
    self.FingerprintHashAlgorithm = obj.FingerprintHashAlgorithm
    self.Issuer = obj.Issuer
    self.KeyUsage = obj.KeyUsage
    self.SerialNumber = obj.SerialNumber
    self.SignatureAlgorithm = obj.SignatureAlgorithm
    self.Subject = obj.Subject
    self.ValidNotAfter = obj.ValidNotAfter
    self.ValidNotBefore = obj.ValidNotBefore
    self.FilePath = obj.FilePath
    self.CommonName = obj.CommonName
    self.CRLStartTime = obj.CRLStartTime
    self.CRLExpireTime = obj.CRLExpireTime
    self.CertCount = obj.CertCount
    self.KeyLength = obj.KeyLength
    self.CertInChainType = obj.CertInChainType
end

function TCertificateInfo:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TCertificateInfo.group)
end

TCertificateInfo.from_obj = TCertificateInfo_from_obj

TCertificateInfo.proto_property = {
    'CertificateType', 'CertificateUsageType', 'Fingerprint', 'FingerprintHashAlgorithm', 'Issuer',
    'KeyUsage', 'SerialNumber', 'SignatureAlgorithm', 'Subject', 'ValidNotAfter', 'ValidNotBefore',
    'FilePath', 'CommonName', 'CRLStartTime', 'CRLExpireTime', 'CertCount', 'KeyLength',
    'CertInChainType'
}

TCertificateInfo.default = {
    defs.CertificateType.default, defs.CertificateUsageType.default, '', '', '', {}, '', '', '', '',
    '', '', '', '', '', 0, 0, defs.CertInChainType.default
}

TCertificateInfo.struct = {
    {name = 'CertificateType', is_array = false, struct = defs.CertificateType.struct},
    {name = 'CertificateUsageType', is_array = false, struct = defs.CertificateUsageType.struct},
    {name = 'Fingerprint', is_array = false, struct = nil},
    {name = 'FingerprintHashAlgorithm', is_array = false, struct = nil},
    {name = 'Issuer', is_array = false, struct = nil},
    {name = 'KeyUsage', is_array = true, struct = defs.KeyUsage.struct},
    {name = 'SerialNumber', is_array = false, struct = nil},
    {name = 'SignatureAlgorithm', is_array = false, struct = nil},
    {name = 'Subject', is_array = false, struct = nil},
    {name = 'ValidNotAfter', is_array = false, struct = nil},
    {name = 'ValidNotBefore', is_array = false, struct = nil},
    {name = 'FilePath', is_array = false, struct = nil},
    {name = 'CommonName', is_array = false, struct = nil},
    {name = 'CRLStartTime', is_array = false, struct = nil},
    {name = 'CRLExpireTime', is_array = false, struct = nil},
    {name = 'CertCount', is_array = false, struct = nil},
    {name = 'KeyLength', is_array = false, struct = nil},
    {name = 'CertInChainType', is_array = false, struct = defs.CertInChainType.struct}
}

function TCertificateInfo:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CertificateType', self.CertificateType, 'defs.CertificateType',
        false, errs)
    validate.Required(prefix .. 'CertificateUsageType', self.CertificateUsageType,
        'defs.CertificateUsageType', false, errs)
    validate.Required(prefix .. 'Fingerprint', self.Fingerprint, 'string', false, errs)
    validate.Required(prefix .. 'FingerprintHashAlgorithm', self.FingerprintHashAlgorithm, 'string',
        false, errs)
    validate.Required(prefix .. 'Issuer', self.Issuer, 'string', false, errs)
    validate.RequiredArray(prefix .. 'KeyUsage', self.KeyUsage, 'defs.KeyUsage', false, errs)
    validate.Required(prefix .. 'SerialNumber', self.SerialNumber, 'string', false, errs)
    validate.Required(prefix .. 'SignatureAlgorithm', self.SignatureAlgorithm, 'string', false, errs)
    validate.Required(prefix .. 'Subject', self.Subject, 'string', false, errs)
    validate.Required(prefix .. 'ValidNotAfter', self.ValidNotAfter, 'string', false, errs)
    validate.Required(prefix .. 'ValidNotBefore', self.ValidNotBefore, 'string', false, errs)
    validate.Required(prefix .. 'FilePath', self.FilePath, 'string', false, errs)
    validate.Required(prefix .. 'CommonName', self.CommonName, 'string', false, errs)
    validate.Required(prefix .. 'CRLStartTime', self.CRLStartTime, 'string', false, errs)
    validate.Required(prefix .. 'CRLExpireTime', self.CRLExpireTime, 'string', false, errs)
    validate.Required(prefix .. 'CertCount', self.CertCount, 'uint32', false, errs)
    validate.Required(prefix .. 'KeyLength', self.KeyLength, 'uint32', false, errs)
    validate.Required(prefix .. 'CertInChainType', self.CertInChainType, 'defs.CertInChainType',
        false, errs)

    TCertificateInfo:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TCertificateInfo.proto_property, errs)
    return self
end

function TCertificateInfo:unpack(raw)
    local CertificateType = utils.unpack_enum(raw, self.CertificateType)
    local CertificateUsageType = utils.unpack_enum(raw, self.CertificateUsageType)
    local KeyUsage =
        utils.unpack_enum(raw, utils.from_obj(defs.KeyUsage, self.KeyUsage, true), true)
    local CertInChainType = utils.unpack_enum(raw, self.CertInChainType)
    return CertificateType, CertificateUsageType, self.Fingerprint, self.FingerprintHashAlgorithm,
        self.Issuer, KeyUsage, self.SerialNumber, self.SignatureAlgorithm, self.Subject,
        self.ValidNotAfter, self.ValidNotBefore, self.FilePath, self.CommonName, self.CRLStartTime,
        self.CRLExpireTime, self.CertCount, self.KeyLength, CertInChainType
end

defs.CertificateInfo = TCertificateInfo

---@class defs.description
local Tdescription = {}
Tdescription.__index = Tdescription
Tdescription.group = {}

local function Tdescription_from_obj(obj)
    return setmetatable(obj, Tdescription)
end

function Tdescription.new()
    return Tdescription_from_obj({})
end
---@param obj defs.description
function Tdescription:init_from_obj(obj)

end

function Tdescription:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, Tdescription.group)
end

Tdescription.from_obj = Tdescription_from_obj

Tdescription.proto_property = {}

Tdescription.default = {}

Tdescription.struct = {}

function Tdescription:validate(prefix, errs)
    prefix = prefix or ''

    Tdescription:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, Tdescription.proto_property, errs)
    return self
end

function Tdescription:unpack(_)
end

defs.description = Tdescription

---@class CertificateService.GetCertChainInfoRsp
---@field CertInfo defs.CertificateInfo[]
local TGetCertChainInfoRsp = {}
TGetCertChainInfoRsp.__index = TGetCertChainInfoRsp
TGetCertChainInfoRsp.group = {}

local function TGetCertChainInfoRsp_from_obj(obj)
    obj.CertInfo = utils.from_obj(defs.CertificateInfo, obj.CertInfo, true)
    return setmetatable(obj, TGetCertChainInfoRsp)
end

function TGetCertChainInfoRsp.new(CertInfo)
    return TGetCertChainInfoRsp_from_obj({CertInfo = CertInfo})
end
---@param obj CertificateService.GetCertChainInfoRsp
function TGetCertChainInfoRsp:init_from_obj(obj)
    self.CertInfo = obj.CertInfo
end

function TGetCertChainInfoRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetCertChainInfoRsp.group)
end

TGetCertChainInfoRsp.from_obj = TGetCertChainInfoRsp_from_obj

TGetCertChainInfoRsp.proto_property = {'CertInfo'}

TGetCertChainInfoRsp.default = {{}}

TGetCertChainInfoRsp.struct = {
    {name = 'CertInfo', is_array = true, struct = defs.CertificateInfo.struct}
}

function TGetCertChainInfoRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.RequiredArray(prefix .. 'CertInfo', self.CertInfo, 'defs.CertificateInfo', false, errs)

    TGetCertChainInfoRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetCertChainInfoRsp.proto_property, errs)
    return self
end

function TGetCertChainInfoRsp:unpack(raw)
    return utils.unpack(raw, self.CertInfo, true)
end

msg.GetCertChainInfoRsp = TGetCertChainInfoRsp

---@class CertificateService.GetCertChainInfoReq
---@field CertificateUsageType defs.CertificateUsageType
---@field Id integer
local TGetCertChainInfoReq = {}
TGetCertChainInfoReq.__index = TGetCertChainInfoReq
TGetCertChainInfoReq.group = {}

local function TGetCertChainInfoReq_from_obj(obj)
    obj.CertificateUsageType = obj.CertificateUsageType and
                                   defs.CertificateUsageType.new(obj.CertificateUsageType)
    return setmetatable(obj, TGetCertChainInfoReq)
end

function TGetCertChainInfoReq.new(CertificateUsageType, Id)
    return TGetCertChainInfoReq_from_obj({CertificateUsageType = CertificateUsageType, Id = Id})
end
---@param obj CertificateService.GetCertChainInfoReq
function TGetCertChainInfoReq:init_from_obj(obj)
    self.CertificateUsageType = obj.CertificateUsageType
    self.Id = obj.Id
end

function TGetCertChainInfoReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetCertChainInfoReq.group)
end

TGetCertChainInfoReq.from_obj = TGetCertChainInfoReq_from_obj

TGetCertChainInfoReq.proto_property = {'CertificateUsageType', 'Id'}

TGetCertChainInfoReq.default = {defs.CertificateUsageType.default, 0}

TGetCertChainInfoReq.struct = {
    {name = 'CertificateUsageType', is_array = false, struct = defs.CertificateUsageType.struct},
    {name = 'Id', is_array = false, struct = nil}
}

function TGetCertChainInfoReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CertificateUsageType', self.CertificateUsageType,
        'defs.CertificateUsageType', false, errs)
    validate.Required(prefix .. 'Id', self.Id, 'uint32', false, errs)

    TGetCertChainInfoReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetCertChainInfoReq.proto_property, errs)
    return self
end

function TGetCertChainInfoReq:unpack(raw)
    local CertificateUsageType = utils.unpack_enum(raw, self.CertificateUsageType)
    return CertificateUsageType, self.Id
end

msg.GetCertChainInfoReq = TGetCertChainInfoReq

---@class CertificateService.DeleteCertRsp
local TDeleteCertRsp = {}
TDeleteCertRsp.__index = TDeleteCertRsp
TDeleteCertRsp.group = {}

local function TDeleteCertRsp_from_obj(obj)
    return setmetatable(obj, TDeleteCertRsp)
end

function TDeleteCertRsp.new()
    return TDeleteCertRsp_from_obj({})
end
---@param obj CertificateService.DeleteCertRsp
function TDeleteCertRsp:init_from_obj(obj)

end

function TDeleteCertRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TDeleteCertRsp.group)
end

TDeleteCertRsp.from_obj = TDeleteCertRsp_from_obj

TDeleteCertRsp.proto_property = {}

TDeleteCertRsp.default = {}

TDeleteCertRsp.struct = {}

function TDeleteCertRsp:validate(prefix, errs)
    prefix = prefix or ''

    TDeleteCertRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TDeleteCertRsp.proto_property, errs)
    return self
end

function TDeleteCertRsp:unpack(_)
end

msg.DeleteCertRsp = TDeleteCertRsp

---@class CertificateService.DeleteCertReq
---@field CertificateUsageType defs.CertificateUsageType
---@field Id integer
local TDeleteCertReq = {}
TDeleteCertReq.__index = TDeleteCertReq
TDeleteCertReq.group = {}

local function TDeleteCertReq_from_obj(obj)
    obj.CertificateUsageType = obj.CertificateUsageType and
                                   defs.CertificateUsageType.new(obj.CertificateUsageType)
    return setmetatable(obj, TDeleteCertReq)
end

function TDeleteCertReq.new(CertificateUsageType, Id)
    return TDeleteCertReq_from_obj({CertificateUsageType = CertificateUsageType, Id = Id})
end
---@param obj CertificateService.DeleteCertReq
function TDeleteCertReq:init_from_obj(obj)
    self.CertificateUsageType = obj.CertificateUsageType
    self.Id = obj.Id
end

function TDeleteCertReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TDeleteCertReq.group)
end

TDeleteCertReq.from_obj = TDeleteCertReq_from_obj

TDeleteCertReq.proto_property = {'CertificateUsageType', 'Id'}

TDeleteCertReq.default = {defs.CertificateUsageType.default, 0}

TDeleteCertReq.struct = {
    {name = 'CertificateUsageType', is_array = false, struct = defs.CertificateUsageType.struct},
    {name = 'Id', is_array = false, struct = nil}
}

function TDeleteCertReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CertificateUsageType', self.CertificateUsageType,
        'defs.CertificateUsageType', false, errs)
    validate.Required(prefix .. 'Id', self.Id, 'uint32', false, errs)

    TDeleteCertReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TDeleteCertReq.proto_property, errs)
    return self
end

function TDeleteCertReq:unpack(raw)
    local CertificateUsageType = utils.unpack_enum(raw, self.CertificateUsageType)
    return CertificateUsageType, self.Id
end

msg.DeleteCertReq = TDeleteCertReq

---@class CertificateService.ExportCertKeyByFIFORsp
---@field FilePath string
local TExportCertKeyByFIFORsp = {}
TExportCertKeyByFIFORsp.__index = TExportCertKeyByFIFORsp
TExportCertKeyByFIFORsp.group = {}

local function TExportCertKeyByFIFORsp_from_obj(obj)
    return setmetatable(obj, TExportCertKeyByFIFORsp)
end

function TExportCertKeyByFIFORsp.new(FilePath)
    return TExportCertKeyByFIFORsp_from_obj({FilePath = FilePath})
end
---@param obj CertificateService.ExportCertKeyByFIFORsp
function TExportCertKeyByFIFORsp:init_from_obj(obj)
    self.FilePath = obj.FilePath
end

function TExportCertKeyByFIFORsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TExportCertKeyByFIFORsp.group)
end

TExportCertKeyByFIFORsp.from_obj = TExportCertKeyByFIFORsp_from_obj

TExportCertKeyByFIFORsp.proto_property = {'FilePath'}

TExportCertKeyByFIFORsp.default = {''}

TExportCertKeyByFIFORsp.struct = {{name = 'FilePath', is_array = false, struct = nil}}

function TExportCertKeyByFIFORsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'FilePath', self.FilePath, 'string', false, errs)

    validate.lens(prefix .. 'FilePath', self.FilePath, 1, 2048, errs)

    TExportCertKeyByFIFORsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TExportCertKeyByFIFORsp.proto_property, errs)
    return self
end

function TExportCertKeyByFIFORsp:unpack(_)
    return self.FilePath
end

msg.ExportCertKeyByFIFORsp = TExportCertKeyByFIFORsp

---@class CertificateService.ExportCertKeyByFIFOReq
---@field CertificateUsageType defs.CertificateUsageType
local TExportCertKeyByFIFOReq = {}
TExportCertKeyByFIFOReq.__index = TExportCertKeyByFIFOReq
TExportCertKeyByFIFOReq.group = {}

local function TExportCertKeyByFIFOReq_from_obj(obj)
    obj.CertificateUsageType = obj.CertificateUsageType and
                                   defs.CertificateUsageType.new(obj.CertificateUsageType)
    return setmetatable(obj, TExportCertKeyByFIFOReq)
end

function TExportCertKeyByFIFOReq.new(CertificateUsageType)
    return TExportCertKeyByFIFOReq_from_obj({CertificateUsageType = CertificateUsageType})
end
---@param obj CertificateService.ExportCertKeyByFIFOReq
function TExportCertKeyByFIFOReq:init_from_obj(obj)
    self.CertificateUsageType = obj.CertificateUsageType
end

function TExportCertKeyByFIFOReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TExportCertKeyByFIFOReq.group)
end

TExportCertKeyByFIFOReq.from_obj = TExportCertKeyByFIFOReq_from_obj

TExportCertKeyByFIFOReq.proto_property = {'CertificateUsageType'}

TExportCertKeyByFIFOReq.default = {defs.CertificateUsageType.default}

TExportCertKeyByFIFOReq.struct = {
    {name = 'CertificateUsageType', is_array = false, struct = defs.CertificateUsageType.struct}
}

function TExportCertKeyByFIFOReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CertificateUsageType', self.CertificateUsageType,
        'defs.CertificateUsageType', false, errs)

    TExportCertKeyByFIFOReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TExportCertKeyByFIFOReq.proto_property, errs)
    return self
end

function TExportCertKeyByFIFOReq:unpack(raw)
    local CertificateUsageType = utils.unpack_enum(raw, self.CertificateUsageType)
    return CertificateUsageType
end

msg.ExportCertKeyByFIFOReq = TExportCertKeyByFIFOReq

---@class CertificateService.ImportCertWithSignedCSRRsp
---@field Id integer
local TImportCertWithSignedCSRRsp = {}
TImportCertWithSignedCSRRsp.__index = TImportCertWithSignedCSRRsp
TImportCertWithSignedCSRRsp.group = {}

local function TImportCertWithSignedCSRRsp_from_obj(obj)
    return setmetatable(obj, TImportCertWithSignedCSRRsp)
end

function TImportCertWithSignedCSRRsp.new(Id)
    return TImportCertWithSignedCSRRsp_from_obj({Id = Id})
end
---@param obj CertificateService.ImportCertWithSignedCSRRsp
function TImportCertWithSignedCSRRsp:init_from_obj(obj)
    self.Id = obj.Id
end

function TImportCertWithSignedCSRRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TImportCertWithSignedCSRRsp.group)
end

TImportCertWithSignedCSRRsp.from_obj = TImportCertWithSignedCSRRsp_from_obj

TImportCertWithSignedCSRRsp.proto_property = {'Id'}

TImportCertWithSignedCSRRsp.default = {0}

TImportCertWithSignedCSRRsp.struct = {{name = 'Id', is_array = false, struct = nil}}

function TImportCertWithSignedCSRRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Id', self.Id, 'uint32', false, errs)

    TImportCertWithSignedCSRRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TImportCertWithSignedCSRRsp.proto_property, errs)
    return self
end

function TImportCertWithSignedCSRRsp:unpack(_)
    return self.Id
end

msg.ImportCertWithSignedCSRRsp = TImportCertWithSignedCSRRsp

---@class CertificateService.ImportCertWithSignedCSRReq
---@field CertificateUsageType defs.CertificateUsageType
---@field FilePath string
local TImportCertWithSignedCSRReq = {}
TImportCertWithSignedCSRReq.__index = TImportCertWithSignedCSRReq
TImportCertWithSignedCSRReq.group = {}

local function TImportCertWithSignedCSRReq_from_obj(obj)
    obj.CertificateUsageType = obj.CertificateUsageType and
                                   defs.CertificateUsageType.new(obj.CertificateUsageType)
    return setmetatable(obj, TImportCertWithSignedCSRReq)
end

function TImportCertWithSignedCSRReq.new(CertificateUsageType, FilePath)
    return TImportCertWithSignedCSRReq_from_obj({
        CertificateUsageType = CertificateUsageType,
        FilePath = FilePath
    })
end
---@param obj CertificateService.ImportCertWithSignedCSRReq
function TImportCertWithSignedCSRReq:init_from_obj(obj)
    self.CertificateUsageType = obj.CertificateUsageType
    self.FilePath = obj.FilePath
end

function TImportCertWithSignedCSRReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TImportCertWithSignedCSRReq.group)
end

TImportCertWithSignedCSRReq.from_obj = TImportCertWithSignedCSRReq_from_obj

TImportCertWithSignedCSRReq.proto_property = {'CertificateUsageType', 'FilePath'}

TImportCertWithSignedCSRReq.default = {defs.CertificateUsageType.default, ''}

TImportCertWithSignedCSRReq.struct = {
    {name = 'CertificateUsageType', is_array = false, struct = defs.CertificateUsageType.struct},
    {name = 'FilePath', is_array = false, struct = nil}
}

function TImportCertWithSignedCSRReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CertificateUsageType', self.CertificateUsageType,
        'defs.CertificateUsageType', false, errs)
    validate.Required(prefix .. 'FilePath', self.FilePath, 'string', false, errs)

    validate.lens(prefix .. 'FilePath', self.FilePath, 1, 2048, errs)

    TImportCertWithSignedCSRReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TImportCertWithSignedCSRReq.proto_property, errs)
    return self
end

function TImportCertWithSignedCSRReq:unpack(raw)
    local CertificateUsageType = utils.unpack_enum(raw, self.CertificateUsageType)
    return CertificateUsageType, self.FilePath
end

msg.ImportCertWithSignedCSRReq = TImportCertWithSignedCSRReq

---@class CertificateService.CheckGenerateCSRRsp
---@field Status defs.CertTaskStatus
---@field FilePath string
local TCheckGenerateCSRRsp = {}
TCheckGenerateCSRRsp.__index = TCheckGenerateCSRRsp
TCheckGenerateCSRRsp.group = {}

local function TCheckGenerateCSRRsp_from_obj(obj)
    obj.Status = obj.Status and defs.CertTaskStatus.new(obj.Status)
    return setmetatable(obj, TCheckGenerateCSRRsp)
end

function TCheckGenerateCSRRsp.new(Status, FilePath)
    return TCheckGenerateCSRRsp_from_obj({Status = Status, FilePath = FilePath})
end
---@param obj CertificateService.CheckGenerateCSRRsp
function TCheckGenerateCSRRsp:init_from_obj(obj)
    self.Status = obj.Status
    self.FilePath = obj.FilePath
end

function TCheckGenerateCSRRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TCheckGenerateCSRRsp.group)
end

TCheckGenerateCSRRsp.from_obj = TCheckGenerateCSRRsp_from_obj

TCheckGenerateCSRRsp.proto_property = {'Status', 'FilePath'}

TCheckGenerateCSRRsp.default = {defs.CertTaskStatus.default, ''}

TCheckGenerateCSRRsp.struct = {
    {name = 'Status', is_array = false, struct = defs.CertTaskStatus.struct},
    {name = 'FilePath', is_array = false, struct = nil}
}

function TCheckGenerateCSRRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Status', self.Status, 'defs.CertTaskStatus', false, errs)
    validate.Required(prefix .. 'FilePath', self.FilePath, 'string', false, errs)

    TCheckGenerateCSRRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TCheckGenerateCSRRsp.proto_property, errs)
    return self
end

function TCheckGenerateCSRRsp:unpack(raw)
    local Status = utils.unpack_enum(raw, self.Status)
    return Status, self.FilePath
end

msg.CheckGenerateCSRRsp = TCheckGenerateCSRRsp

---@class CertificateService.CheckGenerateCSRReq
local TCheckGenerateCSRReq = {}
TCheckGenerateCSRReq.__index = TCheckGenerateCSRReq
TCheckGenerateCSRReq.group = {}

local function TCheckGenerateCSRReq_from_obj(obj)
    return setmetatable(obj, TCheckGenerateCSRReq)
end

function TCheckGenerateCSRReq.new()
    return TCheckGenerateCSRReq_from_obj({})
end
---@param obj CertificateService.CheckGenerateCSRReq
function TCheckGenerateCSRReq:init_from_obj(obj)

end

function TCheckGenerateCSRReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TCheckGenerateCSRReq.group)
end

TCheckGenerateCSRReq.from_obj = TCheckGenerateCSRReq_from_obj

TCheckGenerateCSRReq.proto_property = {}

TCheckGenerateCSRReq.default = {}

TCheckGenerateCSRReq.struct = {}

function TCheckGenerateCSRReq:validate(prefix, errs)
    prefix = prefix or ''

    TCheckGenerateCSRReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TCheckGenerateCSRReq.proto_property, errs)
    return self
end

function TCheckGenerateCSRReq:unpack(_)
end

msg.CheckGenerateCSRReq = TCheckGenerateCSRReq

---@class CertificateService.StartGenerateCSRRsp
local TStartGenerateCSRRsp = {}
TStartGenerateCSRRsp.__index = TStartGenerateCSRRsp
TStartGenerateCSRRsp.group = {}

local function TStartGenerateCSRRsp_from_obj(obj)
    return setmetatable(obj, TStartGenerateCSRRsp)
end

function TStartGenerateCSRRsp.new()
    return TStartGenerateCSRRsp_from_obj({})
end
---@param obj CertificateService.StartGenerateCSRRsp
function TStartGenerateCSRRsp:init_from_obj(obj)

end

function TStartGenerateCSRRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TStartGenerateCSRRsp.group)
end

TStartGenerateCSRRsp.from_obj = TStartGenerateCSRRsp_from_obj

TStartGenerateCSRRsp.proto_property = {}

TStartGenerateCSRRsp.default = {}

TStartGenerateCSRRsp.struct = {}

function TStartGenerateCSRRsp:validate(prefix, errs)
    prefix = prefix or ''

    TStartGenerateCSRRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TStartGenerateCSRRsp.proto_property, errs)
    return self
end

function TStartGenerateCSRRsp:unpack(_)
end

msg.StartGenerateCSRRsp = TStartGenerateCSRRsp

---@class CertificateService.StartGenerateCSRReq
---@field Country string
---@field State string
---@field Location string
---@field OrgName string
---@field OrgUnit string
---@field CommonName string
local TStartGenerateCSRReq = {}
TStartGenerateCSRReq.__index = TStartGenerateCSRReq
TStartGenerateCSRReq.group = {}

local function TStartGenerateCSRReq_from_obj(obj)
    return setmetatable(obj, TStartGenerateCSRReq)
end

function TStartGenerateCSRReq.new(Country, State, Location, OrgName, OrgUnit, CommonName)
    return TStartGenerateCSRReq_from_obj({
        Country = Country,
        State = State,
        Location = Location,
        OrgName = OrgName,
        OrgUnit = OrgUnit,
        CommonName = CommonName
    })
end
---@param obj CertificateService.StartGenerateCSRReq
function TStartGenerateCSRReq:init_from_obj(obj)
    self.Country = obj.Country
    self.State = obj.State
    self.Location = obj.Location
    self.OrgName = obj.OrgName
    self.OrgUnit = obj.OrgUnit
    self.CommonName = obj.CommonName
end

function TStartGenerateCSRReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TStartGenerateCSRReq.group)
end

TStartGenerateCSRReq.from_obj = TStartGenerateCSRReq_from_obj

TStartGenerateCSRReq.proto_property = {
    'Country', 'State', 'Location', 'OrgName', 'OrgUnit', 'CommonName'
}

TStartGenerateCSRReq.default = {'', '', '', '', '', ''}

TStartGenerateCSRReq.struct = {
    {name = 'Country', is_array = false, struct = nil},
    {name = 'State', is_array = false, struct = nil},
    {name = 'Location', is_array = false, struct = nil},
    {name = 'OrgName', is_array = false, struct = nil},
    {name = 'OrgUnit', is_array = false, struct = nil},
    {name = 'CommonName', is_array = false, struct = nil}
}

function TStartGenerateCSRReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Country', self.Country, 'string', false, errs)
    validate.Required(prefix .. 'State', self.State, 'string', false, errs)
    validate.Required(prefix .. 'Location', self.Location, 'string', false, errs)
    validate.Required(prefix .. 'OrgName', self.OrgName, 'string', false, errs)
    validate.Required(prefix .. 'OrgUnit', self.OrgUnit, 'string', false, errs)
    validate.Required(prefix .. 'CommonName', self.CommonName, 'string', false, errs)

    validate.lens(prefix .. 'Country', self.Country, 1, 2, errs)
    validate.lens(prefix .. 'State', self.State, 0, 128, errs)
    validate.lens(prefix .. 'Location', self.Location, 0, 128, errs)
    validate.lens(prefix .. 'OrgName', self.OrgName, 0, 64, errs)
    validate.lens(prefix .. 'OrgUnit', self.OrgUnit, 0, 64, errs)
    validate.lens(prefix .. 'CommonName', self.CommonName, 1, 64, errs)

    TStartGenerateCSRReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TStartGenerateCSRReq.proto_property, errs)
    return self
end

function TStartGenerateCSRReq:unpack(_)
    return self.Country, self.State, self.Location, self.OrgName, self.OrgUnit, self.CommonName
end

msg.StartGenerateCSRReq = TStartGenerateCSRReq

---@class CertificateService.ImportCertWithKeyRsp
---@field Id integer
local TImportCertWithKeyRsp = {}
TImportCertWithKeyRsp.__index = TImportCertWithKeyRsp
TImportCertWithKeyRsp.group = {}

local function TImportCertWithKeyRsp_from_obj(obj)
    return setmetatable(obj, TImportCertWithKeyRsp)
end

function TImportCertWithKeyRsp.new(Id)
    return TImportCertWithKeyRsp_from_obj({Id = Id})
end
---@param obj CertificateService.ImportCertWithKeyRsp
function TImportCertWithKeyRsp:init_from_obj(obj)
    self.Id = obj.Id
end

function TImportCertWithKeyRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TImportCertWithKeyRsp.group)
end

TImportCertWithKeyRsp.from_obj = TImportCertWithKeyRsp_from_obj

TImportCertWithKeyRsp.proto_property = {'Id'}

TImportCertWithKeyRsp.default = {0}

TImportCertWithKeyRsp.struct = {{name = 'Id', is_array = false, struct = nil}}

function TImportCertWithKeyRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Id', self.Id, 'uint32', false, errs)

    TImportCertWithKeyRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TImportCertWithKeyRsp.proto_property, errs)
    return self
end

function TImportCertWithKeyRsp:unpack(_)
    return self.Id
end

msg.ImportCertWithKeyRsp = TImportCertWithKeyRsp

---@class CertificateService.ImportCertWithKeyReq
---@field CertificateUsageType defs.CertificateUsageType
---@field FilePath string
---@field Key string
local TImportCertWithKeyReq = {}
TImportCertWithKeyReq.__index = TImportCertWithKeyReq
TImportCertWithKeyReq.group = {}

local function TImportCertWithKeyReq_from_obj(obj)
    obj.CertificateUsageType = obj.CertificateUsageType and
                                   defs.CertificateUsageType.new(obj.CertificateUsageType)
    return setmetatable(obj, TImportCertWithKeyReq)
end

function TImportCertWithKeyReq.new(CertificateUsageType, FilePath, Key)
    return TImportCertWithKeyReq_from_obj({
        CertificateUsageType = CertificateUsageType,
        FilePath = FilePath,
        Key = Key
    })
end
---@param obj CertificateService.ImportCertWithKeyReq
function TImportCertWithKeyReq:init_from_obj(obj)
    self.CertificateUsageType = obj.CertificateUsageType
    self.FilePath = obj.FilePath
    self.Key = obj.Key
end

function TImportCertWithKeyReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TImportCertWithKeyReq.group)
end

TImportCertWithKeyReq.from_obj = TImportCertWithKeyReq_from_obj

TImportCertWithKeyReq.proto_property = {'CertificateUsageType', 'FilePath', 'Key'}

TImportCertWithKeyReq.default = {defs.CertificateUsageType.default, '', ''}

TImportCertWithKeyReq.struct = {
    {name = 'CertificateUsageType', is_array = false, struct = defs.CertificateUsageType.struct},
    {name = 'FilePath', is_array = false, struct = nil},
    {name = 'Key', is_array = false, struct = nil}
}

function TImportCertWithKeyReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CertificateUsageType', self.CertificateUsageType,
        'defs.CertificateUsageType', false, errs)
    validate.Required(prefix .. 'FilePath', self.FilePath, 'string', false, errs)
    validate.Required(prefix .. 'Key', self.Key, 'string', false, errs)

    validate.lens(prefix .. 'FilePath', self.FilePath, 1, 2048, errs)
    validate.lens(prefix .. 'Key', self.Key, 0, 32, errs)

    TImportCertWithKeyReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TImportCertWithKeyReq.proto_property, errs)
    return self
end

function TImportCertWithKeyReq:unpack(raw)
    local CertificateUsageType = utils.unpack_enum(raw, self.CertificateUsageType)
    return CertificateUsageType, self.FilePath, self.Key
end

msg.ImportCertWithKeyReq = TImportCertWithKeyReq

---@class CertificateService.ImportCertRsp
---@field Id integer
local TImportCertRsp = {}
TImportCertRsp.__index = TImportCertRsp
TImportCertRsp.group = {}

local function TImportCertRsp_from_obj(obj)
    return setmetatable(obj, TImportCertRsp)
end

function TImportCertRsp.new(Id)
    return TImportCertRsp_from_obj({Id = Id})
end
---@param obj CertificateService.ImportCertRsp
function TImportCertRsp:init_from_obj(obj)
    self.Id = obj.Id
end

function TImportCertRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TImportCertRsp.group)
end

TImportCertRsp.from_obj = TImportCertRsp_from_obj

TImportCertRsp.proto_property = {'Id'}

TImportCertRsp.default = {0}

TImportCertRsp.struct = {{name = 'Id', is_array = false, struct = nil}}

function TImportCertRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Id', self.Id, 'uint32', false, errs)

    TImportCertRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TImportCertRsp.proto_property, errs)
    return self
end

function TImportCertRsp:unpack(_)
    return self.Id
end

msg.ImportCertRsp = TImportCertRsp

---@class CertificateService.ImportCertReq
---@field CertificateUsageType defs.CertificateUsageType
---@field FilePath string
local TImportCertReq = {}
TImportCertReq.__index = TImportCertReq
TImportCertReq.group = {}

local function TImportCertReq_from_obj(obj)
    obj.CertificateUsageType = obj.CertificateUsageType and
                                   defs.CertificateUsageType.new(obj.CertificateUsageType)
    return setmetatable(obj, TImportCertReq)
end

function TImportCertReq.new(CertificateUsageType, FilePath)
    return TImportCertReq_from_obj({
        CertificateUsageType = CertificateUsageType,
        FilePath = FilePath
    })
end
---@param obj CertificateService.ImportCertReq
function TImportCertReq:init_from_obj(obj)
    self.CertificateUsageType = obj.CertificateUsageType
    self.FilePath = obj.FilePath
end

function TImportCertReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TImportCertReq.group)
end

TImportCertReq.from_obj = TImportCertReq_from_obj

TImportCertReq.proto_property = {'CertificateUsageType', 'FilePath'}

TImportCertReq.default = {defs.CertificateUsageType.default, ''}

TImportCertReq.struct = {
    {name = 'CertificateUsageType', is_array = false, struct = defs.CertificateUsageType.struct},
    {name = 'FilePath', is_array = false, struct = nil}
}

function TImportCertReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CertificateUsageType', self.CertificateUsageType,
        'defs.CertificateUsageType', false, errs)
    validate.Required(prefix .. 'FilePath', self.FilePath, 'string', false, errs)

    validate.lens(prefix .. 'FilePath', self.FilePath, 1, 2048, errs)

    TImportCertReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TImportCertReq.proto_property, errs)
    return self
end

function TImportCertReq:unpack(raw)
    local CertificateUsageType = utils.unpack_enum(raw, self.CertificateUsageType)
    return CertificateUsageType, self.FilePath
end

msg.ImportCertReq = TImportCertReq

local CertificateService = {}

local Mdb = require 'mc.mdb'

---@class CertificateService: Table
---@field CertOverdueWarnDays integer
CertificateService.CertificateService = Mdb.register_interface('bmc.kepler.CertificateService', {
    CertOverdueWarnDays = {'u', nil, false, 90}
}, {
    ImportCert = {'a{ss}is', 'u', msg.ImportCertReq, msg.ImportCertRsp},
    ImportCertWithKey = {'a{ss}iss', 'u', msg.ImportCertWithKeyReq, msg.ImportCertWithKeyRsp},
    StartGenerateCSR = {'a{ss}ssssss', '', msg.StartGenerateCSRReq, msg.StartGenerateCSRRsp},
    CheckGenerateCSR = {'a{ss}', 'is', msg.CheckGenerateCSRReq, msg.CheckGenerateCSRRsp},
    ImportCertWithSignedCSR = {
        'a{ss}is', 'u', msg.ImportCertWithSignedCSRReq, msg.ImportCertWithSignedCSRRsp
    },
    ExportCertKeyByFIFO = {'a{ss}i', 's', msg.ExportCertKeyByFIFOReq, msg.ExportCertKeyByFIFORsp},
    DeleteCert = {'a{ss}iu', '', msg.DeleteCertReq, msg.DeleteCertRsp},
    GetCertChainInfo = {
        'a{ss}iu', 'a(iisssaisssssssssuui)', msg.GetCertChainInfoReq, msg.GetCertChainInfoRsp
    }
}, {})
return CertificateService
