-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local skynet = require 'skynet'
require 'skynet.manager'
local utils = require 'mc.utils'
local sd_bus = require 'sd_bus'
local log = require 'mc.logging'
local client = require 'syssdm.client'
local m_enums = require 'syssdm.types.enums'
local test_common = require 'test_common.utils'

local function prepare_test_data()
    log:info('== prepare test data')
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')
    os.execute('mkdir -p ' .. test_data_dir)
end

local function clear_test_data(exit_test)
    log:info('== clear test data')
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')
    if not exit_test then
        return utils.remove_file(test_data_dir)
    end

    skynet.timeout(0, function()
        skynet.sleep(20)
        skynet.abort()
        utils.remove_file(test_data_dir)
    end)
end

local function rpc_test_power_ctrl(bus)
    local err, rsp = client.pcall:PowerCtrl(bus, 1, 0, m_enums.PowerCtrlType.On,
        m_enums.RestartCause.ChassisControlCommand)
    if err then
        log:error('rpc call PowerCtrl has error')
    end

    assert(rsp.CmdResult == 0, 'rpc test PowerCtrl with On failed')
    log:info('rpc test PowerCtrl with On success')
end

local function rpc_test_upgrade_prepare(bus)
    local err, rsp = client.pcall:UpgradePrepare(bus, 1, 'BMC')
    if err then
        log:error('rpc call UpgradePrepre has error %s', err)
    end

    assert(rsp.CmdResult == 0, 'rpc test UpgradePrepre failed')
    log:info('rpc test UpgradePrepre success')
end

local function rpc_test_upgrade_prepare_reply(bus)
    local err, rsp = client.pcall:UpgradePrepareReply(bus, 1, 'BMC', 0)
    if err then
        log:error('rpc call UpgradePrepreReply has error %s', err)
    end

    assert(rsp.CmdResult == 0, 'rpc test UpgradePrepreReply failed')
    log:info('rpc test UpgradePrepreReply success')
end

local function rpc_test_upgrade_process(bus)
    local err, rsp = client.pcall:UpgradeProcess(bus, 1, 'BMC', '/tmp/123.hpm')
    if err then
        log:error('rpc call UpgradeProcess has error %s', err)
    end

    assert(rsp.CmdResult == 0, 'rpc test UpgradeProcess failed')
    log:info('rcp test UpgradeProcess success')
end

local function rpc_test_upgrade_process_reply(bus)
    local err, rsp = client.pcall:UpgradeProcessReply(bus, 1, 'BMC', 0)
    if err then
        log:error('rpc call UpgradeProcessReply has error %s', err)
    end

    assert(rsp.CmdResult == 0, 'rpc test UpgradeProcessReply failed')
    log:info('rpc test UpgradeProcessReply success')
end

local function rpc_test_upgrade_status_reply(bus)
    local err, rsp = client.pcall:UpgradeStatusReply(bus, 1, 'BMC', 0, 10, 'normal')
    if err then
        log:error('rpc call UpgradeStatusReply has error %s', err)
    end

    assert(rsp.CmdResult == 0, 'rpc test UpgradeStatusReply failed')
    log:info('rpc test UpgradeStatusReply success')
end

local function rpc_test_upgrade_finish(bus)
    local err, rsp = client.pcall:UpgradeFinish(bus, 1, 'BMC')
    if err then
        log:error('rpc call UpgradeFinish has error %s', err)
    end

    assert(rsp.CmdResult == 0, 'rpc test UpgradeFinish failed')
    log:info('rcp test UpgradeFinish success')
end

local function rpc_test_upgrade_finish_reply(bus)
    local err, rsp = client.pcall:UpgradeFinishReply(bus, 1, 'BMC', 0)
    if err then
        log:error('rpc call UpgradeFinishReply has error %s', err)
    end

    assert(rsp.CmdResult == 0, 'rpc test UpgradeFinishReply failed')
    log:info('rpc test UpgradeFinishReply success')
end

local function test_syssdm()
    log:info('================ test start ================')
    local bus = sd_bus.open_user(true)

    rpc_test_power_ctrl(bus)

    rpc_test_upgrade_prepare(bus)
    rpc_test_upgrade_prepare_reply(bus)

    rpc_test_upgrade_process(bus)
    rpc_test_upgrade_process_reply(bus)

    rpc_test_upgrade_status_reply(bus)

    rpc_test_upgrade_finish(bus)
    rpc_test_upgrade_finish_reply(bus)

    skynet.call('syssdm', 'lua', 'exit')
    log:info('================ test complete ================')
    skynet.sleep(20)
    skynet.abort()
end

skynet.start(function()
    prepare_test_data()
    test_common.dbus_launch()
    skynet.uniqueservice('sd_bus')
    skynet.uniqueservice('persistence/service/main')
    skynet.uniqueservice('main')
    skynet.fork(function()
        local ok, err = pcall(test_syssdm)
        clear_test_data(true)
        if not ok then
            error(err)
        end
    end)
end)
