-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


-- Description: 公共接口
local class_mgnt = require 'mc.class_mgnt'
local common = require 'common'
local ipmi = require 'ipmi'
local comp_code = ipmi.types.Cc
local function_table = {}

function function_table.get_fru_obj_by_fruid(fru_id)
    local objs = class_mgnt('Fru'):get_all()
    for _, obj in pairs(objs) do
        if obj['FruId'] == fru_id then
            return obj
        end
    end
    return nil
end

function function_table.get_component_obj_by_fruid(fru_id)
    local objs = class_mgnt('Component'):get_all()
    for _, obj in pairs(objs) do
        if obj['FruId'] == fru_id then
            return obj
        end
    end
    return nil
end

local function get_board_present(fru_obj)
    if not fru_obj then
        return comp_code.Success, string.char(common.INFO_FRU_UNPRESENT)
    end
    if fru_obj['Type'] == common.COMPONENT_TYPE_PCIE_CARD then
        return comp_code.Success, string.char(common.INFO_FRU_UNPRESENT)
    end
    return comp_code.Success, string.char(common.INFO_FRU_PRESENT)
end


local function get_board_product_id(fru_obj)
    if not fru_obj then
        return comp_code.ParmOutOfRange
    end
    return comp_code.Success, string.char(0)
end


local function get_board_id(fru_obj, component_obj)
    if not component_obj then
        return comp_code.ParmOutOfRange
    end
    if fru_obj and fru_obj['Type'] == common.COMPONENT_TYPE_PCIE_CARD then
        return comp_code.ParmOutOfRange
    end
    if not component_obj or not component_obj['BoardId'] then
        return comp_code.ParmOutOfRange
    end

    local boardid_high = string.char((component_obj['BoardId'] >> 8) & 0xff)
    local boardid_low = string.char(component_obj['BoardId'] & 0xff)
    return comp_code.Success, boardid_low .. boardid_high
end


local function get_board_pcb_id(fru_obj, component_obj)
    if not fru_obj then
        return comp_code.ParmOutOfRange
    end
    if not fru_obj['PcbId'] then
        return comp_code.UnspecifiedError
    end
    return comp_code.Success, string.char(fru_obj['PcbId'])
end


local function get_board_name(fru_obj, component_obj, fruid)
    if not fru_obj then
        return comp_code.ParmOutOfRange
    end
    if not fru_obj['FruName'] then
        return comp_code.UnspecifiedError
    end
    return comp_code.Success, fru_obj['FruName']
end

local function get_compoent_type(fru_obj, component_obj, fruid)
    if not fru_obj then
        return comp_code.ParmOutOfRange
    end
    if fruid == 0 then
        return comp_code.Success, string.char(0xa0)
    end
    return comp_code.Success, string.char(0x0b)
end


local function get_bom_id(fru_obj, component_obj, fruid)
    return comp_code.ParmOutOfRange
end


local function get_fru_slot_id(fru_obj, component_obj, fruid)
    return comp_code.ParmOutOfRange
end


local function get_unique_id(fru_obj, component_obj, fruid)
    if not component_obj or fruid == 0xff then
        return comp_code.ParmOutOfRange
    end
    if not component_obj['UniqueId'] then
        return comp_code.ParmOutOfRange
    end
    return comp_code.Success, component_obj['UniqueId']
end


function_table.get= {
    [common.INFO_TYPE_DFT_PRESENT] = get_board_present,
    [common.INFO_TYPE_DFT_PRODUCT_ID] = get_board_product_id,
    [common.INFO_TYPE_DFT_BOARD_ID] = get_board_id,
    [common.INFO_TYPE_DFT_PCB_ID] = get_board_pcb_id,
    [common.INFO_TYPE_DFT_NAME] = get_board_name,
    [common.INFO_TYPE_DFT_COMPONENT_TYPE] = get_compoent_type,
    [common.INFO_TYPE_DFT_BOM_ID] = get_bom_id,
    [common.INFO_TYPE_DFT_SLOT_ID] = get_fru_slot_id,
    [common.INFO_TYPE_DFT_SDI_UNIQUE_ID] = get_unique_id
}

return function_table