-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local log = require 'mc.logging'
local error = require 'mc.error'
local new_error = error.new_error
local print_log = error.print_log
local print_trace = error.print_trace
local regist_err_eng = error.register_err

local M = {}

local InvalidCommand = {name = 'kepler.ipmi.InvalidCommand', format = [=[Invalid command]=], severity = 'error'}
M.InvalidCommand = InvalidCommand.name
---@return Error
function M.invalid_command()
    local err_data = new_error(InvalidCommand.name, InvalidCommand.format)
    regist_err_eng(InvalidCommand, 400, nil, 0xC1)
    print_log(log.ERROR, InvalidCommand.format)
    return err_data
end

local DesUnavailable = {name = 'kepler.ipmi.DesUnavailable', format = [=[Destination unavailable]=], severity = 'error'}
M.DesUnavailable = DesUnavailable.name
---@return Error
function M.des_unavailable()
    local err_data = new_error(DesUnavailable.name, DesUnavailable.format)
    regist_err_eng(DesUnavailable, 400, nil, 0xD3)
    print_log(log.ERROR, DesUnavailable.format)
    return err_data
end

local Status_Invalid = {
    name = 'kepler.ipmi.Status_Invalid',
    format = [=[Command, or request parameter(s), not supported in]=] .. [=[ present state.]=],
    severity = 'error'
}
M.Status_Invalid = Status_Invalid.name
---@return Error
function M.status__invalid()
    local err_data = new_error(Status_Invalid.name, Status_Invalid.format)
    regist_err_eng(Status_Invalid, 400, nil, 0xD5)
    print_log(log.ERROR, Status_Invalid.format)
    return err_data
end

local OutOfRange = {name = 'kepler.ipmi.OutOfRange', format = [=[Parameter out of range]=], severity = 'error'}
M.OutOfRange = OutOfRange.name
---@return Error
function M.out_of_range()
    local err_data = new_error(OutOfRange.name, OutOfRange.format)
    regist_err_eng(OutOfRange, 400, nil, 0xC9)
    print_log(log.ERROR, OutOfRange.format)
    return err_data
end

local InvalidField = {
    name = 'kepler.ipmi.InvalidField',
    format = [=[Invalid data field in request]=],
    severity = 'error'
}
M.InvalidField = InvalidField.name
---@return Error
function M.invalid_field()
    local err_data = new_error(InvalidField.name, InvalidField.format)
    regist_err_eng(InvalidField, 400, nil, 0xCC)
    print_log(log.ERROR, InvalidField.format)
    return err_data
end

local ReqDataLenInvalid = {
    name = 'kepler.ipmi.ReqDataLenInvalid',
    format = [=[Request data length invalid]=],
    severity = 'error'
}
M.ReqDataLenInvalid = ReqDataLenInvalid.name
---@return Error
function M.req_data_len_invalid()
    local err_data = new_error(ReqDataLenInvalid.name, ReqDataLenInvalid.format)
    regist_err_eng(ReqDataLenInvalid, 400, nil, 0xC7)
    print_log(log.ERROR, ReqDataLenInvalid.format)
    return err_data
end

local UnknownCause = {name = 'kepler.ipmi.UnknownCause', format = [=[Unspecified error]=], severity = 'error'}
M.UnknownCause = UnknownCause.name
---@return Error
function M.unknown_cause()
    local err_data = new_error(UnknownCause.name, UnknownCause.format)
    regist_err_eng(UnknownCause, 400, nil, 0xFF)
    print_log(log.ERROR, UnknownCause.format)
    return err_data
end

return M
