-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local mdb = require 'mc.mdb'
local class = require 'mc.class_mgnt'
local privilege = require 'mc.privilege'

local fru_data_class_types = require 'class.types.FruData'
local persistent_information_class_types = require 'class.types.PersistentInformation'
local component_mapping_class_types = require 'class.types.ComponentMapping'
local position_fru_id_mapping_class_types = require 'class.types.PositionFruIdMapping'
local dft_eeprom_class_types = require 'class.types.DftEeprom'
local dft_version_class_types = require 'class.types.DftVersion'
local dft_eeprom_wp_class_types = require 'class.types.DftEepromWp'
local persist_poweroff_class_types = require 'class.types.PersistPoweroff'
local overview_intf_types = require 'frudata.json_types.Overview'
local chassis_intf_types = require 'frudata.json_types.Chassis'
local board_intf_types = require 'frudata.json_types.Board'
local product_intf_types = require 'frudata.json_types.Product'
local system_intf_types = require 'frudata.json_types.System'
local frudata_intf_types = require 'frudata.json_types.Frudata'
local properties_intf_types = require 'frudata.json_types.Properties'
local component_intf_types = require 'frudata.json_types.Component'
local components_intf_types = require 'frudata.json_types.Components'
local frus_intf_types = require 'frudata.json_types.Frus'
local fru_intf_types = require 'frudata.json_types.Fru'
local manufacture_intf_types = require 'frudata.json_types.Manufacture'
local extension_intf_types = require 'frudata.json_types.Extension'

local FruData = {
    ['prop_configs'] = {
        ['FruDev'] = {
            ['usage'] = {'CSR'},
            ['baseType'] = 'U8[]',
            ['refInterface'] = 'bmc.kepler.Chip.BlockIO',
            ['validator'] = fru_data_class_types.FruDev
        },
        ['EepromWp'] = {['usage'] = {'CSR'}, ['baseType'] = 'U8', ['validator'] = fru_data_class_types.EepromWp},
        ['StorageType'] = {
            ['usage'] = {'CSR'},
            ['baseType'] = 'String',
            ['validator'] = fru_data_class_types.StorageType
        },
        ['StorageLoc'] = {['usage'] = {'CSR'}, ['baseType'] = 'String', ['validator'] = fru_data_class_types.StorageLoc},
        ['FruAreaSizeBytes'] = {
            ['usage'] = {'CSR'},
            ['baseType'] = 'U16',
            ['default'] = 2048,
            ['description'] = 'Fru电子标签空间大小',
            ['validator'] = fru_data_class_types.FruAreaSizeBytes
        },
        ['HeaderOffset'] = {
            ['usage'] = {'CSR'},
            ['baseType'] = 'U16',
            ['default'] = 0,
            ['description'] = 'eeprom起始地址偏移',
            ['validator'] = fru_data_class_types.HeaderOffset
        }
    },
    ['default_props'] = {
        ['FruDev'] = fru_data_class_types.FruDev.default[1],
        ['EepromWp'] = fru_data_class_types.EepromWp.default[1],
        ['StorageType'] = fru_data_class_types.StorageType.default[1],
        ['StorageLoc'] = fru_data_class_types.StorageLoc.default[1],
        ['FruAreaSizeBytes'] = 2048,
        ['HeaderOffset'] = 0
    },
    ['mdb_prop_configs'] = {
        ['bmc.kepler.Systems.FruData.Overview'] = {
            ['FruId'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['description'] = '关联所在的Fru的Fruid',
                ['usage'] = {'CSR'},
                ['validator'] = overview_intf_types.FruId
            },
            ['FruType'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '关联所在的Fru的类型',
                ['usage'] = {'CSR'},
                ['validator'] = overview_intf_types.FruType
            },
            ['FruName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '关联所在的Fru的名字',
                ['usage'] = {'CSR'},
                ['validator'] = overview_intf_types.FruName
            }
        },
        ['bmc.kepler.Systems.FruData.Chassis'] = {
            ['ChassisType'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '关联的Chassis类型',
                ['validator'] = chassis_intf_types.ChassisType
            },
            ['ChassisPartNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '关联的Chassis部件号',
                ['validator'] = chassis_intf_types.ChassisPartNumber
            },
            ['ChassisSerialNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '关联的Chassis序列号',
                ['validator'] = chassis_intf_types.ChassisSerialNumber
            },
            ['ChassisCustomInfo'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '关联的Chassis扩展域',
                ['validator'] = chassis_intf_types.ChassisCustomInfo
            }
        },
        ['bmc.kepler.Systems.FruData.Board'] = {
            ['MfgDate'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '生产日期',
                ['validator'] = board_intf_types.MfgDate
            },
            ['BoardManufacturer'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '单板生产厂商默认Huawei',
                ['usage'] = {'CSR'},
                ['validator'] = board_intf_types.BoardManufacturer
            },
            ['BoardProductName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '单板产品名称',
                ['usage'] = {'CSR'},
                ['validator'] = board_intf_types.BoardProductName
            },
            ['BoardSerialNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '单板系列化编号',
                ['usage'] = {'CSR'},
                ['validator'] = board_intf_types.BoardSerialNumber
            },
            ['BoardPartNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '单板部件号',
                ['usage'] = {'CSR'},
                ['validator'] = board_intf_types.BoardPartNumber
            },
            ['BoardFRUFileID'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '单板FRufileID',
                ['validator'] = board_intf_types.BoardFRUFileID
            },
            ['BoardCustomInfo'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '单板自定义信息',
                ['validator'] = board_intf_types.BoardCustomInfo
            }
        },
        ['bmc.kepler.Systems.FruData.Product'] = {
            ['ManufacturerName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '制造商名称',
                ['validator'] = product_intf_types.ManufacturerName
            },
            ['ProductName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '产品名，由整机测写入',
                ['validator'] = product_intf_types.ProductName
            },
            ['ProductPartNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '产品部件号',
                ['validator'] = product_intf_types.ProductPartNumber
            },
            ['ProductVersion'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '产品网元类型/版本号，由产品软件定义',
                ['validator'] = product_intf_types.ProductVersion
            },
            ['ProductSerialNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '产品序列号',
                ['usage'] = {'CSR'},
                ['validator'] = product_intf_types.ProductSerialNumber
            },
            ['AssetTag'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '客户资产号，由客户定义',
                ['privilege'] = {['read'] = {'ReadOnly'}, ['write'] = {'BasicSetting'}},
                ['validator'] = product_intf_types.AssetTag
            },
            ['ProductFRUFileID'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '产品fru编号',
                ['validator'] = product_intf_types.ProductFRUFileID
            },
            ['ProductCustomInfo'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '产品定制信息',
                ['validator'] = product_intf_types.ProductCustomInfo
            }
        },
        ['bmc.kepler.Systems.FruData.System'] = {
            ['SystemManufacturer'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '系统制造商',
                ['validator'] = system_intf_types.SystemManufacturer
            },
            ['SystemProductName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '系统产品名称',
                ['usage'] = {'CSR'},
                ['validator'] = system_intf_types.SystemProductName
            },
            ['SystemVersion'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '系统版本',
                ['validator'] = system_intf_types.SystemVersion
            },
            ['SystemSerialNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '系统序列号',
                ['usage'] = {'CSR'},
                ['validator'] = system_intf_types.SystemSerialNumber
            }
        },
        ['bmc.kepler.FrudataService.Frudata'] = {},
        ['bmc.kepler.Object.Properties'] = {
            ['ClassName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象类名',
                ['validator'] = properties_intf_types.ClassName
            },
            ['ObjectName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象名',
                ['validator'] = properties_intf_types.ObjectName
            },
            ['ObjectIdentifier'] = {
                ['baseType'] = 'Struct',
                ['$ref'] = '#/defs/StructIdentifier',
                ['readOnly'] = true,
                ['description'] = '对象标志符',
                ['validator'] = properties_intf_types.ObjectIdentifier
            }
        }
    },
    ['mdb_method_configs'] = {
        ['bmc.kepler.FrudataService.Frudata'] = {
            ['SetProductAssetTag'] = {
                ['description'] = '设置产品资产标签',
                ['req'] = {
                    {['baseType'] = 'U8', ['description'] = '电子标签ID', ['param'] = 'FruId'},
                    {['baseType'] = 'String', ['description'] = '资产标签', ['param'] = 'AssetTag'}
                },
                ['rsp'] = {},
                ['privilege'] = {'BasicSetting'}
            },
            ['SetSysProductName'] = {
                ['description'] = '设置系统产品名称接口',
                ['req'] = {
                    {['baseType'] = 'U8', ['description'] = '电子标签唯一标识FruId', ['param'] = 'FruId'},
                    {['baseType'] = 'String', ['description'] = '产品名称', ['param'] = 'ProductName'}
                },
                ['rsp'] = {},
                ['privilege'] = {'BasicSetting'}
            },
            ['Update'] = {
                ['description'] = '非标准电子标签更新资源树属性值接口',
                ['req'] = {
                    {
                        ['baseType'] = 'String[]',
                        ['description'] = '电子标签属性名数组',
                        ['param'] = 'PropertyName'
                    }, {['baseType'] = 'String[]', ['description'] = '电子标签属性值数组', ['param'] = 'Value'}
                },
                ['rsp'] = {},
                ['privilege'] = {'BasicSetting'}
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['GetOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'},
                    {['baseType'] = 'String', ['description'] = '属性名', ['param'] = 'Property'}
                },
                ['rsp'] = {
                    {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['description'] = '按接口名和属性名查询属性的能力选项'
            },
            ['GetPropertiesByOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'}, {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['rsp'] = {{['baseType'] = 'String[]', ['description'] = '属性名列表', ['param'] = 'Properties'}},
                ['description'] = '按接口名和能力选项查询符合条件的所有属性'
            }
        }
    },
    ['mdb_classes'] = mdb.get_class_obj('/bmc/kepler/Systems/:SystemId/FruDatas/:Id'),
    ['new_mdb_objects'] = mdb.new_objects_builder({
        ['bmc.kepler.Systems.FruData.Overview'] = {
            ['property_defaults'] = {
                ['FruId'] = overview_intf_types.FruId.default[1],
                ['FruType'] = overview_intf_types.FruType.default[1],
                ['FruName'] = overview_intf_types.FruName.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = overview_intf_types
        },
        ['bmc.kepler.Systems.FruData.Chassis'] = {
            ['property_defaults'] = {
                ['ChassisType'] = chassis_intf_types.ChassisType.default[1],
                ['ChassisPartNumber'] = chassis_intf_types.ChassisPartNumber.default[1],
                ['ChassisSerialNumber'] = chassis_intf_types.ChassisSerialNumber.default[1],
                ['ChassisCustomInfo'] = chassis_intf_types.ChassisCustomInfo.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = chassis_intf_types
        },
        ['bmc.kepler.Systems.FruData.Board'] = {
            ['property_defaults'] = {
                ['MfgDate'] = board_intf_types.MfgDate.default[1],
                ['BoardManufacturer'] = board_intf_types.BoardManufacturer.default[1],
                ['BoardProductName'] = board_intf_types.BoardProductName.default[1],
                ['BoardSerialNumber'] = board_intf_types.BoardSerialNumber.default[1],
                ['BoardPartNumber'] = board_intf_types.BoardPartNumber.default[1],
                ['BoardFRUFileID'] = board_intf_types.BoardFRUFileID.default[1],
                ['BoardCustomInfo'] = board_intf_types.BoardCustomInfo.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = board_intf_types
        },
        ['bmc.kepler.Systems.FruData.Product'] = {
            ['property_defaults'] = {
                ['ManufacturerName'] = product_intf_types.ManufacturerName.default[1],
                ['ProductName'] = product_intf_types.ProductName.default[1],
                ['ProductPartNumber'] = product_intf_types.ProductPartNumber.default[1],
                ['ProductVersion'] = product_intf_types.ProductVersion.default[1],
                ['ProductSerialNumber'] = product_intf_types.ProductSerialNumber.default[1],
                ['AssetTag'] = product_intf_types.AssetTag.default[1],
                ['ProductFRUFileID'] = product_intf_types.ProductFRUFileID.default[1],
                ['ProductCustomInfo'] = product_intf_types.ProductCustomInfo.default[1]
            },
            ['privileges'] = {
                ['path'] = privilege.ReadOnly,
                ['props'] = {['AssetTag'] = {['read'] = privilege.ReadOnly, ['write'] = privilege.BasicSetting}}
            },
            ['interface_types'] = product_intf_types
        },
        ['bmc.kepler.Systems.FruData.System'] = {
            ['property_defaults'] = {
                ['SystemManufacturer'] = system_intf_types.SystemManufacturer.default[1],
                ['SystemProductName'] = system_intf_types.SystemProductName.default[1],
                ['SystemVersion'] = system_intf_types.SystemVersion.default[1],
                ['SystemSerialNumber'] = system_intf_types.SystemSerialNumber.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = system_intf_types
        },
        ['bmc.kepler.FrudataService.Frudata'] = {
            ['property_defaults'] = {},
            ['privileges'] = {
                ['path'] = privilege.ReadOnly,
                ['methods'] = {
                    ['SetProductAssetTag'] = privilege.BasicSetting,
                    ['SetSysProductName'] = privilege.BasicSetting,
                    ['Update'] = privilege.BasicSetting
                }
            },
            ['interface_types'] = frudata_intf_types
        },
        ['bmc.kepler.Object.Properties'] = {
            ['property_defaults'] = {
                ['ClassName'] = properties_intf_types.ClassName.default[1],
                ['ObjectName'] = properties_intf_types.ObjectName.default[1],
                ['ObjectIdentifier'] = properties_intf_types.ObjectIdentifier.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = properties_intf_types
        }
    })
}

local Component = {
    ['mdb_prop_configs'] = {
        ['bmc.kepler.Systems.Component'] = {
            ['FruId'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '关联所在的Fru的Fruid',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.FruId
            },
            ['Instance'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['description'] = '组件设备Number',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.Instance
            },
            ['Type'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '组件类型',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.Type
            },
            ['Name'] = {
                ['baseType'] = 'String',
                ['readOnly'] = false,
                ['description'] = '组件设备名称',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.Name
            },
            ['Presence'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['description'] = '组件在位情况',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.Presence
            },
            ['Health'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = false,
                ['description'] = '组件健康状态',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.Health
            },
            ['PowerState'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '组件上电状态',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.PowerState
            },
            ['BoardId'] = {
                ['baseType'] = 'U16',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '组件单板BoardId',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.BoardId
            },
            ['UniqueId'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '组件唯一标识,Vendor + ComponentID',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.UniqueId
            },
            ['Manufacturer'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '组件厂商信息',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.Manufacturer
            },
            ['GroupId'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['default'] = 1,
                ['description'] = '组件逻辑组Id',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.GroupId
            },
            ['Location'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '组件的容器',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.Location
            },
            ['SerialNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '组件序列号',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.SerialNumber
            },
            ['PartNumber'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '组件号',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.PartNumber
            },
            ['SegmentId'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '组件容器ID',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.SegmentId
            },
            ['Function'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '部件功能信息',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.Function
            },
            ['PreviousSN'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '更换前SN,SerialNumber,来自board区域掉电持久化',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.PreviousSN
            },
            ['ReplaceFlag'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['description'] = '部件是否发生了更换,用于产生SEL日志',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.ReplaceFlag
            },
            ['NodeId'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '节点的ID',
                ['usage'] = {'CSR'},
                ['validator'] = component_intf_types.NodeId
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['ClassName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象类名',
                ['validator'] = properties_intf_types.ClassName
            },
            ['ObjectName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象名',
                ['validator'] = properties_intf_types.ObjectName
            },
            ['ObjectIdentifier'] = {
                ['baseType'] = 'Struct',
                ['$ref'] = '#/defs/StructIdentifier',
                ['readOnly'] = true,
                ['description'] = '对象标志符',
                ['validator'] = properties_intf_types.ObjectIdentifier
            }
        }
    },
    ['mdb_method_configs'] = {
        ['bmc.kepler.Systems.Component'] = {
            ['UpdateHealth'] = {
                ['description'] = '更新组件健康状态',
                ['req'] = {{['baseType'] = 'U8', ['description'] = '健康状态', ['param'] = 'Health'}},
                ['rsp'] = {},
                ['privilege'] = {'BasicSetting'}
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['GetOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'},
                    {['baseType'] = 'String', ['description'] = '属性名', ['param'] = 'Property'}
                },
                ['rsp'] = {
                    {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['description'] = '按接口名和属性名查询属性的能力选项'
            },
            ['GetPropertiesByOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'}, {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['rsp'] = {{['baseType'] = 'String[]', ['description'] = '属性名列表', ['param'] = 'Properties'}},
                ['description'] = '按接口名和能力选项查询符合条件的所有属性'
            }
        }
    },
    ['mdb_classes'] = mdb.get_class_obj('/bmc/kepler/Systems/:SystemId/Components/:Id'),
    ['new_mdb_objects'] = mdb.new_objects_builder({
        ['bmc.kepler.Systems.Component'] = {
            ['property_defaults'] = {
                ['FruId'] = component_intf_types.FruId.default[1],
                ['Instance'] = component_intf_types.Instance.default[1],
                ['Type'] = component_intf_types.Type.default[1],
                ['Name'] = component_intf_types.Name.default[1],
                ['Presence'] = component_intf_types.Presence.default[1],
                ['Health'] = component_intf_types.Health.default[1],
                ['PowerState'] = component_intf_types.PowerState.default[1],
                ['BoardId'] = component_intf_types.BoardId.default[1],
                ['UniqueId'] = component_intf_types.UniqueId.default[1],
                ['Manufacturer'] = component_intf_types.Manufacturer.default[1],
                ['GroupId'] = 1,
                ['Location'] = component_intf_types.Location.default[1],
                ['SerialNumber'] = component_intf_types.SerialNumber.default[1],
                ['PartNumber'] = component_intf_types.PartNumber.default[1],
                ['SegmentId'] = component_intf_types.SegmentId.default[1],
                ['Function'] = component_intf_types.Function.default[1],
                ['PreviousSN'] = component_intf_types.PreviousSN.default[1],
                ['ReplaceFlag'] = component_intf_types.ReplaceFlag.default[1],
                ['NodeId'] = component_intf_types.NodeId.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly, ['methods'] = {['UpdateHealth'] = privilege.BasicSetting}},
            ['interface_types'] = component_intf_types
        },
        ['bmc.kepler.Object.Properties'] = {
            ['property_defaults'] = {
                ['ClassName'] = properties_intf_types.ClassName.default[1],
                ['ObjectName'] = properties_intf_types.ObjectName.default[1],
                ['ObjectIdentifier'] = properties_intf_types.ObjectIdentifier.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = properties_intf_types
        }
    })
}

local Components = {
    ['mdb_prop_configs'] = {
        ['bmc.kepler.Systems.Components'] = {},
        ['bmc.kepler.Object.Properties'] = {
            ['ClassName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象类名',
                ['validator'] = properties_intf_types.ClassName
            },
            ['ObjectName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象名',
                ['validator'] = properties_intf_types.ObjectName
            },
            ['ObjectIdentifier'] = {
                ['baseType'] = 'Struct',
                ['$ref'] = '#/defs/StructIdentifier',
                ['readOnly'] = true,
                ['description'] = '对象标志符',
                ['validator'] = properties_intf_types.ObjectIdentifier
            }
        }
    },
    ['mdb_method_configs'] = {
        ['bmc.kepler.Systems.Components'] = {
            ['GetComponentTypes'] = {
                ['description'] = '获取当前存在的所有对象的部件类型',
                ['req'] = {},
                ['rsp'] = {
                    {
                        ['baseType'] = 'Array',
                        ['items'] = {['$ref'] = '#/defs/ComponentTypes'},
                        ['description'] = '所有对象的部件类型列表',
                        ['param'] = 'ComponentTypes'
                    }
                },
                ['privilege'] = {'ReadOnly'}
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['GetOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'},
                    {['baseType'] = 'String', ['description'] = '属性名', ['param'] = 'Property'}
                },
                ['rsp'] = {
                    {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['description'] = '按接口名和属性名查询属性的能力选项'
            },
            ['GetPropertiesByOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'}, {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['rsp'] = {{['baseType'] = 'String[]', ['description'] = '属性名列表', ['param'] = 'Properties'}},
                ['description'] = '按接口名和能力选项查询符合条件的所有属性'
            }
        }
    },
    ['mdb_classes'] = mdb.get_class_obj('/bmc/kepler/Systems/:SystemId/Components'),
    ['new_mdb_objects'] = mdb.new_objects_builder({
        ['bmc.kepler.Systems.Components'] = {
            ['property_defaults'] = {},
            ['privileges'] = {['path'] = privilege.ReadOnly, ['methods'] = {['GetComponentTypes'] = privilege.ReadOnly}},
            ['interface_types'] = components_intf_types
        },
        ['bmc.kepler.Object.Properties'] = {
            ['property_defaults'] = {
                ['ClassName'] = properties_intf_types.ClassName.default[1],
                ['ObjectName'] = properties_intf_types.ObjectName.default[1],
                ['ObjectIdentifier'] = properties_intf_types.ObjectIdentifier.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = properties_intf_types
        }
    })
}

local Frus = {
    ['mdb_prop_configs'] = {
        ['bmc.kepler.Systems.Frus'] = {},
        ['bmc.kepler.Object.Properties'] = {
            ['ClassName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象类名',
                ['validator'] = properties_intf_types.ClassName
            },
            ['ObjectName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象名',
                ['validator'] = properties_intf_types.ObjectName
            },
            ['ObjectIdentifier'] = {
                ['baseType'] = 'Struct',
                ['$ref'] = '#/defs/StructIdentifier',
                ['readOnly'] = true,
                ['description'] = '对象标志符',
                ['validator'] = properties_intf_types.ObjectIdentifier
            }
        }
    },
    ['mdb_method_configs'] = {
        ['bmc.kepler.Object.Properties'] = {
            ['GetOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'},
                    {['baseType'] = 'String', ['description'] = '属性名', ['param'] = 'Property'}
                },
                ['rsp'] = {
                    {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['description'] = '按接口名和属性名查询属性的能力选项'
            },
            ['GetPropertiesByOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'}, {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['rsp'] = {{['baseType'] = 'String[]', ['description'] = '属性名列表', ['param'] = 'Properties'}},
                ['description'] = '按接口名和能力选项查询符合条件的所有属性'
            }
        }
    },
    ['mdb_signal_configs'] = {
        ['bmc.kepler.Systems.Frus'] = {
            ['FruAdded'] = {
                {['baseType'] = 'U8', ['description'] = '已经添加的Fru的标识', ['param'] = 'FruId'},
                {['baseType'] = 'String', ['description'] = '已经添加的Fru的名称', ['param'] = 'FruName'},
                {['baseType'] = 'String', ['description'] = '已经添加的Fru的资源路径', ['param'] = 'Path'}
            },
            ['FruRemoved'] = {
                {['baseType'] = 'U8', ['description'] = '即将卸载的Fru的标识', ['param'] = 'FruId'},
                {['baseType'] = 'String', ['description'] = '即将卸载的Fru的名称', ['param'] = 'FruName'},
                {['baseType'] = 'String', ['description'] = '即将卸载的Fru的资源路径', ['param'] = 'Path'}
            }
        }
    },
    ['mdb_classes'] = mdb.get_class_obj('/bmc/kepler/Systems/:SystemId/Frus'),
    ['new_mdb_objects'] = mdb.new_objects_builder({
        ['bmc.kepler.Systems.Frus'] = {
            ['property_defaults'] = {},
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = frus_intf_types
        },
        ['bmc.kepler.Object.Properties'] = {
            ['property_defaults'] = {
                ['ClassName'] = properties_intf_types.ClassName.default[1],
                ['ObjectName'] = properties_intf_types.ObjectName.default[1],
                ['ObjectIdentifier'] = properties_intf_types.ObjectIdentifier.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = properties_intf_types
        }
    })
}

local Fru = {
    ['mdb_prop_configs'] = {
        ['bmc.kepler.Systems.Fru'] = {
            ['PcbId'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['description'] = 'PcbId, 需要关联到硬件, 从硬件读取Id',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.PcbId
            },
            ['PcbVersion'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = 'PCB版本, 根据PcbId的值进行计算得出',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.PcbVersion
            },
            ['FruId'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['description'] = 'FruId',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.FruId
            },
            ['FruName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = 'Fru名称',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.FruName
            },
            ['PowerState'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = 'Fru的热插拔状态',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.PowerState
            },
            ['Health'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['description'] = '健康状态',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.Health
            },
            ['EepStatus'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = 'EEPROM状态',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.EepStatus
            },
            ['GroupPosition'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = 'FRU的位置信息, 来自自发现的Position属性, 该属性用于确定FruId',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.GroupPosition
            },
            ['Type'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = 'FRU类型,TYPE值请参照IPD Lite的BMC IPMI接口说明附录',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.Type
            },
            ['BoardId'] = {
                ['baseType'] = 'U16',
                ['readOnly'] = true,
                ['description'] = '非天池:组件单板BoardID',
                ['usage'] = {'CSR'},
                ['default'] = 65535,
                ['validator'] = fru_intf_types.BoardId
            },
            ['UniqueId'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '天池:组件唯一标识,Vendor + ComponentID',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.UniqueId
            },
            ['FruDataId'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '关联frudata对象名',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.FruDataId
            },
            ['ConnectorGroupId'] = {
                ['baseType'] = 'U32',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '关联Connector传过来的GroupId',
                ['usage'] = {'CSR'},
                ['validator'] = fru_intf_types.ConnectorGroupId
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['ClassName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象类名',
                ['validator'] = properties_intf_types.ClassName
            },
            ['ObjectName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象名',
                ['validator'] = properties_intf_types.ObjectName
            },
            ['ObjectIdentifier'] = {
                ['baseType'] = 'Struct',
                ['$ref'] = '#/defs/StructIdentifier',
                ['readOnly'] = true,
                ['description'] = '对象标志符',
                ['validator'] = properties_intf_types.ObjectIdentifier
            }
        }
    },
    ['mdb_method_configs'] = {
        ['bmc.kepler.Object.Properties'] = {
            ['GetOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'},
                    {['baseType'] = 'String', ['description'] = '属性名', ['param'] = 'Property'}
                },
                ['rsp'] = {
                    {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['description'] = '按接口名和属性名查询属性的能力选项'
            },
            ['GetPropertiesByOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'}, {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['rsp'] = {{['baseType'] = 'String[]', ['description'] = '属性名列表', ['param'] = 'Properties'}},
                ['description'] = '按接口名和能力选项查询符合条件的所有属性'
            }
        }
    },
    ['mdb_classes'] = mdb.get_class_obj('/bmc/kepler/Systems/:SystemId/Frus/:Id'),
    ['new_mdb_objects'] = mdb.new_objects_builder({
        ['bmc.kepler.Systems.Fru'] = {
            ['property_defaults'] = {
                ['PcbId'] = fru_intf_types.PcbId.default[1],
                ['PcbVersion'] = fru_intf_types.PcbVersion.default[1],
                ['FruId'] = fru_intf_types.FruId.default[1],
                ['FruName'] = fru_intf_types.FruName.default[1],
                ['PowerState'] = fru_intf_types.PowerState.default[1],
                ['Health'] = fru_intf_types.Health.default[1],
                ['EepStatus'] = fru_intf_types.EepStatus.default[1],
                ['GroupPosition'] = fru_intf_types.GroupPosition.default[1],
                ['Type'] = fru_intf_types.Type.default[1],
                ['BoardId'] = 65535,
                ['UniqueId'] = fru_intf_types.UniqueId.default[1],
                ['FruDataId'] = fru_intf_types.FruDataId.default[1],
                ['ConnectorGroupId'] = fru_intf_types.ConnectorGroupId.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = fru_intf_types
        },
        ['bmc.kepler.Object.Properties'] = {
            ['property_defaults'] = {
                ['ClassName'] = properties_intf_types.ClassName.default[1],
                ['ObjectName'] = properties_intf_types.ObjectName.default[1],
                ['ObjectIdentifier'] = properties_intf_types.ObjectIdentifier.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = properties_intf_types
        }
    })
}

local PersistentInformation = {
    ['table_name'] = 't_persistence_info',
    ['prop_configs'] = {
        ['FruId'] = {
            ['baseType'] = 'U8',
            ['primaryKey'] = true,
            ['validator'] = persistent_information_class_types.FruId
        },
        ['PersistentName'] = {
            ['baseType'] = 'String',
            ['validator'] = persistent_information_class_types.PersistentName
        },
        ['PersistentValue'] = {
            ['baseType'] = 'String',
            ['validator'] = persistent_information_class_types.PersistentValue
        }
    },
    ['default_props'] = {
        ['FruId'] = persistent_information_class_types.FruId.default[1],
        ['PersistentName'] = persistent_information_class_types.PersistentName.default[1],
        ['PersistentValue'] = persistent_information_class_types.PersistentValue.default[1]
    }
}

local ComponentMapping = {
    ['table_name'] = 't_component_mapping',
    ['prop_configs'] = {
        ['PreviousSN'] = {
            ['baseType'] = 'String',
            ['usage'] = {'PoweroffPer'},
            ['validator'] = component_mapping_class_types.PreviousSN
        },
        ['ComponentName'] = {
            ['usage'] = {'PoweroffPer'},
            ['primaryKey'] = true,
            ['baseType'] = 'String',
            ['validator'] = component_mapping_class_types.ComponentName
        }
    },
    ['default_props'] = {
        ['PreviousSN'] = component_mapping_class_types.PreviousSN.default[1],
        ['ComponentName'] = component_mapping_class_types.ComponentName.default[1]
    }
}

local PositionFruIdMapping = {
    ['table_name'] = 't_position_fru_id_mapping',
    ['prop_configs'] = {
        ['Position'] = {
            ['baseType'] = 'String',
            ['usage'] = {'PoweroffPer'},
            ['validator'] = position_fru_id_mapping_class_types.Position
        },
        ['Id'] = {
            ['usage'] = {'PoweroffPer'},
            ['primaryKey'] = true,
            ['baseType'] = 'U8',
            ['validator'] = position_fru_id_mapping_class_types.Id
        }
    },
    ['default_props'] = {
        ['Position'] = position_fru_id_mapping_class_types.Position.default[1],
        ['Id'] = position_fru_id_mapping_class_types.Id.default[1]
    }
}

local DftEeprom = {
    ['prop_configs'] = {
        ['FruData'] = {['usage'] = {'CSR'}, ['baseType'] = 'String', ['validator'] = dft_eeprom_class_types.FruData}
    },
    ['default_props'] = {['FruData'] = dft_eeprom_class_types.FruData.default[1]},
    ['mdb_prop_configs'] = {
        ['bmc.kepler.Manufacture'] = {
            ['Type'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '装备测试项类型，1人工自检，2需要人工准备前置条件，3拷机测试，4人工检查结果，5人工操作测试，6与装备交互测试',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.Type
            },
            ['Id'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '装备测试项id',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.Id
            },
            ['Slot'] = {
                ['baseType'] = 'U32',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '槽位号',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.Slot
            },
            ['Param'] = {
                ['baseType'] = 'U8[]',
                ['readOnly'] = false,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '测试项传入的参数',
                ['validator'] = manufacture_intf_types.Param
            },
            ['DeviceNum'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '设备id',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.DeviceNum
            },
            ['ItemName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '设备名称',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.ItemName
            },
            ['PrompteReady'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '测试前交互提示',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.PrompteReady
            },
            ['PrompteFinish'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '测试完成交互提示',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.PrompteFinish
            },
            ['ProcessPeriod'] = {
                ['baseType'] = 'U16',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '测试所需时间',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.ProcessPeriod
            },
            ['Status'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '当前测试状态',
                ['enum'] = {'Complete', 'Testing', 'Unstart'},
                ['default'] = 'Unstart',
                ['validator'] = manufacture_intf_types.Status
            }
        },
        ['bmc.kepler.Manufacture.Extension'] = {
            ['SystemId'] = {
                ['baseType'] = 'U16',
                ['readOnly'] = true,
                ['description'] = 'Multihost环境下system编号',
                ['default'] = 1,
                ['usage'] = {'CSR'},
                ['validator'] = extension_intf_types.SystemId
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['ClassName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象类名',
                ['validator'] = properties_intf_types.ClassName
            },
            ['ObjectName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象名',
                ['validator'] = properties_intf_types.ObjectName
            },
            ['ObjectIdentifier'] = {
                ['baseType'] = 'Struct',
                ['$ref'] = '#/defs/StructIdentifier',
                ['readOnly'] = true,
                ['description'] = '对象标志符',
                ['validator'] = properties_intf_types.ObjectIdentifier
            }
        }
    },
    ['mdb_method_configs'] = {
        ['bmc.kepler.Manufacture'] = {
            ['Start'] = {['req'] = {}, ['rsp'] = {}, ['description'] = '开始装备测试项'},
            ['Stop'] = {['req'] = {}, ['rsp'] = {}, ['description'] = '停止装备测试项'},
            ['GetResult'] = {
                ['req'] = {},
                ['rsp'] = {
                    {
                        ['baseType'] = 'String',
                        ['enum'] = {'Succeed', 'Failed', 'Non'},
                        ['description'] = '装备测试项状态',
                        ['param'] = 'Status'
                    }, {['baseType'] = 'String', ['description'] = '装备测试项描述', ['param'] = 'Description'}
                },
                ['description'] = '获取装备测试项结果'
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['GetOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'},
                    {['baseType'] = 'String', ['description'] = '属性名', ['param'] = 'Property'}
                },
                ['rsp'] = {
                    {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['description'] = '按接口名和属性名查询属性的能力选项'
            },
            ['GetPropertiesByOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'}, {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['rsp'] = {{['baseType'] = 'String[]', ['description'] = '属性名列表', ['param'] = 'Properties'}},
                ['description'] = '按接口名和能力选项查询符合条件的所有属性'
            }
        }
    },
    ['mdb_classes'] = mdb.get_class_obj('/bmc/kepler/Manufacture/SelfTest/DftEeprom/:Id'),
    ['new_mdb_objects'] = mdb.new_objects_builder({
        ['bmc.kepler.Manufacture'] = {
            ['property_defaults'] = {
                ['Type'] = manufacture_intf_types.Type.default[1],
                ['Id'] = manufacture_intf_types.Id.default[1],
                ['Slot'] = manufacture_intf_types.Slot.default[1],
                ['Param'] = manufacture_intf_types.Param.default[1],
                ['DeviceNum'] = manufacture_intf_types.DeviceNum.default[1],
                ['ItemName'] = manufacture_intf_types.ItemName.default[1],
                ['PrompteReady'] = manufacture_intf_types.PrompteReady.default[1],
                ['PrompteFinish'] = manufacture_intf_types.PrompteFinish.default[1],
                ['ProcessPeriod'] = manufacture_intf_types.ProcessPeriod.default[1],
                ['Status'] = 'Unstart'
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = manufacture_intf_types
        },
        ['bmc.kepler.Manufacture.Extension'] = {
            ['property_defaults'] = {['SystemId'] = 1},
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = extension_intf_types
        },
        ['bmc.kepler.Object.Properties'] = {
            ['property_defaults'] = {
                ['ClassName'] = properties_intf_types.ClassName.default[1],
                ['ObjectName'] = properties_intf_types.ObjectName.default[1],
                ['ObjectIdentifier'] = properties_intf_types.ObjectIdentifier.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = properties_intf_types
        }
    })
}

local DftVersion = {
    ['prop_configs'] = {
        ['FruId'] = {['usage'] = {'CSR'}, ['baseType'] = 'U8', ['validator'] = dft_version_class_types.FruId},
        ['VersionType'] = {
            ['usage'] = {'CSR'},
            ['baseType'] = 'U8',
            ['validator'] = dft_version_class_types.VersionType
        },
        ['Version'] = {['usage'] = {'CSR'}, ['baseType'] = 'String', ['validator'] = dft_version_class_types.Version},
        ['UnitNum'] = {['usage'] = {'CSR'}, ['baseType'] = 'U32', ['validator'] = dft_version_class_types.UnitNum},
        ['NeedUintNum'] = {
            ['usage'] = {'CSR'},
            ['baseType'] = 'U8',
            ['validator'] = dft_version_class_types.NeedUintNum
        }
    },
    ['default_props'] = {
        ['FruId'] = dft_version_class_types.FruId.default[1],
        ['VersionType'] = dft_version_class_types.VersionType.default[1],
        ['Version'] = dft_version_class_types.Version.default[1],
        ['UnitNum'] = dft_version_class_types.UnitNum.default[1],
        ['NeedUintNum'] = dft_version_class_types.NeedUintNum.default[1]
    }
}

local DftEepromWp = {
    ['prop_configs'] = {
        ['FruData'] = {['usage'] = {'CSR'}, ['baseType'] = 'String', ['validator'] = dft_eeprom_wp_class_types.FruData}
    },
    ['default_props'] = {['FruData'] = dft_eeprom_wp_class_types.FruData.default[1]},
    ['mdb_prop_configs'] = {
        ['bmc.kepler.Manufacture'] = {
            ['Type'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '装备测试项类型，1人工自检，2需要人工准备前置条件，3拷机测试，4人工检查结果，5人工操作测试，6与装备交互测试',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.Type
            },
            ['Id'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '装备测试项id',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.Id
            },
            ['DeviceNum'] = {
                ['baseType'] = 'U8',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '设备id',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.DeviceNum
            },
            ['ItemName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '设备名称',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.ItemName
            },
            ['PrompteReady'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '测试前交互提示',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.PrompteReady
            },
            ['PrompteFinish'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '测试完成交互提示',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.PrompteFinish
            },
            ['ProcessPeriod'] = {
                ['baseType'] = 'U16',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '测试所需时间',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.ProcessPeriod
            },
            ['Slot'] = {
                ['baseType'] = 'U32',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '槽位号',
                ['usage'] = {'CSR'},
                ['validator'] = manufacture_intf_types.Slot
            },
            ['Param'] = {
                ['baseType'] = 'U8[]',
                ['readOnly'] = false,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '测试项传入的参数',
                ['validator'] = manufacture_intf_types.Param
            },
            ['Status'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['options'] = {['emitsChangedSignal'] = 'false'},
                ['description'] = '当前测试状态',
                ['enum'] = {'Complete', 'Testing', 'Unstart'},
                ['default'] = 'Unstart',
                ['validator'] = manufacture_intf_types.Status
            }
        },
        ['bmc.kepler.Manufacture.Extension'] = {
            ['SystemId'] = {
                ['baseType'] = 'U16',
                ['readOnly'] = true,
                ['description'] = 'Multihost环境下system编号',
                ['default'] = 1,
                ['usage'] = {'CSR'},
                ['validator'] = extension_intf_types.SystemId
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['ClassName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象类名',
                ['validator'] = properties_intf_types.ClassName
            },
            ['ObjectName'] = {
                ['baseType'] = 'String',
                ['readOnly'] = true,
                ['description'] = '对象名',
                ['validator'] = properties_intf_types.ObjectName
            },
            ['ObjectIdentifier'] = {
                ['baseType'] = 'Struct',
                ['$ref'] = '#/defs/StructIdentifier',
                ['readOnly'] = true,
                ['description'] = '对象标志符',
                ['validator'] = properties_intf_types.ObjectIdentifier
            }
        }
    },
    ['mdb_method_configs'] = {
        ['bmc.kepler.Manufacture'] = {
            ['Start'] = {['req'] = {}, ['rsp'] = {}, ['description'] = '开始装备测试项'},
            ['Stop'] = {['req'] = {}, ['rsp'] = {}, ['description'] = '停止装备测试项'},
            ['GetResult'] = {
                ['req'] = {},
                ['rsp'] = {
                    {
                        ['baseType'] = 'String',
                        ['enum'] = {'Succeed', 'Failed', 'Non'},
                        ['description'] = '装备测试项状态',
                        ['param'] = 'Status'
                    }, {['baseType'] = 'String', ['description'] = '装备测试项描述', ['param'] = 'Description'}
                },
                ['description'] = '获取装备测试项结果'
            }
        },
        ['bmc.kepler.Object.Properties'] = {
            ['GetOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'},
                    {['baseType'] = 'String', ['description'] = '属性名', ['param'] = 'Property'}
                },
                ['rsp'] = {
                    {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['description'] = '按接口名和属性名查询属性的能力选项'
            },
            ['GetPropertiesByOptions'] = {
                ['req'] = {
                    {['baseType'] = 'String', ['description'] = '接口名', ['param'] = 'Interface'}, {
                        ['baseType'] = 'Struct',
                        ['$ref'] = '#/defs/Options',
                        ['description'] = '属性能力选项，例如是否弃用、是否是易变属性等',
                        ['param'] = 'Options'
                    }
                },
                ['rsp'] = {{['baseType'] = 'String[]', ['description'] = '属性名列表', ['param'] = 'Properties'}},
                ['description'] = '按接口名和能力选项查询符合条件的所有属性'
            }
        }
    },
    ['mdb_classes'] = mdb.get_class_obj('/bmc/kepler/Manufacture/SelfTest/DftEepromWp/:Id'),
    ['new_mdb_objects'] = mdb.new_objects_builder({
        ['bmc.kepler.Manufacture'] = {
            ['property_defaults'] = {
                ['Type'] = manufacture_intf_types.Type.default[1],
                ['Id'] = manufacture_intf_types.Id.default[1],
                ['DeviceNum'] = manufacture_intf_types.DeviceNum.default[1],
                ['ItemName'] = manufacture_intf_types.ItemName.default[1],
                ['PrompteReady'] = manufacture_intf_types.PrompteReady.default[1],
                ['PrompteFinish'] = manufacture_intf_types.PrompteFinish.default[1],
                ['ProcessPeriod'] = manufacture_intf_types.ProcessPeriod.default[1],
                ['Slot'] = manufacture_intf_types.Slot.default[1],
                ['Param'] = manufacture_intf_types.Param.default[1],
                ['Status'] = 'Unstart'
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = manufacture_intf_types
        },
        ['bmc.kepler.Manufacture.Extension'] = {
            ['property_defaults'] = {['SystemId'] = 1},
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = extension_intf_types
        },
        ['bmc.kepler.Object.Properties'] = {
            ['property_defaults'] = {
                ['ClassName'] = properties_intf_types.ClassName.default[1],
                ['ObjectName'] = properties_intf_types.ObjectName.default[1],
                ['ObjectIdentifier'] = properties_intf_types.ObjectIdentifier.default[1]
            },
            ['privileges'] = {['path'] = privilege.ReadOnly},
            ['interface_types'] = properties_intf_types
        }
    })
}

local PersistPoweroff = {
    ['table_name'] = 't_frudata_poweroff',
    ['prop_configs'] = {
        ['PerId'] = {['primaryKey'] = true, ['baseType'] = 'String', ['validator'] = persist_poweroff_class_types.PerId},
        ['Key'] = {['primaryKey'] = true, ['baseType'] = 'String', ['validator'] = persist_poweroff_class_types.Key},
        ['Value'] = {['baseType'] = 'String', ['validator'] = persist_poweroff_class_types.Value}
    },
    ['default_props'] = {
        ['PerId'] = persist_poweroff_class_types.PerId.default[1],
        ['Key'] = persist_poweroff_class_types.Key.default[1],
        ['Value'] = persist_poweroff_class_types.Value.default[1]
    }
}

local M = {}

function M.init(bus)
    class('FruData', FruData):set_bus(bus)
    class('Component', Component):set_bus(bus)
    class('Components', Components):set_bus(bus)
    class('Frus', Frus):set_bus(bus)
    class('Fru', Fru):set_bus(bus)
    class('PersistentInformation', PersistentInformation):set_bus(bus)
    class('ComponentMapping', ComponentMapping):set_bus(bus)
    class('PositionFruIdMapping', PositionFruIdMapping):set_bus(bus)
    class('DftEeprom', DftEeprom):set_bus(bus)
    class('DftVersion', DftVersion):set_bus(bus)
    class('DftEepromWp', DftEepromWp):set_bus(bus)
    class('PersistPoweroff', PersistPoweroff):set_bus(bus)
end

-- The callback needs to be registered during app initialization
function M.ImplFruDataFrudataSetProductAssetTag(cb)
    class('FruData')['bmc.kepler.FrudataService.Frudata'].SetProductAssetTag = function(obj, ctx, ...)
        local req = frudata_intf_types.SetProductAssetTagReq.new(...):validate(nil, nil, true)
        local rsp = frudata_intf_types.SetProductAssetTagRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplFruDataFrudataSetSysProductName(cb)
    class('FruData')['bmc.kepler.FrudataService.Frudata'].SetSysProductName = function(obj, ctx, ...)
        local req = frudata_intf_types.SetSysProductNameReq.new(...):validate(nil, nil, true)
        local rsp = frudata_intf_types.SetSysProductNameRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplFruDataFrudataUpdate(cb)
    class('FruData')['bmc.kepler.FrudataService.Frudata'].Update = function(obj, ctx, ...)
        local req = frudata_intf_types.UpdateReq.new(...):validate(nil, nil, true)
        local rsp = frudata_intf_types.UpdateRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplFruDataPropertiesGetOptions(cb)
    class('FruData')['bmc.kepler.Object.Properties'].GetOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplFruDataPropertiesGetPropertiesByOptions(cb)
    class('FruData')['bmc.kepler.Object.Properties'].GetPropertiesByOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetPropertiesByOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetPropertiesByOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplComponentComponentUpdateHealth(cb)
    class('Component')['bmc.kepler.Systems.Component'].UpdateHealth = function(obj, ctx, ...)
        local req = component_intf_types.UpdateHealthReq.new(...):validate(nil, nil, true)
        local rsp = component_intf_types.UpdateHealthRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplComponentPropertiesGetOptions(cb)
    class('Component')['bmc.kepler.Object.Properties'].GetOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplComponentPropertiesGetPropertiesByOptions(cb)
    class('Component')['bmc.kepler.Object.Properties'].GetPropertiesByOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetPropertiesByOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetPropertiesByOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplComponentsComponentsGetComponentTypes(cb)
    class('Components')['bmc.kepler.Systems.Components'].GetComponentTypes = function(obj, ctx, ...)
        local req = components_intf_types.GetComponentTypesReq.new(...):validate(nil, nil, true)
        local rsp = components_intf_types.GetComponentTypesRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplComponentsPropertiesGetOptions(cb)
    class('Components')['bmc.kepler.Object.Properties'].GetOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplComponentsPropertiesGetPropertiesByOptions(cb)
    class('Components')['bmc.kepler.Object.Properties'].GetPropertiesByOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetPropertiesByOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetPropertiesByOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplFrusPropertiesGetOptions(cb)
    class('Frus')['bmc.kepler.Object.Properties'].GetOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplFrusPropertiesGetPropertiesByOptions(cb)
    class('Frus')['bmc.kepler.Object.Properties'].GetPropertiesByOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetPropertiesByOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetPropertiesByOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplFruPropertiesGetOptions(cb)
    class('Fru')['bmc.kepler.Object.Properties'].GetOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplFruPropertiesGetPropertiesByOptions(cb)
    class('Fru')['bmc.kepler.Object.Properties'].GetPropertiesByOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetPropertiesByOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetPropertiesByOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromManufactureStart(cb)
    class('DftEeprom')['bmc.kepler.Manufacture'].Start = function(obj, ctx, ...)
        local req = manufacture_intf_types.StartReq.new(...):validate(nil, nil, true)
        local rsp = manufacture_intf_types.StartRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromManufactureStop(cb)
    class('DftEeprom')['bmc.kepler.Manufacture'].Stop = function(obj, ctx, ...)
        local req = manufacture_intf_types.StopReq.new(...):validate(nil, nil, true)
        local rsp = manufacture_intf_types.StopRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromManufactureGetResult(cb)
    class('DftEeprom')['bmc.kepler.Manufacture'].GetResult = function(obj, ctx, ...)
        local req = manufacture_intf_types.GetResultReq.new(...):validate(nil, nil, true)
        local rsp = manufacture_intf_types.GetResultRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromPropertiesGetOptions(cb)
    class('DftEeprom')['bmc.kepler.Object.Properties'].GetOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromPropertiesGetPropertiesByOptions(cb)
    class('DftEeprom')['bmc.kepler.Object.Properties'].GetPropertiesByOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetPropertiesByOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetPropertiesByOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromWpManufactureStart(cb)
    class('DftEepromWp')['bmc.kepler.Manufacture'].Start = function(obj, ctx, ...)
        local req = manufacture_intf_types.StartReq.new(...):validate(nil, nil, true)
        local rsp = manufacture_intf_types.StartRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromWpManufactureStop(cb)
    class('DftEepromWp')['bmc.kepler.Manufacture'].Stop = function(obj, ctx, ...)
        local req = manufacture_intf_types.StopReq.new(...):validate(nil, nil, true)
        local rsp = manufacture_intf_types.StopRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromWpManufactureGetResult(cb)
    class('DftEepromWp')['bmc.kepler.Manufacture'].GetResult = function(obj, ctx, ...)
        local req = manufacture_intf_types.GetResultReq.new(...):validate(nil, nil, true)
        local rsp = manufacture_intf_types.GetResultRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromWpPropertiesGetOptions(cb)
    class('DftEepromWp')['bmc.kepler.Object.Properties'].GetOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

-- The callback needs to be registered during app initialization
function M.ImplDftEepromWpPropertiesGetPropertiesByOptions(cb)
    class('DftEepromWp')['bmc.kepler.Object.Properties'].GetPropertiesByOptions = function(obj, ctx, ...)
        local req = properties_intf_types.GetPropertiesByOptionsReq.new(...):validate(nil, nil, true)
        local rsp = properties_intf_types.GetPropertiesByOptionsRsp.new(cb(obj, ctx, req:unpack())):validate()
        return rsp:unpack(true)
    end
end

return M
