-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: 配置导入导出
local class = require 'mc.class'
local log = require 'mc.logging'

local custom_messages = require 'messages.custom'
local c_object = require 'mc.orm.object'
local c_super_pod_label_object = require 'super_pod_label_object'
local business_config = {}

-- 导入超节点集合ID
local function import_super_pod_id(self, ctx, config_info)
    local value = config_info.SuperPodId.Value
    local super_pod_label_object = c_object('SuperPodLabel').collection:find({Id = 1})
    if not super_pod_label_object then
        log:operation(ctx:get_initiator(), 'Chassis', 'Set SuperPodId(%s) failed', value)
        error(custom_messages.ResourceNotFound('SuperPodLabel'))
    end
    local res = super_pod_label_object:update_super_pod_id(value)
    if not res then
        log:operation(ctx:get_initiator(), 'Chassis', 'Set SuperPodId(%s) failed', value)
        error(custom_messages.ResourceNotFound('SuperPodId'))
    end
    log:operation(ctx:get_initiator(), 'Chassis', 'Set SuperPodId(%s) successfully', value)
end

-- 导入超节点内的计算节点编号
local function import_server_index(self, ctx, config_info)
    local value = config_info.ServerIndex.Value
    local super_pod_label_object = c_object('SuperPodLabel').collection:find({Id = 1})
    if not super_pod_label_object then
        log:operation(ctx:get_initiator(), 'Chassis', 'Set ServerIndex(%s) failed', value)
        error(custom_messages.ResourceNotFound('SuperPodLabel'))
    end
    local res = super_pod_label_object:update_server_index(value)
    if not res then
        log:operation(ctx:get_initiator(), 'Chassis', 'Set ServerIndex(%s) failed', value)
        error(custom_messages.ResourceNotFound('ServerIndex'))
    end
    log:operation(ctx:get_initiator(), 'Chassis', 'Set ServerIndex(%s) successfully', value)
end

-- 导入超节点规模（NPU卡数）
local function import_super_pod_size(self, ctx, config_info)
    local value = config_info.SuperPodSize.Value
    local super_pod_label_object = c_super_pod_label_object.collection:find({Id = 1})
    if not super_pod_label_object then
        log:operation(ctx:get_initiator(), 'Chassis', 'Set SuperPodSize(%s) failed', value)
        error(custom_messages.ResourceNotFound('SuperPodLabel'))
    end

    local res = super_pod_label_object:update_super_pod_size(value)
    if not res then
        log:operation(ctx:get_initiator(), 'Chassis', 'Set SuperPodSize(%s) failed', value)
        error(custom_messages.ResourceNotFound('SuperPodSize'))
    end
    log:operation(ctx:get_initiator(), 'Chassis', 'Set SuperPodSize(%s) successfully', value)
end

-- 导入节点模式
local function import_super_pod_enabled(self, ctx, config_info)
    local value = config_info.Value
    local super_pod_label_object = c_super_pod_label_object.collection:find({Id = 1})
    if not super_pod_label_object then
        log:operation(ctx:get_initiator(), 'Chassis', 'Set SuperPodEnabled(%s) failed', value)
        error(custom_messages.ResourceNotFound('SuperPodLabel'))
    end
    super_pod_label_object:update_super_pod_enabled(value)
end

local config_import_func_table = {
    ['SuperPodEnabled']            = { 
        config_func = import_super_pod_enabled 
    },
    ['SuperPodLabel']            = {
        ['SuperPodId'] = { config_func = import_super_pod_id },
        ['ServerIndex'] = { config_func = import_server_index },
        ['SuperPodSize'] = { config_func = import_super_pod_size }
    }
}

-- 配置导入
function business_config:on_import(ctx, business_data)
    local import_table
    for name, config_info in pairs(business_data) do
        import_table = config_import_func_table[name]
        if not import_table then
            log:error('Import object[%s] is invalid', name)
            goto continue
        end
        if import_table.config_func then
            import_table.config_func(self , ctx, config_info)
            goto continue
        end
        for prop, _ in pairs(config_info) do
            if not import_table[prop] then
                log:error('Import prop [%s] is invalid', prop)
                goto next
            end
            if import_table[prop].config_func then
                import_table[prop].config_func(self, ctx, config_info, import_table[prop].type)
            end
            ::next::
        end
        log:notice('Import object [%s] successfully', name)
        ::continue::
    end
end

-- 配置导出
function business_config:on_export(ctx)
    local super_pod_label_obj = {}
    local super_pod_label_object = c_object('SuperPodLabel').collection:find({Id = 1})
    local data = {}
    if super_pod_label_object == nil then
        log:error('SuperPodLabel is nil.')
        return data
    end
    super_pod_label_obj.SuperPodId = super_pod_label_object.SuperPodId
    super_pod_label_obj.ServerIndex = super_pod_label_object.ServerIndex
    super_pod_label_obj.SuperPodSize = super_pod_label_object.SuperPodSize
    data.SuperPodEnabled = super_pod_label_object.SuperPodEnabled
    data.SuperPodLabel = super_pod_label_obj
    log:notice('Export SuperPodLabel successfully')
    return data
end

return business_config
