-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local log = require 'mc.logging'
local bs_util = require 'util.base_util'
local mdb = require 'mc.mdb'
local prop_def = require "macros.property_def"
local json = require 'cjson'
require 'bios.json_types.SecureBootOptions'
local test_bios_common = require 'test_bios_common'
local lu = require 'luaunit'

local BOOT_OPTIONS_PATH = '/bmc/kepler/Systems/1/BootOptions'
local BOOT_OPTIONS_INTF = 'bmc.kepler.Systems.BootOptions'

local SECURE_BOOT_OPTIONS_PATH = '/bmc/kepler/Systems/1/SecureBootOptions'
local SECURE_BOOT_OPTIONS_INTF = 'bmc.kepler.Systems.SecureBootOptions'

local test_bios_certificate = {}

-- 测试导入Https证书并获取
function test_bios_certificate.test_boot_import_certificate(bus)
    log:info('================ test bios certificate start ================')
    local obj = mdb.get_object(bus, BOOT_OPTIONS_PATH, BOOT_OPTIONS_INTF)
    local cert_string = '-----BEGIN CERTIFICATE-----\n' ..
        'MIIDoTCCAomgAwIBAgIBAjANBgkqhkiG9w0BAQsFADBZMQswCQYDVQQGEwJDTjEP\n' ..
        'MA0GA1UECBMGU0hBTlhJMQ0wCwYDVQQHEwRYSUFOMQ8wDQYDVQQKEwZIdWF3ZWkx\n' ..
        'CzAJBgNVBAsTAklUMQwwCgYDVQQDEwNCTUMwIBcNMjAwODI5MDc1OTAwWhgPMjA1\n' ..
        'MDA4MjkwNzU5MDBaMFkxCzAJBgNVBAYTAkNOMQ8wDQYDVQQIEwZTSEFOWEkxDTAL\n' ..
        'BgNVBAcTBFhJQU4xDzANBgNVBAoTBkh1YXdlaTELMAkGA1UECxMCSVQxDDAKBgNV\n' ..
        'BAMTA0JNQzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALTut4k6ljew\n' ..
        '8OsEXleCSn7sU2WdcIBuH5BJLmKJ43jLCovFGszyeuv9/kIbaoBMDBYHtiDkbzUk\n' ..
        'xQz6Y0CF8tmFTfrF9yD8eAcTG8mGThpWspeyy2mfr8LWOu0Ny01HOkWsQ6kiXX1p\n' ..
        'Yv3nGVCs4q/eVIQUx+oWJP8jeynHFS7l+q2UuUpBbVggE3e0UT+Ua8v3cW3eoMze\n' ..
        'CGHqBaJxEooOKBBLsS014SqjiH2xGr1vgqX8VbigPmZEYAaLcYHungq2AvhaoEEO\n' ..
        'jaV9rbPrPObv36D/mXfE72ww4E9b4YCM3tti1qGCo6jtWV42U4BNP72yW4nldsb8\n' ..
        'LKJFXuuE4qUCAwEAAaNyMHAwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQUM35X\n' ..
        'GhBNPfoxqjE1Wv/FULUWO3wwCwYDVR0PBAQDAgEGMBEGCWCGSAGG+EIBAQQEAwIA\n' ..
        'BzAeBglghkgBhvhCAQ0EERYPeGNhIGNlcnRpZmljYXRlMA0GCSqGSIb3DQEBCwUA\n' ..
        'A4IBAQCW06TnfLdNxi2uUvL3UX4rRS0rxIazLg2BC9ez07YpqqKwLrOE8wzN17on\n' ..
        'xkCzEVcOioa7as6UsuiqvrRz/P7owtIk75XX1PXyex1f8GZwhoBXvtQZnyxBtKUK\n' ..
        'M/o7ftpcrXa6JTfcQbI46vWo44LZmMnKKGlxGCPjxrV6R7BvE87dSkTWNTC/3lnt\n' ..
        'qsn+U13VEklTvy9uzn1GbkqGnJPT7K0ZeLZxhmUEQHNhFuRGQr/grFXnUUnnzfAa\n' ..
        'dJDQoWK39rDRz5MV5Y7iOHYRiBepWrQ4cTYUfSsvEhuraf+jTC5D/Mqvqh3v5KwD\n' ..
        'd0UaV+oktUP+uKfUfHQcFuS7BZWF\n' ..
        '-----END CERTIFICATE-----'
    obj:ImportCertificate_PACKED(require'mc.context'.new(), cert_string, 'PEM')
    os.execute('cp ' .. prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.NEW_SECUREBOOT_FILE .. ' ' ..
                prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.CURRENT_SECUREBOOT_FILE)
    local get_resp = obj:GetCertificate_PACKED(require'mc.context'.new())
    local cert_json = json.decode(get_resp.CertInfo)
    assert(cert_json['Boot']['HttpsCert'][1]['ValidNotBefore'] == 'Aug 29 2020 UTC')
    assert(cert_json['Boot']['HttpsCert'][1]['Subject']['CommonName'] == 'BMC')
    log:info('================ test bios certificate end ================')
end

-- 测试导入证书吊销列表并获取
function test_bios_certificate.test_crl_import_certificate(bus)
    log:info('================ test bios crl start ================')
    local obj = mdb.get_object(bus, BOOT_OPTIONS_PATH, BOOT_OPTIONS_INTF)
    local cert_string = '-----BEGIN X509 CRL-----\n' ..
    'MIICIzCCAQsCAQEwDQYJKoZIhvcNAQELBQAwWTELMAkGA1UEBhMCQ04xDzANBgNV\n' ..
    'BAgTBlNIQU5YSTENMAsGA1UEBxMEWElBTjEPMA0GA1UEChMGSHVhd2VpMQswCQYD\n' ..
    'VQQLEwJJVDEMMAoGA1UEAxMDQk1DFw0yMTAxMDEwNzAxMDBaGA8yMDUxMDEwMTA3\n' ..
    'MDEwMFowbDAuAgEDFw0yMDA5MzAwODEzMDVaMBowGAYDVR0YBBEYDzIwMjAwOTMw\n' ..
    'MDgxMjAwWjA6AgEFFw0yMDA5MjQwNjM4NDRaMCYwGAYDVR0YBBEYDzIwMjAwOTI0\n' ..
    'MDYzODAwWjAKBgNVHRUEAwoBA6AOMAwwCgYDVR0UBAMCAQEwDQYJKoZIhvcNAQEL\n' ..
    'BQADggEBAHD10OqbOLPOmgLDIEHMduLrZJXoYo8altLDfhUzUkViupLjSgT0WEn0\n' ..
    'jJSwlnJ9XJ0Uc89eHHaKrJEkhdt88GnhHf+xWg0R6UZmJzH6VHe5jwB0g4Z9i0hr\n' ..
    'z2XcqioZ0WvPm1qdSIw6Ea/rEuEZRcXFL7XeB3GR3rz8BeRGcbkQ/r7v2tL4KwWF\n' ..
    'UMNNx0PGITz6qqsBwQk2gel+QLPzpv+VJTX+08atmFoI5hGcI8ji717jvi78i3CQ\n' ..
    'FKVuakYzc/evJaVwl3GNdwiGn+iIuxfM03q9ktVZdyDEUVQ3QUYZyvlt3tQJKlPj\n' ..
    'FQ5o/0VUaafqbDHAQIOlEqIeL4ciDGE=\n' ..
    '-----END X509 CRL-----'
    
    obj:ImportCrl_PACKED(require'mc.context'.new(), cert_string, 'PEM')
    os.execute('cp ' .. prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.NEW_SECUREBOOT_FILE .. ' ' ..
                prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.CURRENT_SECUREBOOT_FILE)
    local get_resp = obj:GetCertificate_PACKED(require'mc.context'.new())
    local cert_json = json.decode(get_resp.CertInfo)
    assert(cert_json['Boot']['HttpsCrl'][1]['CertificateType'] == 'PEM')
    log:info('================ test bios crl end ================')
end

-- 测试导入安全启动证书并获取
function test_bios_certificate.test_secure_boot_import_certificate(bus)
    log:info('================ test bios secureboot certificate start ================')
    local obj = mdb.get_object(bus, SECURE_BOOT_OPTIONS_PATH, SECURE_BOOT_OPTIONS_INTF)
    local cert_string = '-----BEGIN CERTIFICATE-----\n' ..
        'MIIDoTCCAomgAwIBAgIBAjANBgkqhkiG9w0BAQsFADBZMQswCQYDVQQGEwJDTjEP\n' ..
        'MA0GA1UECBMGU0hBTlhJMQ0wCwYDVQQHEwRYSUFOMQ8wDQYDVQQKEwZIdWF3ZWkx\n' ..
        'CzAJBgNVBAsTAklUMQwwCgYDVQQDEwNCTUMwIBcNMjAwODI5MDc1OTAwWhgPMjA1\n' ..
        'MDA4MjkwNzU5MDBaMFkxCzAJBgNVBAYTAkNOMQ8wDQYDVQQIEwZTSEFOWEkxDTAL\n' ..
        'BgNVBAcTBFhJQU4xDzANBgNVBAoTBkh1YXdlaTELMAkGA1UECxMCSVQxDDAKBgNV\n' ..
        'BAMTA0JNQzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALTut4k6ljew\n' ..
        '8OsEXleCSn7sU2WdcIBuH5BJLmKJ43jLCovFGszyeuv9/kIbaoBMDBYHtiDkbzUk\n' ..
        'xQz6Y0CF8tmFTfrF9yD8eAcTG8mGThpWspeyy2mfr8LWOu0Ny01HOkWsQ6kiXX1p\n' ..
        'Yv3nGVCs4q/eVIQUx+oWJP8jeynHFS7l+q2UuUpBbVggE3e0UT+Ua8v3cW3eoMze\n' ..
        'CGHqBaJxEooOKBBLsS014SqjiH2xGr1vgqX8VbigPmZEYAaLcYHungq2AvhaoEEO\n' ..
        'jaV9rbPrPObv36D/mXfE72ww4E9b4YCM3tti1qGCo6jtWV42U4BNP72yW4nldsb8\n' ..
        'LKJFXuuE4qUCAwEAAaNyMHAwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQUM35X\n' ..
        'GhBNPfoxqjE1Wv/FULUWO3wwCwYDVR0PBAQDAgEGMBEGCWCGSAGG+EIBAQQEAwIA\n' ..
        'BzAeBglghkgBhvhCAQ0EERYPeGNhIGNlcnRpZmljYXRlMA0GCSqGSIb3DQEBCwUA\n' ..
        'A4IBAQCW06TnfLdNxi2uUvL3UX4rRS0rxIazLg2BC9ez07YpqqKwLrOE8wzN17on\n' ..
        'xkCzEVcOioa7as6UsuiqvrRz/P7owtIk75XX1PXyex1f8GZwhoBXvtQZnyxBtKUK\n' ..
        'M/o7ftpcrXa6JTfcQbI46vWo44LZmMnKKGlxGCPjxrV6R7BvE87dSkTWNTC/3lnt\n' ..
        'qsn+U13VEklTvy9uzn1GbkqGnJPT7K0ZeLZxhmUEQHNhFuRGQr/grFXnUUnnzfAa\n' ..
        'dJDQoWK39rDRz5MV5Y7iOHYRiBepWrQ4cTYUfSsvEhuraf+jTC5D/Mqvqh3v5KwD\n' ..
        'd0UaV+oktUP+uKfUfHQcFuS7BZWF\n' ..
        '-----END CERTIFICATE-----'
    obj:ImportCertificate_PACKED(require'mc.context'.new(), 'dbx', cert_string, 'PEM')
    os.execute('cp ' .. prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.NEW_SECUREBOOT_FILE .. ' ' ..
                prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.CURRENT_SECUREBOOT_FILE)
    local get_resp = obj:GetCertificate_PACKED(require'mc.context'.new())
    local cert_json = json.decode(get_resp.CertInfo)
    assert(cert_json['SecureBoot']['dbx'][1]['ValidNotBefore'] == 'Aug 29 2020 UTC')
    assert(cert_json['SecureBoot']['dbx'][1]['Subject']['CommonName'] == 'BMC')
    log:info('================ test bios secureboot certificate end ================')
end

-- 测试重置Https证书和吊销列表并查看修改后的json文件
function test_bios_certificate.test_boot_reset_certificate(bus)
    log:info('================ test bios reset certificate start ================')
    local obj = mdb.get_object(bus, BOOT_OPTIONS_PATH, BOOT_OPTIONS_INTF)
    obj:ResetCertificate_PACKED(require'mc.context'.new(), 'DeleteAllKeys')
    local file_json = bs_util.get_file_json(prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.NEW_SECUREBOOT_FILE)
    assert(file_json['Boot']['Certificates'][1]['Operation'] == 'DeleteAllKeys')
    obj:ResetCrl_PACKED(require'mc.context'.new(), 'ResetAllKeysToDefault')
    local file_json = bs_util.get_file_json(prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.NEW_SECUREBOOT_FILE)
    assert(file_json['Boot']['Crl'][1]['Operation'] == 'ResetAllKeysToDefault')
    log:info('================ test bios reset certificate end ================')
end

-- 测试重置安全启动证书并查看修改后的json文件
function test_bios_certificate.test_secure_boot_reset_certificate(bus)
    log:info('================ test bios reset secureboot certificate start ================')
    local obj = mdb.get_object(bus, SECURE_BOOT_OPTIONS_PATH, SECURE_BOOT_OPTIONS_INTF)
    obj:ResetCertificate_PACKED(require'mc.context'.new(), 'db', 'DeleteAllKeys')
    local file_json = bs_util.get_file_json(prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.NEW_SECUREBOOT_FILE)
    assert(file_json['SecureBoot']['SecureBootDataBases']['db']['Certificates'][1]['Operation'] == 'DeleteAllKeys')
    obj:ResetCertificate_PACKED(require'mc.context'.new(), 'dbx', 'ResetAllKeysToDefault')
    local file_json = bs_util.get_file_json(prop_def.BIOS_CONFIG_PATH .. '/' .. prop_def.NEW_SECUREBOOT_FILE)
    assert(file_json['SecureBoot']['SecureBootDataBases']['dbx']['Certificates'][1]['Operation'] == 'ResetAllKeysToDefault')
    log:info('================ test bios reset secureboot certificate end ================')
end

local function send_ipmi(data)
    return test_bios_common.send_ipmi(data)
end

function test_bios_certificate.test_set_cert_assertion()
    local ok, _ = send_ipmi('0x30 0x92 0xdb 0x07 0x00 0x05 0x11 0x00 0x00 0x00')
    lu.assertEquals(ok, true)

    ok, _ = send_ipmi('0x30 0x92 0xdb 0x07 0x00 0x05 0x11 0x00 0x01 0x00')
    lu.assertEquals(ok, true)
end

return test_bios_certificate