-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local skynet = require 'skynet'
local log = require 'mc.logging'
local bios_factory = require 'factory.bios_factory'
local prop_def = require 'macros.property_def'
local msg = require 'messages.base'

local retirement = {}

-- 数据报废处理接口, 触发执行后直接返回
function retirement.handle(ctx)
    local bios_service = bios_factory.get_service("bios_service")
    if not bios_service then
        log:error("Get bios service object fail.")
        error(msg.InternalError())
    end
    -- 初始化资源树状态
    retirement.init_report(bios_service)

    --启动任务执行数据报废
    skynet.fork_once(function ()
        local ok = retirement.reset_bios_setting(ctx, bios_service)
        if not ok then
            log:error("Data wipe handle fail.")
        end
        -- 数据报废执行完成后，更新数据报废报告
        retirement.update_retirement(ok, bios_service)
    end)
end

-- 执行bios菜单数据销毁
function retirement.reset_bios_setting(ctx, bios_service)
    local ok, ret = pcall(bios_service.reset_bios, bios_service, ctx)
    if not ok then
        log:error("Fail to execute the reset bios interface(%s).", ret)
        return false
    end
    log:notice("BIOS configuration data wipe successfully.")
    return ret == prop_def.RESPONSE_OK
end

function retirement.init_report(bios_service)
    bios_service:set_prop("Progress", 0)
    bios_service:set_prop("State", "Idle")
end

function retirement.update_retirement(ret, bios_service)
    bios_service:set_prop("Media", "Flash")
    bios_service:set_prop("Source", "Huawei")
    bios_service:set_prop("Description", "Clear")
    bios_service:set_prop("Method", "BlockErase")
    bios_service:set_prop("Result", ret and "Successful" or "Failed")
    bios_service:set_prop("Progress", 100)
    bios_service:set_prop("State", "Finish")
end

local report = {
    Manufacturer = "NA",
    Model = "NA",
    SerialNumber = "NA",
    Media = "Flash",
    Source = "Huawei",
    Description = "Clear",
    Method = "BlockErase",
    Result = nil
}

-- 返回数据报废报告
function retirement.get_report(obj, ctx)
    report.Result = obj.Result
    return report
end

return retirement


