#!/bin/bash

# =================================================================
#  Multi-User/Multi-Project Nginx Control Script
# =================================================================

# 检查参数数量
if [ "$#" -ne 2 ]; then
    echo "Usage: $0 {start|stop|reload|status} <bmc_studio server path>"
    exit 1
fi

ACTION="$1"
SERVER_DIR="$2"

# --- 路径定义 ---
# 所有路径都使用大写命名并添加注释说明

# 用户主目录下的配置和日志目录
USER_CONFIG_DIR="$HOME/.bmc_studio/config" # 生成的配置文件存放位置
USER_LOG_DIR="$HOME/.bmc_studio/logs"      # 日志和PID文件存放位置

# 项目相关路径
PROJECT_CONFIG_TEMPLATE="$SERVER_DIR/config/bmc_studio.conf"  # 项目配置模板
GENERATED_CONFIG="$USER_CONFIG_DIR/bmc_studio.generated.conf" # 最终生成的配置文件
PID_FILE="$USER_LOG_DIR/nginx.pid"                            # Nginx PID文件位置

# 检查项目目录是否存在
if [ ! -d "$SERVER_DIR" ]; then
    echo "Error: BMC Studio directory '$SERVER_DIR' not found."
    exit 1
fi

# 转换为绝对路径
SERVER_DIR=$(cd "$SERVER_DIR" && pwd)

# 检查 Nginx 可执行文件
NGINX_BIN=$(which nginx)
if [ -z "$NGINX_BIN" ]; then
    echo "Error: nginx executable not found in PATH."
    exit 1
fi

# --- 动态端口计算 ---
USER_ID=$(id -u)
FRONTEND_PORT=$((10000 + USER_ID % 10000))
BACKEND_PORT=$((8000 + USER_ID % 10000))
MAX_PORT_ATTEMPTS=100 

# --- 函数定义 ---

# 查找可用端口
# @param $1: 起始探测的基础端口号
# @return: 成功时，标准输出一个可用的端口号；失败时，返回非0值
find_available_port() {
    local base_port="$1"
    local warned=false  # 占用警告标志
    
    for i in $(seq 0 $((MAX_PORT_ATTEMPTS - 1))); do
        local current_port=$((base_port + i))
        
        if ! ss -lnt | grep -q ":${current_port}\b"; then
            echo "$current_port"  # 返回可用端口
            return 0
        elif ! $warned; then
            # 首次发现占用时提示
            echo "Port ${base_port} is occupied, scanning next ${MAX_PORT_ATTEMPTS} ports..." >&2
            warned=true
        fi
    done
    
    # 所有端口均被占用
    echo "All ports in range ${base_port}-$((base_port + MAX_PORT_ATTEMPTS -1)) are occupied!" >&2
    return 1
}


# 生成配置文件
generate_config() {
    local frontend_port="$1"
    local backend_port="$2"

    # 检查配置文件是否存在
    if [ -f "$GENERATED_CONFIG" ]; then
        echo "Config file already exists at '$GENERATED_CONFIG'. Overwriting...." >&2
    fi

    # 检查模版文件是否存在
    if [ ! -f "$PROJECT_CONFIG_TEMPLATE" ]; then
        echo "Error: Config template not found at '$PROJECT_CONFIG_TEMPLATE'" >&2
        return 1
    fi

    # 确保目录存在
    mkdir -p "$USER_CONFIG_DIR"
    mkdir -p "$USER_LOG_DIR"

    echo "Generating config with frontend:$frontend_port backend:$backend_port..." >&2

    # 替换模板中的变量
    sed -e "s|##PROJECT_PATH##|$SERVER_DIR|g" \
        -e "s|##LOG_PID_PATH##|$USER_LOG_DIR|g" \
        -e "s/##FRONTEND_PORT##/$frontend_port/g" \
        -e "s/##BACKEND_PORT##/$backend_port/g" \
        "$PROJECT_CONFIG_TEMPLATE" >"$GENERATED_CONFIG"
}

# 获取运行中的前端端口
get_frontend_port() {
    # 检查PID文件是否存在
    if [ ! -f "$PID_FILE" ]; then
        return 1
    fi

    # 检查进程是否在运行
    if ! ps -p "$(cat "$PID_FILE")" >/dev/null; then
        return 1
    fi

    # 检查配置文件是否存在
    if [ ! -f "$GENERATED_CONFIG" ]; then
        echo "Warning: Nginx process is running, but config file is missing!" >&2
        return 1
    fi

    # 从配置文件中提取端口
    grep 'listen' "$GENERATED_CONFIG" | awk '{print $2}' | tr -d ';'
}

# 获取运行中的后端端口
get_backend_port() {
    # 检查PID文件是否存在
    if [ ! -f "$PID_FILE" ]; then
        return 1
    fi

    # 检查进程是否在运行
    if ! ps -p "$(cat "$PID_FILE")" >/dev/null; then
        return 1
    fi

    # 检查配置文件是否存在
    if [ ! -f "$GENERATED_CONFIG" ]; then
        echo "Warning: Nginx process is running, but config file is missing!" >&2
        return 1
    fi

    # 从配置文件中提取端口
    grep 'proxy_pass' "$GENERATED_CONFIG" |
        awk '{print $2}' |
        tr -d ';' |
        awk -F':' '{print $NF}' |
        uniq
}

# 检查状态
check_status() {
    if RUNNING_PORT=$(get_frontend_port); then
        PID=$(cat "$PID_FILE")
        echo "Nginx is RUNNING for project '$SERVER_DIR'. PID: $PID, Port: $RUNNING_PORT" >&2
    else
        echo "Nginx is STOPPED for project '$SERVER_DIR'." >&2
    fi
}

# 启动服务
start() {
    if RUNNING_PORT=$(get_frontend_port); then
        echo "Nginx is already running on port $RUNNING_PORT." >&2
        echo "$RUNNING_PORT"
        exit 0
    fi
    local available_frontend_port=$(find_available_port "$FRONTEND_PORT")
    local available_backend_port=$(find_available_port "$BACKEND_PORT")
    generate_config "$available_frontend_port" "$available_backend_port"

    echo "Starting Nginx for project '$SERVER_DIR'..." >&2
    "$NGINX_BIN" -p "$SERVER_DIR" -c "$GENERATED_CONFIG"

    sleep 0.5

    if ! get_frontend_port >/dev/null; then
        echo "Error: Nginx failed to start. Check logs at $USER_LOG_DIR/error.log" >&2
        exit 1
    fi

    echo "Nginx started successfully." >&2
    echo "$FRONTEND_PORT"
}

# 停止服务
stop() {
    if ! get_frontend_port >/dev/null; then
        echo "Nginx is not running for this project." >&2
        exit 0
    fi

    PID=$(cat "$PID_FILE")
    echo "Stopping Nginx (PID: $PID)..." >&2
    kill "$PID"

    while ps -p "$PID" >/dev/null; do
        sleep 0.1
    done

    echo "Nginx stopped." >&2
}

reload() {
    if ! RUNNING_FRONTEND_PORT=$(get_frontend_port); then
        echo "Nginx is not running. Use 'start' instead." >&2
        exit 1
    fi

    # 获取当前运行的后端端口
    if ! RUNNING_BACKEND_PORT=$(get_backend_port); then
        echo "Error: Failed to get backend port." >&2
        exit 1
    fi

    PID=$(cat "$PID_FILE")
    echo "Reloading Nginx configuration for PID: $PID..." >&2

    # 传递两个端口参数
    generate_config "$RUNNING_FRONTEND_PORT" "$RUNNING_BACKEND_PORT"
    kill -HUP "$PID"

    echo "Configuration reloaded." >&2
}

# --- 主脚本 ---
case "$ACTION" in
    start) start ;;
    stop) stop ;;
    reload) reload ;;
    status) check_status ;;
    get-frontend-port)
        if RUNNING_PORT=$(get_frontend_port); then
            echo "$RUNNING_PORT"
        else
            echo "Error: Nginx is not running for this project." >&2
            exit 1
        fi
        ;;
    get-backend-port)
        if RUNNING_PORT=$(get_backend_port); then
            echo "$RUNNING_PORT"
        else
            echo "Error: Nginx is not running for this project." >&2
            exit 1
        fi
        ;;
    *)
        echo "Error: Invalid action '$ACTION'." >&2
        echo "Usage: $0 {start|stop|reload|status|get-port} /path/to/bmc studio" >&2
        exit 1
        ;;
esac

exit 0
