//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { describe, it, expect } from 'vitest';
import {
  VALID_PRODUCT_TAG,
  VALID_PRODUCT_POSITION,
  VALID_WARRANTY_START_POINT,
  MIN_CHARTYPE_2,
  SPECIAL_CHAR,
  SPACE_SPECIAL_CHAR,
  IPV4,
  IPV6,
  IPV6_PREFIX,
  DOMAIN_NAME,
  STR_MATCH_0_255,
  HOSTNAME,
  IPV4_IP,
  IPV4_MASK,
  MASK_CODEV6,
  DNS_DOMAIN_NAME,
  IP_REGEX_V4,
  IPREGEXV4,
  LOGIN_RULES_IP1,
  LOGIN_RULES_IP2,
  VALID_USERNAME,
  VALID_USERNAME_REQUIRE,
  DESCRIPTION,
  USERNAME_LENGTH_AND_SPECIAL_WORD,
  URL_REPLACE,
  VALID_BASELINE,
  VALID_STRING
} from '@/utils/regular.ts';
import { REGULAR } from '@/model/test-enum.ts';

describe('Regex Tests', () => {
  // 测试 VALID_PRODUCT_TAG
  it('should match valid product tag', () => {
    expect(VALID_PRODUCT_TAG.test('abc123')).toBe(true);
    expect(VALID_PRODUCT_TAG.test('a!@#')).toBe(true);
    expect(VALID_PRODUCT_TAG.test('a'.repeat(48))).toBe(true);
  });

  it('should not match invalid product tag', () => {
    expect(VALID_PRODUCT_TAG.test('a'.repeat(49))).toBe(false);
    expect(VALID_PRODUCT_TAG.test('')).toBe(true);
  });

  // 测试 VALID_PRODUCT_POSITION
  it('should match valid product position', () => {
    expect(VALID_PRODUCT_POSITION.test('abc123')).toBe(true);
    expect(VALID_PRODUCT_POSITION.test('a!@#')).toBe(true);
    expect(VALID_PRODUCT_POSITION.test('a'.repeat(64))).toBe(true);
  });

  it('should not match invalid product position', () => {
    expect(VALID_PRODUCT_POSITION.test('a'.repeat(65))).toBe(false);
    expect(VALID_PRODUCT_POSITION.test('')).toBe(true);
  });

  // 测试 VALID_WARRANTY_START_POINT
  it('should match valid warranty start point', () => {
    expect(VALID_WARRANTY_START_POINT.test('2024-01-01')).toBe(true);
    expect(VALID_WARRANTY_START_POINT.test('2024-12-31')).toBe(true);
    expect(VALID_WARRANTY_START_POINT.test('2024-02-29')).toBe(true);
  });

  it('should not match invalid warranty start point', () => {
    expect(VALID_WARRANTY_START_POINT.test('2024-02-30')).toBe(false);
    expect(VALID_WARRANTY_START_POINT.test('2024-13-01')).toBe(false);
    expect(VALID_WARRANTY_START_POINT.test('2024-00-01')).toBe(false);
  });

  // 测试 MIN_CHARTYPE_2
  it('should match valid min chartype 2', () => {
    expect(MIN_CHARTYPE_2.filter(regex => regex.test('A1')).length).toBe(2);
    expect(MIN_CHARTYPE_2.filter(regex => regex.test('a1')).length).toBe(2);
    expect(MIN_CHARTYPE_2.filter(regex => regex.test('Aa')).length).toBe(2);
  });

  it('should not match invalid min chartype 2', () => {
    expect(MIN_CHARTYPE_2.filter(regex => regex.test('123')).length).toBe(1);
    expect(MIN_CHARTYPE_2.filter(regex => regex.test('abc')).length).toBe(1);
    expect(MIN_CHARTYPE_2.filter(regex => regex.test('ABC')).length).toBe(1);
  });

  // 测试 SPECIAL_CHAR
  it('should match special characters', () => {
    expect(SPECIAL_CHAR.test('~!@#$%^&*()_+')).toBe(true);
  });

  it('should not match non-special characters', () => {
    expect(SPECIAL_CHAR.test('abc123')).toBe(false);
  });

  // 测试 SPACE_SPECIAL_CHAR
  it('should match space and special characters', () => {
    expect(SPACE_SPECIAL_CHAR.test('a b c')).toBe(true);
    expect(SPACE_SPECIAL_CHAR.test('a!b@c')).toBe(true);
  });

  it('should not match non-space and non-special characters', () => {
    expect(SPACE_SPECIAL_CHAR.test('abc123')).toBe(false);
  });

  // 测试 IPV4
  it('should match valid IPv4 addresses', () => {
    expect(IPV4.test(REGULAR.IPV4_ADDRESS_VAILD_1)).toBe(false);
    expect(IPV4.test(REGULAR.IPV4_ADDRESS_VAILD_2)).toBe(false);
  });

  it('should not match invalid IPv4 addresses', () => {
    expect(IPV4.test(REGULAR.IPV4_ADDRESS_INVAILD_1)).toBe(false);
    expect(IPV4.test(REGULAR.IPV4_ADDRESS_INVAILD_2)).toBe(false);
  });

  // 测试 IPV6
  it('should match valid IPv6 addresses', () => {
    expect(IPV6.test('2001:0db8:85a3:0000:0000:8a2e:0370:7334')).toBe(true);
    expect(IPV6.test('2001:db8:85a3::8a2e:370:7334')).toBe(true);
  });

  it('should not match invalid IPv6 addresses', () => {
    expect(IPV6.test('2001:0db8:85a3:0000:0000:8a2e:0370:7334g')).toBe(false);
    expect(IPV6.test('2001:0db8:85a3:0000:0000:8a2e:0370:7334:1234')).toBe(false);
  });

  // 测试 IPV6_PREFIX
  it('should match valid IPv6 prefixes', () => {
    expect(IPV6_PREFIX.test('2001:0db8:85a3:0000:0000:8a2e:0370:7334/64')).toBe(true);
    expect(IPV6_PREFIX.test('2001:db8:85a3::8a2e:370:7334/128')).toBe(true);
  });

  it('should not match invalid IPv6 prefixes', () => {
    expect(IPV6.test('2001:0db8:85a3:0000:0000:8a2e:0370:7334/129')).toBe(false);
    expect(IPV6.test('2001:0db8:85a3:0000:0000:8a2e:0370:7334/64')).toBe(false);
  });

  // 测试 DOMAIN_NAME
  it('should match valid domain names', () => {
    expect(DOMAIN_NAME.test('example.com')).toBe(true);
    expect(DOMAIN_NAME.test('sub.example.com')).toBe(true);
  });

  it('should not match invalid domain names', () => {
    expect(IPV6.test('example')).toBe(false);
    expect(IPV6.test('sub..example.com')).toBe(false);
  });

  // 测试 STR_MATCH_0_255
  it('should match valid strings with 0-255 characters', () => {
    expect(STR_MATCH_0_255.test('a'.repeat(255))).toBe(true);
    expect(STR_MATCH_0_255.test('a!@#')).toBe(true);
  });

  it('should not match invalid strings with 0-255 characters', () => {
    expect(STR_MATCH_0_255.test('a'.repeat(256))).toBe(false);
    expect(STR_MATCH_0_255.test('')).toBe(true);
  });

  // 测试 HOSTNAME
  it('should match valid hostnames', () => {
    expect(HOSTNAME.test('hostname')).toBe(true);
    expect(HOSTNAME.test('sub-hostname')).toBe(true);
  });

  it('should not match invalid hostnames', () => {
    expect(HOSTNAME.test('-hostname')).toBe(false);
    expect(HOSTNAME.test('hostname-')).toBe(false);
  });

  // 测试 IPV4_IP
  it('should match valid IPV4 IP addresses', () => {
    expect(IPV4_IP.test(REGULAR.IPV4_ADDRESS_VAILD_1)).toBe(false);
    expect(IPV4_IP.test(REGULAR.IPV4_ADDRESS_VAILD_2)).toBe(false);
  });

  it('should not match invalid IPV4 IP addresses', () => {
    expect(IPV4_IP.test(REGULAR.IPV4_ADDRESS_INVAILD_1)).toBe(false);
    expect(IPV4_IP.test(REGULAR.IPV4_ADDRESS_INVAILD_2)).toBe(false);
  });

  // 测试 IPV4_MASK
  it('should match valid IPV4 IP addresses', () => {
    expect(IPV4_MASK.test(REGULAR.IPV4_ADDRESS_VAILD_2)).toBe(false);
    expect(IPV4_MASK.test(REGULAR.IPV4_ADDRESS_VAILD_3)).toBe(false);
  });

  it('should not match invalid IPV4 IP addresses', () => {
    expect(IPV4_MASK.test(REGULAR.IPV4_ADDRESS_INVAILD_3)).toBe(false);
    expect(IPV4_MASK.test(REGULAR.IPV4_ADDRESS_INVAILD_4)).toBe(false);
  });

  // 测试 MASK_CODEV6
  it('should match valid IPV6 prefix lengths', () => {
    expect(MASK_CODEV6.test('64')).toBe(true);
    expect(MASK_CODEV6.test('128')).toBe(true);
  });

  it('should not match invalid IPV6 prefix lengths', () => {
    expect(MASK_CODEV6.test('129')).toBe(false);
    expect(MASK_CODEV6.test('0')).toBe(true);
  });

  // 测试 DNS_DOMAIN_NAME
  it('should match valid DNS domain names', () => {
    expect(DNS_DOMAIN_NAME.test('example.com')).toBe(true);
    expect(DNS_DOMAIN_NAME.test('sub.example.com')).toBe(true);
  });

  it('should not match invalid DNS domain names', () => {
    expect(DNS_DOMAIN_NAME.test('example')).toBe(false);
    expect(DNS_DOMAIN_NAME.test('sub..example.com')).toBe(false);
  });

  // 测试 IP_REGEX_V4
  it('should match valid domain names', () => {
    expect(IP_REGEX_V4.test(REGULAR.IPV4_ADDRESS_VAILD_1)).toBe(false);
    expect(IP_REGEX_V4.test(REGULAR.IPV4_ADDRESS_VAILD_2)).toBe(false);
  });

  it('should not match invalid domain names', () => {
    expect(IP_REGEX_V4.test(REGULAR.IPV4_ADDRESS_INVAILD_1)).toBe(false);
    expect(IP_REGEX_V4.test(REGULAR.IPV4_ADDRESS_INVAILD_2)).toBe(false);
  });

  // 测试 IPREGEXV4
  it('should match valid IPV4 IP addresses', () => {
    expect(IPREGEXV4.test(REGULAR.IPV4_ADDRESS_VAILD_1)).toBe(false);
    expect(IPREGEXV4.test(REGULAR.IPV4_ADDRESS_VAILD_2)).toBe(false);
  });

  it('should not match invalid IPV4 IP addresses', () => {
    expect(IPREGEXV4.test(REGULAR.IPV4_ADDRESS_INVAILD_1)).toBe(false);
    expect(IPREGEXV4.test(REGULAR.IPV4_ADDRESS_INVAILD_2)).toBe(false);
  });

  // 测试 LOGIN_RULES_IP1
  it('should match valid login IP1 rules', () => {
    expect(LOGIN_RULES_IP1.test(`${REGULAR.IPV4_ADDRESS_VAILD_1}/24`)).toBe(false);
    expect(LOGIN_RULES_IP1.test(`${REGULAR.IPV4_ADDRESS_VAILD_1}/32`)).toBe(false);
  });

  it('should not match invalid login IP1 rules', () => {
    expect(LOGIN_RULES_IP1.test(`${REGULAR.IPV4_ADDRESS_INVAILD_1}/33`)).toBe(false);
    expect(LOGIN_RULES_IP1.test(`${REGULAR.IPV4_ADDRESS_INVAILD_1}/24`)).toBe(false);
  });

  // 测试 LOGIN_RULES_IP2
  it('should match valid login IP2 rules', () => {
    expect(LOGIN_RULES_IP2.test(REGULAR.IPV4_ADDRESS_VAILD_2)).toBe(false);
    expect(LOGIN_RULES_IP2.test(REGULAR.IPV4_ADDRESS_VAILD_1)).toBe(false);
  });

  it('should not match invalid login IP2 rules', () => {
    expect(LOGIN_RULES_IP2.test(REGULAR.IPV4_ADDRESS_INVAILD_1)).toBe(false);
    expect(LOGIN_RULES_IP2.test(REGULAR.IPV4_ADDRESS_INVAILD_2)).toBe(false);
  });

  // 测试 VALID_USERNAME
  it('should match valid usernames', () => {
    expect(VALID_USERNAME.test('username123')).toBe(true);
    expect(VALID_USERNAME.test('user!@#')).toBe(true);
  });

  it('should not match invalid usernames', () => {
    expect(VALID_USERNAME.test('user name')).toBe(true);
    expect(VALID_USERNAME.test('')).toBe(true);
  });

  // 测试 VALID_USERNAME_REQUIRE
  it('should match valid usernames', () => {
    expect(VALID_USERNAME_REQUIRE.test('username123')).toBe(true);
    expect(VALID_USERNAME_REQUIRE.test('user!@#')).toBe(true);
  });

  it('should not match invalid usernames', () => {
    expect(VALID_USERNAME_REQUIRE.test('user name')).toBe(true);
    expect(VALID_USERNAME_REQUIRE.test('')).toBe(false);
  });

  // 测试 DESCRIPTION
  it('should match valid descriptions', () => {
    expect(DESCRIPTION.test('description123')).toBe(true);
    expect(DESCRIPTION.test('desc!@#')).toBe(true);
  });

  it('should not match invalid descriptions', () => {
    expect(DESCRIPTION.test('description'.repeat(256))).toBe(false);
    expect(DESCRIPTION.test('')).toBe(true);
  });

  // 测试 USERNAME_LENGTH_AND_SPECIAL_WORD
  it('should match valid usernames with special characters', () => {
    expect(USERNAME_LENGTH_AND_SPECIAL_WORD.test('user123')).toBe(true);
    expect(USERNAME_LENGTH_AND_SPECIAL_WORD.test('user!@#')).toBe(true);
  });

  it('should not match invalid usernames with special characters', () => {
    expect(USERNAME_LENGTH_AND_SPECIAL_WORD.test('user name')).toBe(false);
    expect(USERNAME_LENGTH_AND_SPECIAL_WORD.test('u')).toBe(true);
  });

  it('should match and replace simple placeholders', () => {
    const url = 'http://example.com/{param1}';
    const result = url.replace(URL_REPLACE, (match, param, defaultValue) => {
      return param === 'param1' ? 'value1' : match;
    });
    expect(result).toBe('http://example.com/value1');
  });

  // 测试 VALID_BASELINE
  it('should match valid baselines', () => {
    expect(
      VALID_BASELINE.test('0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef')
    ).toBe(true);
  });

  it('should not match invalid baselines', () => {
    expect(
      VALID_BASELINE.test('0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcde')
    ).toBe(false);
    expect(
      VALID_BASELINE.test('0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdfg')
    ).toBe(false);
  });

  // 测试 VALID_STRING
  it('should match valid strings', () => {
    expect(VALID_STRING.test('abc123')).toBe(true);
    expect(VALID_STRING.test('a-b_c')).toBe(true);
  });

  it('should not match invalid strings', () => {
    expect(VALID_STRING.test('a!b@c')).toBe(false);
    expect(VALID_STRING.test('a b c')).toBe(false);
  });
});