//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { setActivePinia, createPinia } from 'pinia';
// 导入测试模块
import {
  setLangData,
  computeWebLanguage,
  traduction,
  saveI18nInstance,
  getErrorTranslate
} from '@/utils/language.ts';

// 创建 mock 函数
const setStoreDataMock = vi.fn();
const saveStoreToLocalMock = vi.fn();

// Mock 模块
vi.mock('@/utils/common-service', () => ({
  saveStoreToLocal: saveStoreToLocalMock,
}));

vi.mock('@/utils/composition', () => ({
  setStoreData: setStoreDataMock,
}));

// Mock localStorage
const localStorageMock = {
  store: {},
  getItem: vi.fn(key => localStorageMock.store[key] || null),
  setItem: vi.fn((key, value) => {
    localStorageMock.store[key] = value;
  }),
  removeItem: vi.fn(key => {
    delete localStorageMock.store[key];
  }),
  clear: vi.fn(() => {
    localStorageMock.store = {};
  }),
};

// Mock i18n instance
const i18nInstanceMock = {
  global: {
    t: vi.fn((key, arg) => {
      if (arg) {
        return `${key}-${arg}`;
      }
      return key;
    }),
  },
};

// Mock document.querySelector 返回一个有效的元素
const mockElement = { className: '' };
const querySelectorMock = vi.fn(() => mockElement);

beforeEach(() => {
  vi.clearAllMocks();
  localStorageMock.clear();
  mockElement.className = '';

  global.localStorage = localStorageMock;
  global.navigator = { language: 'en-US' };
  global.document = { querySelector: querySelectorMock };
});

describe('i18n Tests', () => {
  it('should compute web language correctly', () => {
    localStorageMock.getItem.mockImplementation(key => {
      if (key === 'kvmLocale') {
        return 'zh';
      }
      if (key === 'loct') {
        return JSON.stringify({ locale: 'fr' });
      }
      return null;
    });

    const langSet = ['en', 'zh', 'fr'];
    expect(computeWebLanguage(langSet)).toBe('zh');
    expect(localStorageMock.removeItem).toHaveBeenCalledWith('kvmLocale');

    localStorageMock.getItem.mockImplementation(key => {
      if (key === 'loct') {
        return JSON.stringify({ locale: 'fr' });
      }
      return null;
    });
    expect(computeWebLanguage(langSet)).toBe('fr');

    localStorageMock.getItem.mockImplementation(() => null);
    expect(computeWebLanguage(langSet)).toBe('en');
  });

  it('should save i18n instance correctly', () => {
    saveI18nInstance(i18nInstanceMock);
    expect(i18nInstanceMock.global.t('test')).toBe('test');
  });

  it('should translate strings correctly', () => {
    saveI18nInstance(i18nInstanceMock);
    expect(traduction('test')).toBe('test');
    expect(traduction('test', 123)).toBe('test-123');
  });

  it('should get error translate correctly', () => {
    saveI18nInstance(i18nInstanceMock);
    expect(getErrorTranslate('error1')).toBe('COMMON_FAILED');
    expect(getErrorTranslate('error2')).toBe('COMMON_FAILED');
    expect(getErrorTranslate('error3')).toBe('COMMON_FAILED');
    expect(getErrorTranslate('error4')).toBe('COMMON_FAILED');
  });
});

describe('setLangData', () => {
  // 创建并激活 Pinia 实例
  setActivePinia(createPinia());

  it('应该设置正确的 CSS class', () => {
    localStorageMock.setItem('kvmLocale', 'zh');
    setLangData(['en', 'zh', 'ja']);
    // 验证 document.querySelector 被调用
    expect(querySelectorMock).toHaveBeenCalledWith('#app');
    // 验证 className 被设置
    expect(mockElement.className).toBe('en');
  });
});
