//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { describe, it, expect, vi, beforeEach } from 'vitest';

// Mock 依赖
vi.mock('./language', () => ({
  traduction: vi.fn(key => (key === 'COMMON_SUCCESS' ? '成功' : '失败')),
}));

// 创建 store mock
const mockStore = {
  state: {
    loct: {
      setLoctState: vi.fn(),
      resetLoctState: vi.fn(),
      loctData: 'loctValue',
    },
    event: {
      setEventState: vi.fn(),
      resetLoctState: vi.fn(),
      eventData: 'eventValue',
    },
    glob: {
      setGlobState: vi.fn(),
      resetLoctState: vi.fn(),
      globData: 'globValue',
    },
    theme: {
      setThemeState: vi.fn(),
      resetLoctState: vi.fn(),
      themeData: 'themeValue',
      initThemeState: vi.fn(),
    },
    menu: {
      setChildMenu: vi.fn(),
      resetLoctState: vi.fn(),
      menuData: 'menuValue',
      mainMenu: [{ id: 'home', name: '首页' }],
    },
  },
};

vi.mock('@/stores', () => ({
  default: vi.fn(() => mockStore),
}));

describe('Store 工具函数基础测试', () => {
  beforeEach(() => {
    vi.clearAllMocks();
  });

  describe('getStoreData', () => {
    it('应该获取 loct 模块的数据', async () => {
      const module = await import('@/utils/composition.ts');
      const result = module.getStoreData('loct', 'loctData');

      expect(result).toBe('loctValue');
    });

    it('应该获取 event 模块的数据', async () => {
      const module = await import('@/utils/composition.ts');

      const result = module.getStoreData('event', 'eventData');
      expect(result).toBe('eventValue');
    });

    it('应该获取 glob 模块的数据', async () => {
      const module = await import('@/utils/composition.ts');
      const result = module.getStoreData('glob', 'globData');

      expect(result).toBe('globValue');
    });

    it('应该返回 undefined 对于不存在的键', async () => {
      const module = await import('@/utils/composition.ts');
      const result = module.getStoreData('loct', 'nonexistent');

      expect(result).toBeUndefined();
    });
  });

  describe('setStoreData', () => {
    it('应该调用 loct 模块的 setLoctState', async () => {
      const module = await import('@/utils/composition.ts');
      module.setStoreData('loct', 'key', 'value');

      expect(mockStore.state.loct.setLoctState).toHaveBeenCalledWith({
        type: 'key',
        value: 'value',
      });
    });

    it('应该调用 event 模块的 setEventState', async () => {
      const module = await import('@/utils/composition.ts');
      module.setStoreData('event', 'testKey', 'testValue');

      expect(mockStore.state.event.setEventState).toHaveBeenCalledWith({
        type: 'testKey',
        value: 'testValue',
      });
    });

    it('应该调用 glob 模块的 setGlobState', async () => {
      const module = await import('@/utils/composition.ts');
      module.setStoreData('glob', 'testKey', 'testValue');

      expect(mockStore.state.glob.setGlobState).toHaveBeenCalledWith({
        type: 'testKey',
        value: 'testValue',
      });
    });

    it('应该调用 theme 模块的 setThemeState', async () => {
      const module = await import('@/utils/composition.ts');
      module.setStoreData('theme', 'testKey', 'testValue');

      expect(mockStore.state.theme.setThemeState).toHaveBeenCalledWith({
        type: 'testKey',
        value: 'testValue',
      });
    });
  });

  describe('消息显示函数', () => {
    it('showSuccessMessage 应该显示成功消息', async () => {
      const module = await import('@/utils/composition.ts');
      module.showSuccessMessage('自定义成功消息');

      expect(mockStore.state.event.setEventState).toHaveBeenCalledWith({
        type: 'alertMessage',
        value: {
          type: 'success',
          message: '自定义成功消息',
        },
      });
    });

    it('showSuccessMessage 应该使用默认成功消息', async () => {
      const module = await import('@/utils/composition.ts');
      module.showSuccessMessage('操作成功');

      expect(mockStore.state.event.setEventState).toHaveBeenCalledWith({
        type: 'alertMessage',
        value: {
          type: 'success',
          message: '操作成功',
        },
      });
    });

    it('showWarningMessage 应该显示警告消息', async () => {
      const module = await import('@/utils/composition.ts');
      module.showWarningMessage('警告消息');

      expect(mockStore.state.event.setEventState).toHaveBeenCalledWith({
        type: 'alertMessage',
        value: {
          type: 'warning',
          message: '警告消息',
        },
      });
    });

    it('showFailedMessage 应该显示失败消息', async () => {
      const module = await import('@/utils/composition.ts');
      module.showFailedMessage('自定义失败消息');

      expect(mockStore.state.event.setEventState).toHaveBeenCalledWith({
        type: 'alertMessage',
        value: {
          type: 'error',
          message: '自定义失败消息',
        },
      });
    });

    it('showFailedMessage 应该使用默认失败消息', async () => {
      const module = await import('@/utils/composition.ts');
      module.showFailedMessage('操作失败');

      expect(mockStore.state.event.setEventState).toHaveBeenCalledWith({
        type: 'alertMessage',
        value: {
          type: 'error',
          message: '操作失败',
        },
      });
    });
  });

  describe('findMainMenu', () => {
    it('应该找到存在的菜单项', async () => {
      const module = await import('@/utils/composition.ts');
      const result = module.findMainMenu('home');
      expect(result).toEqual({ id: 'home', name: '首页' });
    });
  });

  describe('setMenuStoreData', () => {
    it('应该调用 setChildMenu', async () => {
      const module = await import('@/utils/composition.ts');
      module.setMenuStoreData('mainId', ['sub1', 'sub2'], true);

      expect(mockStore.state.menu.setChildMenu).toHaveBeenCalledWith({
        mainId: 'mainId',
        subIds: ['sub1', 'sub2'],
        supported: true,
      });
    });
  });

  describe('setThemeStoreData', () => {
    it('应该调用 initThemeState', async () => {
      const module = await import('@/utils/composition.ts');
      const themeData = { color: 'blue', size: 'large' };
      module.setThemeStoreData(themeData);

      expect(mockStore.state.theme.initThemeState).toHaveBeenCalledWith(themeData);
    });
  });

  describe('loading', () => {
    it('应该设置加载状态为 true', async () => {
      const module = await import('@/utils/composition.ts');
      module.loading(true);

      expect(mockStore.state.glob.setGlobState).toHaveBeenCalledWith({
        type: 'isLoading',
        value: true,
      });
    });

    it('应该设置加载状态为 false', async () => {
      const module = await import('@/utils/composition.ts');
      module.loading(false);

      expect(mockStore.state.glob.setGlobState).toHaveBeenCalledWith({
        type: 'isLoading',
        value: false,
      });
    });
  });

  describe('resetLoctData', () => {
    it('应该调用 resetLoctState', async () => {
      const module = await import('@/utils/composition.ts');
      module.resetLoctData();

      expect(mockStore.state.loct.resetLoctState).toHaveBeenCalled();
    });
  });

  it('loading 应该设置加载状态', async () => {
    const module = await import('@/utils/composition.ts');
    module.loading(true);

    expect(mockStore.state.glob.setGlobState).toHaveBeenCalledWith({
      type: 'isLoading',
      value: true,
    });
  });
});
