//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { VideoOpCode } from '@/model/base-enum';

export const packed = {
  /**
   * video消息格式定义，命名规则如下
   * @param opcode 操作码，即消息类型（1字节）
   * @param blade 单板槽位号(默认值0)
   * @param reserved 保留字
   * @param data 消息体数据
   * 
   * 消息格式定义：消息体长度(4B)+消息类型(2B)+单板槽位号(1B)+预留块(1B)+消息体
   */
   packageData(opcode: number, data: number[] = [], blade: number = 0x00) {
    const packetedData = new Uint8Array(data.length + 8);
    packetedData[0] = opcode & 0xff;
    packetedData[1] = blade & 0xff;
    packetedData[2] = (data.length & 0xff000000) >> 24;
    packetedData[3] = (data.length & 0xff0000) >> 16;
    packetedData[4] = (data.length & 0xff00) >> 8;
    packetedData[5] = data.length & 0xff;
    packetedData[6] = 0xff;
    packetedData[7] = 0xff;
    data.forEach((item, index) => packetedData[8 + index] = item);
    return packetedData;
  },
  // 连接请求
  packConnect(bodyData: number[]) {
    return packed.packageData(VideoOpCode.CONNECT, bodyData);
  },
   // 心跳包
   packHeartBeat() {
    return packed.packageData(VideoOpCode.HEARTBEAT);
  },
  // 断开连接请求
  packDisconnect() {
    return packed.packageData(VideoOpCode.DISCONNECT);
  },
  // 开始播放
  packStartPlay(videoType: number, playMode: number = 1, blade: number = 0x00) {
    // playMode为1代表只播放,现只支持此一种模式
    const data = new Uint8Array(2);
    data[0] = videoType & 0xff;
    data[1] = playMode & 0xff;
    return packed.packageData(VideoOpCode.PLAY_START, [...data], blade);
  },
  // 暂停
  packPausePlay() {
    return packed.packageData(VideoOpCode.PLAY_PAUSE);
  },
  // 继续播放
  packContinuePlay() {
    return packed.packageData(VideoOpCode.PLAY_CONTINUE);
  },
  // 停止
  packStopPlay() {
    return packed.packageData(VideoOpCode.PLAY_STOP);
  },
  // 倍速播放
  packSpeedPlay(speed: number = 0) {
    const data = new Uint8Array(1);
    data[0] = speed & 0xff;
    return packed.packageData(VideoOpCode.PLAY_SPEED, [...data]);
  },
  // 拖动进度条
  packProgressPlay(progress: number = 0) {
    const data = new Uint8Array(4);
    data[0] = (progress & 0xff000000) >> 24;
    data[1] = (progress & 0xff0000) >> 16;
    data[2] = (progress & 0xff00) >> 8;
    data[3] = progress & 0xff;
    return packed.packageData(VideoOpCode.PLAY_PROGRESS, [...data]);
  },
};