//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { traduction } from '@/utils/language';
import { LOGIN_RULES_IP1, LOGIN_RULES_IP2, IPV6, IPV6_PREFIX } from '@/utils/regular';
// 时间格式校验错误提示信息
export const getErrorMsg = (num: number): string => {
  const errorMssage = {
    1: traduction('VALIDATOR_TIME_FORMAT_INCORRECT'),
    2: traduction('VALIDATOR_START_ENDING'),
    3: traduction('VALIDATOR_RIGHT_TIME1'),
    4: traduction('VALIDATOR_RIGHT_TIME2'),
    5: traduction('VALIDATOR_WRONG_TIME1'),
    6: traduction('VALIDATOR_WRONG_TIME2'),
  };
  return errorMssage[num];
};

// 自身格式校验
export const selfTimeValid = (isStart: boolean, item: any, val: string, callback: any) => {
  if (!val) {
    callback();
  } else {
    // 支持的时间格式
    const reg = /(^\d{4}-\d{2}-\d{2} \d{2}:\d{2}$)|(^\d{4}-\d{2}-\d{2}$)|(^\d{2}:\d{2})/;
    if (!reg.test(val)) {
      callback(new Error(getErrorMsg(1)));
      return false;
    }

    let valStr = val;
    if (valStr.indexOf('-') === -1) {
      valStr = `1970-01-01 ${val}`;
    }

    valStr = valStr.replace(' ', 'T');

    // 判断时间格式错误
    if (new Date(valStr).toString() === 'Invalid Date') {
      callback(new Error(getErrorMsg(1)));
      return false;
    }

    const MIN_YEAR = 1970;
    const MAX_YEAR = 2050;
    const transformedValueYear = new Date(val).getFullYear();
    if (transformedValueYear < MIN_YEAR || transformedValueYear > MAX_YEAR) {
      callback(new Error(getErrorMsg(2)));
      return false;
    }
  }

  if (isStart) {
    return interTimeValid(isStart, val, item.endTime, callback);
  } else {
    return interTimeValid(isStart, val, item.startTime, callback);
  }
};

// 和另外一个时间对比
export const interTimeValid = (isStart: boolean, valTemp: string, compareValTemp: string, callback: any): any => {
  let val = valTemp;
  let compareVal = compareValTemp;
  if (!val || !compareVal) {
    callback();
    return true;
  }
  // 验证开始、结束时间提示
  if (!isStart) {
    if (val.length !== compareVal.length) {
      callback(new Error(getErrorMsg(4)));
      return false;
    }
  } else {
    if (compareVal && (!val || compareVal.length !== val.length)) {
      callback(new Error(getErrorMsg(3)));
      return false;
    }
  }

  if (compareVal.indexOf('-') === -1) {
    const extraDateValue = '1970-01-01T';
    compareVal = extraDateValue + compareVal;
    val = extraDateValue + val;
  }

  if (isStart && new Date(compareVal) <= new Date(val)) {
    callback(new Error(getErrorMsg(5)));
    return false;
  }

  if (!isStart && new Date(compareVal) >= new Date(val)) {
    callback(new Error(getErrorMsg(6)));
    return false;
  }

  callback();
  return true;
};

// ip 地址校验
export const ipValid = (val: string, callback: any) => {
  if (!val) {
    callback();
    return true;
  }

  if (!LOGIN_RULES_IP1.test(val)) {
    if (!LOGIN_RULES_IP2.test(val)) {
      if (!IPV6.test(val) && !IPV6_PREFIX.test(val)) {
        callback(new Error(traduction('VALIDATOR_INCORRECT')));
        return false;
      }
    }
  }
  callback();
  return true;
};

// mac地址校验
export const macValid = (val: string, callback: any) => {
  if (!val) {
    callback();
    return true;
  }
  const reg6 = /^([A-Fa-f0-9]{2}[:]){5}[A-Fa-f0-9]{2}$/;
  const reg3 = /^([A-Fa-f0-9]{2}[:]){2}[A-Fa-f0-9]{2}$/;
  if (!(reg6.test(val) || reg3.test(val))) {
    callback(new Error(traduction('VALIDATOR_MAC_ERROR')));
    return false;
  } else {
    callback();
    return true;
  }
};

