// Copyright (c) Huawei Technologies Co., Ltd. 2022-2024. All rights reserved.
// 
// this file licensed under the Mulan PSL v2.
// You can use this software according to the terms and conditions of the Mulan PSL v2.
// You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
//
// THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
// PURPOSE.
// See the Mulan PSL v2 for more details.
import { getCurrentInstance } from 'vue';
import { useI18n } from 'vue-i18n';
import { traduction } from '@/utils/language';

interface Entry {
	zh: string;
	en: string;
}
type Dict = Record<string, Entry>;

const KEY_BY_CANON: Record<string, string> = {
	CPU: 'HOME_PROCESSOR',
	Dimm: 'HOME_MEMORY',
	Disk: 'HOME_STORE',
	Fan: 'HOME_FAN',
	Psu: 'HOME_POWER',
};

export const UI_KEYS: Dict = {
	/** ====== 统一兜底词典 ====== */
	Chassis: { zh: '机箱', en: 'Chassis' },
	Cover: { zh: '机箱盖', en: 'Chassis Cover' },
	Mainboard: { zh: '主板', en: 'Mainboard' },

	// —— 顶栏 Topbar —— //
	TOPBAR_ORIGIN: { zh: '原始视图', en: 'Origin' },
	TOPBAR_EXPLODE: { zh: '爆炸视图', en: 'Exploded' },
	TOPBAR_RESTORE: { zh: '还原', en: 'Restore' },
	TOPBAR_RESET_CAMERA: { zh: '重置视角', en: 'Reset View' },
	TOPBAR_TOGGLE_BG: { zh: '切换背景', en: 'Toggle BG' },
	TOPBAR_MODE_CAMERA: { zh: '视角', en: 'Camera' },
	TOPBAR_MODE_MODEL: { zh: '整体', en: 'Model' },

	// —— 快照按钮 & 面板 —— //
	SNAP_SAVE: { zh: '保存快照', en: 'Save Snapshot' },
	SNAP_VIEW_DIFF: { zh: '查看差异', en: 'View Diff' },
	SNAP_VIEW_SNAPSHOTS: { zh: '查看快照', en: 'View Snapshots' },

	SNAP_DIFF_PANEL_TITLE: { zh: '硬件配置差异', en: 'Hardware Differences' },
	SNAP_DIFF_PANEL_TIME_LABEL: { zh: '快照时间', en: 'Snapshot Time' },
	SNAP_SECTION_ABSENT: { zh: '不在位（已保存但当前缺失）', en: 'Absent (Saved but missing now)' },
	SNAP_SECTION_ADDED: { zh: '新增（当前存在但快照未记录）', en: 'Added (Now present, not in snapshot)' },
	SNAP_SECTION_CHANGED: { zh: '序列号变化（名称相同）', en: 'Serial Changed (same name)' },
	SNAP_EMPTY_DIFF: { zh: '当前与快照一致，未发现差异。', en: 'No differences from snapshot.' },

	SNAP_LIST_PANEL_TITLE: { zh: '当前快照清单', en: 'Snapshot Items' },
	SNAP_LIST_PANEL_TIME_LABEL: { zh: '快照时间', en: 'Snapshot Time' },
	SNAP_EMPTY_LIST: { zh: '暂无快照，请先点击“保存快照”。', en: 'No snapshot yet. Click “Save Snapshot”.' },
	SNAP_COUNT_UNIT: { zh: '项', en: 'items' },

	SNAP_TOOLTIP_SAVE: { zh: '保存当前硬件配置快照（名称+序列号）', en: 'Save current hardware snapshot (name + SN)' },
	SNAP_TOOLTIP_VIEW_DIFF: { zh: '查看与快照的差异', en: 'View differences vs. snapshot' },
	SNAP_TOOLTIP_VIEW_LIST: { zh: '查看当前已保存的快照清单', en: 'View saved snapshot items' },

	SNAP_LABEL_SN: { zh: '序列号', en: 'SN' },

	// —— 侧边栏 —— //
	SIDEBAR_EXPAND: { zh: '展开列表', en: 'Expand List' },
	SIDEBAR_COLLAPSE: { zh: '收起列表', en: 'Collapse List' },
	SIDEBAR_TITLE_PARTS: { zh: '部件', en: 'Parts' },
	SIDEBAR_SHOW_ALL: { zh: '全部显示', en: 'Show All' },
	SIDEBAR_HIDE_ALL: { zh: '全部隐藏', en: 'Hide All' },
	SIDEBAR_PLACEHOLDER_SEARCH: { zh: '搜索部件或分类…', en: 'Search parts or categories…' },
	SIDEBAR_FILTER_ALL: { zh: '全部分类', en: 'All Categories' },
	SIDEBAR_HINT_HAS_INFO: { zh: '有详细信息', en: 'Has details' },

	BOOT_LOADING_TITLE: { zh: '正在加载 3D 模型', en: 'Loading 3D Model' },
	BOOT_LOADING_SUB: { zh: '请稍候，不要关闭页面…', en: 'Please wait, do not close the page…' },
	BOOT_LOADING_PROGRESS: { zh: '加载进度', en: 'Loading progress' },
};

function resolveLocale(explicit?: string): string {
	/** 当前语言：优先 vue-i18n，其次浏览器，默认 en */
	if (explicit) {
		return explicit;
	}
	try {
		const inst = getCurrentInstance();
		if (inst) {
			const { locale } = useI18n();
			const v =
				typeof (locale as any)?.value === 'string'
					? (locale as any).value
					: '';
			if (v) {
				return v;
			}
		}
	} catch {
		/* ignore */
	}
	if (typeof navigator !== 'undefined' && navigator.language) {
		return navigator.language;
	}
	return 'en';
}

export function tUI(key: string, locale?: string): string {
	/** ====== 统一翻译函数：既支持 UI key，也支持分类名 ====== */
	if (!key) {
		return '';
	}

	// 1) 先看全局 i18n 是否有同名 key
	const viaSelf = traduction(key);
	if (viaSelf && viaSelf !== key) {
		return viaSelf;
	}

	// 2) 若自定义名，尝试映射到全局 key 再查一次
	const mapped = KEY_BY_CANON[key];
	if (mapped) {
		const viaCanon = traduction(mapped);
		if (viaCanon && viaCanon !== mapped) {
			return viaCanon;
		}
	}

	// 3) 走本地兜底
	const hit = UI_KEYS[key];
	if (!hit) {
		return key;
	}

	const lang = resolveLocale(locale).toLowerCase();
	return lang.startsWith('zh') ? hit.zh : hit.en;
}
