// Copyright (c) Huawei Technologies Co., Ltd. 2022-2024. All rights reserved.
// 
// this file licensed under the Mulan PSL v2.
// You can use this software according to the terms and conditions of the Mulan PSL v2.
// You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
//
// THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
// PURPOSE.
// See the Mulan PSL v2 for more details.
/**
 * 右侧“部件目录”侧栏的数据与筛选逻辑
 * - 负责：分类结构、折叠状态、搜索关键字、当前过滤值
 * - 具体 UI 行为（全显/全隐、分类显隐等）由 PartsSidebar 自己完成
 */
import { ref, reactive, computed } from 'vue';
import type { Ref, ComputedRef } from 'vue';

interface PartsCatalogState {
	partMap: Record<string, any>;
	categoryMap: Record<string, string[]>;
	infoMap: Record<string, boolean>;
	collapsed: Record<string, boolean>;
	currentFilter: Ref<'all' | string>;
	searchKey: Ref<string>;
	sidebarCollapsed: Ref<boolean>;
	sortedCats: ComputedRef<string[]>;
	rebuildList: () => void;
	toggleCollapse: (cat: string) => void;
}

export function usePartsCatalog(): PartsCatalogState {
	/**
	 * 右侧“部件目录”侧栏的数据与筛选逻辑
	 * - 负责：分类结构、折叠状态、搜索关键字、当前过滤值
	 * - 具体 UI 行为（全显/全隐、分类显隐等）由 PartsSidebar 自己完成
	 */
	const partMap = reactive<Record<string, any>>({});
	const categoryMap = reactive<Record<string, string[]>>({});
	const infoMap = reactive<Record<string, boolean>>({});

	const collapsed = reactive<Record<string, boolean>>({});
	const currentFilter = ref<'all' | string>('all');
	const searchKey = ref('');
	const sidebarCollapsed = ref(false);

	// 分类排序
	const sortedCats = computed(() =>
		Object.keys(categoryMap).sort((a, b) => a.localeCompare(b, 'zh-CN')),
	);

	/**
	 * three 初始化后调用：为每个分类建立折叠状态
	 */
	function rebuildList(): void {
		sortedCats.value.forEach((c) => {
			if (!(c in collapsed)) {
				collapsed[c] = false;
			}
		});
	}

	// 切换分类折叠
	const toggleCollapse = (cat: string): void => {
		collapsed[cat] = !collapsed[cat];
	};

	return {
		partMap,
		categoryMap,
		infoMap,
		collapsed,
		currentFilter,
		searchKey,
		sidebarCollapsed,
		sortedCats,
		rebuildList,
		toggleCollapse,
	};
}
