// Copyright (c) Huawei Technologies Co., Ltd. 2022-2024. All rights reserved.
// 
// this file licensed under the Mulan PSL v2.
// You can use this software according to the terms and conditions of the Mulan PSL v2.
// You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
//
// THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
// PURPOSE.
// See the Mulan PSL v2 for more details.
/**
 * 悬浮信息卡片（鼠标跟随 / 贴近元素显示）
 * - 仅处理位置与样式，不涉及业务文案
 */
import { ref, reactive, type Ref } from 'vue';
export function useHoverLabel(
	rootRef: any,
): {
	labelRef: Ref<HTMLDivElement | null>;
	hoverLabel: Ref<string>;
	hoverPairs: Ref<Array<{ k: string; v: string }>>;
	hoverStyle: { left: string; top: string };
	positionLabel: (e: PointerEvent) => void;
	clampLabel: () => void;
} {
	const labelRef = ref<HTMLDivElement | null>(null);
	const hoverLabel = ref('');
	const hoverPairs = ref<Array<{ k: string; v: string }>>([]);
	const hoverStyle = reactive({ left: '0px', top: '0px' });

	/** 跟随指针放置标签（自动防止越界） */
	function positionLabel(e: PointerEvent): void {
		const pad = 10;
		const el = labelRef.value;
		const root = rootRef.value;
		if (!root || !el) {
			return;
		}

		const rootRect = root.getBoundingClientRect();
		let x = e.clientX - rootRect.left + 12;
		let y = e.clientY - rootRect.top + 12;
		const w = el.offsetWidth || 260;
		const h = el.offsetHeight || 40;
		const maxX = rootRect.width - w - pad;
		const maxY = rootRect.height - h - pad;

		if (x > maxX) {
			x = e.clientX - rootRect.left - w - 12;
		}
		if (y > maxY) {
			y = e.clientY - rootRect.top - h - 12;
		}

		hoverStyle.left = `${Math.max(pad, x)}px`;
		hoverStyle.top = `${Math.max(pad, y)}px`;
		clampLabel();
	}

	/** 二次收紧，确保在容器内 */
	function clampLabel(): void {
		const el = labelRef.value;
		const root = rootRef.value;
		if (!el || !root) {
			return;
		}

		const pad = 10;
		const rootRect = root.getBoundingClientRect();
		const w = el.offsetWidth;
		const h = el.offsetHeight;

		let left = parseFloat(hoverStyle.left);
		let top = parseFloat(hoverStyle.top);

		if (left + w + pad > rootRect.width) {
			left = Math.max(pad, rootRect.width - w - pad);
		}
		if (top + h + pad > rootRect.height) {
			top = Math.max(pad, rootRect.height - h - pad);
		}

		hoverStyle.left = `${left}px`;
		hoverStyle.top = `${top}px`;
	}

	return {
		labelRef,
		hoverLabel,
		hoverPairs,
		hoverStyle,
		positionLabel,
		clampLabel,
	};
}
