//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import Date from '@/utils/date';
import { getFdmItemData } from '../fdm.server';

import {
  createCPUData,
  createDIMM,
  createDisk,
  createPSU,
  createFans,
  createDiskBP,
  createRaid,
  createPcieCard,
  createLom,
  createRiserCard,
  createMainboard,
  createChassis,
} from './detialsFctory';

import { getCases } from './tableFctory';

// 对接口返回字段进行数据整合,数据初始化
export function initIalizingData(
  value: string | number | Array<any[]>,
  unit?: string,
): string | Array<any[]> | number | null {
  if (value instanceof Array && value.length > 0) {
    if (value.length > 0) {
      // 如果是数组且长度大于0,直接返回数组
      return value;
    } else {
      // 为空数组的时候返回null，方便做 || '--' 判断
      return '--';
    }
  } else if (value instanceof Boolean) {
    // 是布尔数据直接返回
    return value;
  } else if (value === 0 || value) {
    // 是数字 或者 字符串的时候
    if (unit) {
      // 传了单位，那么进行单位拼接
      return value + unit;
    }
    // 没有单位，直接返回该数据
    return value;
  }
  return '--';
}

// 根据reportTimeStamp和reportTimeZone得到最终时间
export function displayDateTime(reportTimeStamp: string, reportTimeZone: string) {
  if (/^\d+$/.test(reportTimeStamp) || /^\d+$/.test(reportTimeZone)) {
    const date = parseInt(reportTimeStamp, 10);
    const timeZone = parseInt(reportTimeZone, 10) * 60;
    return new Date((date + timeZone) * 1000).format('yyyy-MM-dd hh:mm:ss');
  } else {
    return '--';
  }
}

function detailsDataFctory(data: any, node: any, detailsData: any, reqRes: any, url: string) {
  let result: any[] = [];
  if (data) {
    const liveTime = initIalizingData(node.onlineTime)?.toString() || '--';
    switch (node.componentID) {
      case 0:
        result = createCPUData(data, liveTime);
        break;
      case 1:
        result = createDIMM(data, liveTime);
        break;
      case 2:
        result = createDisk(data, liveTime);
        break;
      case 3:
        result = createPSU(data, liveTime, url);
        break;
      case 4:
        result = createFans(data, liveTime);
        break;
      case 5:
        result = createDiskBP(data, liveTime);
        break;
      case 6:
        result = createRaid(data, liveTime);
        break;
      case 8:
        result = createPcieCard(data, liveTime);
        break;
      case 13:
        result = createLom(data, liveTime);
        break;
      case 15:
        result = createRiserCard(data, liveTime);
        break;
      case 16:
        result = createMainboard(data, liveTime);
        break;
      case 18:
        result = createChassis(liveTime);
        break;
      default:
        result = [];
    }
  }
  detailsData.push(...result);
  reqRes.details = true;
}

// 获取节点详情数据
export function getDetailsData(node: any, detailsData: any, reqRes: any) {
  reqRes.details = false;
  const url = node.deviceInfo;
  getFdmItemData(url)
    .then((data: any) => {
      detailsDataFctory(data, node, detailsData, reqRes, url);
    })
    .catch(() => {
      reqRes.details = true;
    });
}

// 获取表格数据
export function getTableData(node: any, tableData: any, reqRes: any) {
  reqRes.table = false;
  const url = node.eventReports;
  getFdmItemData(url)
    .then((data: any) => {
      if (data) {
        const caseArr = getCases(data.EventRecords);
        tableData.push(...caseArr);
      }
      reqRes.table = true;
    })
    .catch(() => {
      reqRes.table = true;
    });
}
