//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { triggerRouteError, validateSystemIds } from '@/utils/system-ids';
import type { NavigationGuardNext, RouteLocationNormalized } from 'vue-router';

/**
 * 系统ID路由守卫
 * 
 * 主要功能：
 * 1. 验证环境是否支持多主机功能，防止不支持的环境通过路由跳转进入相关页面
 * 2. 管理路由中的systemId参数，确保双主机环境下参数的有效性
 * 3. 单主机环境下清理不必要的systemId参数
 * 
 * 处理逻辑：
 * - 如果SystemIds数据未加载（非数组），先请求数据再进行判断
 * - 数据获取成功但为空时，按单主机处理（兼容旧接口）
 * - 双主机环境（SystemIds长度为2）：
 *   - 验证路由是否携带有效的systemId参数
 *   - 如参数无效或缺失，默认使用第一个系统ID
 * - 单主机环境（SystemIds长度为1）：
 *   - 正常放行，清理路由中多余的systemId参数
 * - 请求报错时触发路由错误
 */
export async function systemIdsGuard(to: RouteLocationNormalized, _:any, next: NavigationGuardNext): Promise<void> {
  try {
    // 使用公共工具函数进行systemIds验证
    const validationResult = await validateSystemIds(to);
    
    // 检查是否需要修正查询参数
    const hasQueryChanged = JSON.stringify(validationResult.validatedQuery) !== JSON.stringify(to.query);
    
    if (hasQueryChanged) {
      // 需要修正路由参数，进行重定向
      next({
        ...to,
        query: validationResult.validatedQuery,
      });
    } else {
      // 参数正确，正常放行
      next();
    }
  } catch (error) {
    // 验证失败，触发路由错误处理
    triggerRouteError();
  }
}