-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

require 'skynet.manager'
require 'thermal_mgmt.json_types.Fan'
require 'thermal_mgmt.json_types.FanType'
require 'thermal_mgmt.json_types.CoolingConfig'
require 'thermal_mgmt.json_types.CoolingRequirement'
require 'thermal_mgmt.json_types.CoolingPolicy'
require 'thermal_mgmt.json_types.CoolingFan'
require 'thermal_mgmt.json_types.LiquidCoolingConfig'
require 'thermal_mgmt.json_types.CoolingPump'
require 'thermal_mgmt.json_types.Pump'
require 'thermal_mgmt.json_types.FanSnapshot'

local skynet = require 'skynet'
local sdbus = require 'sd_bus'
local utils = require 'mc.utils'
local test_common = require 'test_common.utils'
local log = require 'mc.logging'
local mdb = require 'mc.mdb'
local test_ipmi = require 'test_ipmi'
local context = require'mc.context'
local lu = require 'luaunit'
local ctx = require 'mc.context'
local rpc_test = require 'test_rpc'
local cooling_obj = require 'test_cooling_job'
local test_min_pwm = require 'test_min_pwm'
local data_keeping_test = require 'test_data_keeping'
local special_rpc_test = require 'test_special_rpc'
local on_reboot_test = require 'test_on_reboot'
local test_pumps_config = require 'test_pumps_config'
local test_conflict_requirement_obj = require 'test_conflict_requirement_obj'
local test_business_config = require 'test_business_config'
local test_customize_config = require 'test_customize_config'

log:set_debug_log_type(log.OUT_TYPE_LOCAL)
log:setLevel(log.INFO)

local ctx_new = ctx.new()

local FAN_PATH<const> = '/bmc/kepler/Systems/1/Thermal/Fans'
local FAN_INTERFACE<const> = 'bmc.kepler.Systems.Fan'

local project_dir = os.getenv('PROJECT_DIR')
local test_dir = project_dir .. '/test/integration/.test_temp_data/'
skynet.setenv('PROG_APP_PATH', test_dir .. 'apps')
skynet.setenv('PROG_CSR_PATH', test_dir .. 'sr')
skynet.setenv('DATA_CSR_PATH', test_dir .. 'data/sr')

local function prepare_test_data()
    log:info('== prepare test data')
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')
    os.execute('mkdir -p ' .. test_data_dir)
    -- 创建目录
    local dir_list = {'apps/thermal_mgmt/mds', 'apps/hwdiscovery/mds', 'apps/ipmi_core/mds' , 'apps/pcie_device/mds',
        'apps/general_hardware/mds', 'apps/soctrl/mds', 'apps/bmc_soc/mds', 'apps/chassis/mds', 'sr'}
    for _, path in pairs(dir_list) do
        os.execute('mkdir -p ' .. test_data_dir .. '/' .. path)
    end
    utils.copy_file('test/integration/test_data/sr/14100513_BCU_01.sr',
        test_data_dir .. '/sr/14100513_BCU_01.sr')
    utils.copy_file('test/integration/test_data/sr/14100513_EXU_01.sr',
        test_data_dir .. '/sr/14100513_EXU_01.sr')
    utils.copy_file('test/integration/test_data/sr/14100363_00000001050302044490.sr',
        test_data_dir .. '/sr/14100363_00000001050302044490.sr')
    utils.copy_file('test/integration/test_data/sr/root.sr', test_data_dir .. '/sr/root.sr')
    log:info('====== %s', test_data_dir .. '/sr/root.csr')
    utils.copy_file('mds/schema.json', test_data_dir .. '/apps/thermal_mgmt/mds/schema.json')
    log:info('====== %s', test_data_dir .. '/apps/thermal_mgmt/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/hwdiscovery/mds/schema.json',
        test_data_dir .. '/apps/hwdiscovery/mds/schema.json')
    log:info('====== %s', test_data_dir .. '/apps/hwdiscovery/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/ipmi_core/mds/schema.json',
        test_data_dir .. '/apps/ipmi_core/mds/schema.json')
    log:info('====== %s', test_data_dir .. '/apps/ipmi_core/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/fructrl/mds/schema.json', test_data_dir .. '/apps/fructrl/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/pcie_device/mds/schema.json',
        test_data_dir .. '/apps/pcie_device/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/general_hardware/mds/schema.json',
        test_data_dir .. '/apps/general_hardware/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/soctrl/mds/schema.json',
        test_data_dir .. '/apps/soctrl/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/bmc_soc/mds/schema.json',
        test_data_dir .. '/apps/bmc_soc/mds/schema.json')
        utils.copy_file('temp/opt/bmc/apps/chassis/mds/schema.json',
        test_data_dir .. '/apps/chassis/mds/schema.json')
end

local function clear_test_data(exit_test)
    log:info('== clear test data')
    local test_data_dir = utils.realpath('.') .. '/' .. skynet.getenv('TEST_DATA_DIR')
    if not exit_test then
        return utils.remove_file(test_data_dir)
    end

    skynet.timeout(0, function()
        skynet.sleep(20)
        skynet.abort()
        utils.remove_file(test_data_dir)
    end)
end

local function test_init(bus)
    local fan_4_path = '/bmc/kepler/Systems/1/Thermal/Fans/Fan_4_0101'
    local ok, fan_4_obj = pcall(mdb.get_object, bus, fan_4_path, 'bmc.kepler.Systems.Fan')
    assert(ok)

    local c_config_path = '/bmc/kepler/Systems/1/CoolingConfig'
    local pumps_config_path = '/bmc/kepler/Systems/1/LiquidCoolingConfig'
    local c_requirement_1_path = '/bmc/kepler/Systems/1/CoolingRequirement/CoolingRequirement_1_1_0101'
    local c_requirement_2_path = '/bmc/kepler/Systems/1/CoolingRequirement/CoolingRequirement_1_2_0101'
    local c_policy_1_path = '/bmc/kepler/Systems/1/CoolingPolicy/CoolingPolicy_1_1_0101'
    local c_fan_1_path = '/bmc/kepler/Systems/1/CoolingFan/CoolingFan_1_1_0101'
    local c_fan_2_path = '/bmc/kepler/Systems/1/CoolingFan/CoolingFan_1_2_0101'
    local t_pump_1_path = '/bmc/kepler/Systems/1/CoolingPumps/CoolingPump_1_0101'
    local t_pump_2_path = '/bmc/kepler/Systems/1/CoolingPumps/CoolingPump_2_0101'
    local c_config_obj, pumps_config_obj, c_requirement_1_obj, c_policy_1_obj, c_requirement_2_obj,
        c_pump_1_obj, c_pump_2_obj, c_fan_1_obj, c_fan_2_obj

    ok, c_config_obj = pcall(mdb.get_object, bus, c_config_path,
        'bmc.kepler.Systems.CoolingConfig')
    assert(ok)
    ok, pumps_config_obj = pcall(mdb.get_object, bus, pumps_config_path,
        'bmc.kepler.Systems.LiquidCoolingConfig')
    assert(ok)
    ok, c_requirement_1_obj = pcall(mdb.get_object, bus, c_requirement_1_path,
        'bmc.kepler.Systems.CoolingRequirement')
    assert(ok)
    ok, c_requirement_2_obj = pcall(mdb.get_object, bus, c_requirement_2_path,
        'bmc.kepler.Systems.CoolingRequirement')
    assert(ok)
    ok, c_policy_1_obj = pcall(mdb.get_object, bus, c_policy_1_path, 'bmc.kepler.Systems.CoolingPolicy')
    assert(ok)
    ok, c_fan_1_obj = pcall(mdb.get_object, bus, c_fan_1_path, 'bmc.kepler.Systems.CoolingFan')
    assert(ok)
    ok, c_fan_2_obj = pcall(mdb.get_object, bus, c_fan_2_path, 'bmc.kepler.Systems.CoolingFan')
    assert(ok)
    ok, c_pump_1_obj = pcall(mdb.get_object, bus, t_pump_1_path, 'bmc.kepler.Systems.CoolingPump')
    assert(ok)
    ok, c_pump_2_obj = pcall(mdb.get_object, bus, t_pump_2_path, 'bmc.kepler.Systems.CoolingPump')
    assert(ok)

    return {
        ['fan_4_obj'] = fan_4_obj,
        ['conf'] = c_config_obj,
        ['pumps_conf'] = pumps_config_obj,
        ['req_1'] = c_requirement_1_obj,
        ['req_2'] = c_requirement_2_obj,
        ['policy_1'] = c_policy_1_obj,
        ['c_pump_1'] = c_pump_1_obj,
        ['c_pump_2'] = c_pump_2_obj,
        ['c_fan_1'] = c_fan_1_obj,
        ['c_fan_2'] = c_fan_2_obj,
    }
end

-- 测试风扇类型识别
local function test_fan_type_identify(fan_4_obj)
    assert(fan_4_obj.Model == '32030275 8038+', 'Actual: ' .. fan_4_obj.Model)
end
local function test_thermal_mgmt_set_fan_pwm(bus)
    log:info('================ test_thermal_mgmt_set_fan_pwm start ================')
    local fan_objs = mdb.get_sub_objects(bus, FAN_PATH, FAN_INTERFACE)
    local PWM<const> = 50
    local RET_SUCCESS<const> = 0
    fan_objs:fold(function (obj)
        local rsp = obj:SetFanPWM_PACKED(context.new(), PWM)
        lu.assertTrue(rsp ~= nil)
        lu.assertEquals(rsp.Result, RET_SUCCESS)
    end)
    log:info('================ test_thermal_mgmt_set_fan_pwm complete ================')
end

-- 数据采集GetItems接口
local function test_thermal_mgmt_get_items(bus)
    log:info('================ test_thermal_mgmt_get_items start ================')
    local component, classification, identification, metricName = bus:call('bmc.kepler.thermal_mgmt', 
        '/bmc/kepler/Systems/1/Thermal/Fans/Fan_1_0101', 'bmc.kepler.Metric', 'GetItems', 'a{ss}', context.new())
    lu.assertEquals(component, 'Fan')
    lu.assertEquals(classification[1][1], 'Model')
    lu.assertEquals(identification[1][1], 'FANSlot')
    lu.assertEquals(metricName[2], 'fan.rrpm')
    log:info('================ test_thermal_mgmt_get_items complete ================')
    return metricName
end

-- 数据采集GetData接口
local function test_thermal_mgmt_get_data(bus, metricName)
    log:info('================ test_thermal_mgmt_get_data start ================')
    local val = bus:call('bmc.kepler.thermal_mgmt', '/bmc/kepler/Systems/1/Thermal/Fans/Fan_1_0101',
        'bmc.kepler.Metric', 'GetData', 'a{ss}as', context.new(), metricName)
    lu.assertEquals(#val, 2)
    lu.assertEquals(val[1][1], 'fan.frpm')
    lu.assertEquals(val[1][2][1], '0')
    lu.assertEquals(val[2][1], 'fan.rrpm')
    lu.assertEquals(val[2][2][1], '4000')

    val = bus:call('bmc.kepler.thermal_mgmt', '/bmc/kepler/Systems/1/Thermal/Fans/Fan_2_0101',
        'bmc.kepler.Metric', 'GetData', 'a{ss}as', context.new(), metricName)
    lu.assertEquals(#val, 2)
    lu.assertEquals(val[1][1], 'fan.frpm')
    lu.assertEquals(val[1][2][1], '5000')
    lu.assertEquals(val[2][1], 'fan.rrpm')
    lu.assertEquals(val[2][2][1], "5000")
    log:info('================ test_thermal_mgmt_get_data complete ================')
end

-- 设置本地日志等级
local function set_log_level(bus, apps, level)
    for _, app in ipairs(apps) do
        local service = string.format('bmc.kepler.%s', app)
        local path = string.format('/bmc/kepler/%s/MicroComponent', app)
        local intf = 'bmc.kepler.MicroComponent.Debug'
        -- 设置日志等级持续时间1hour
        local ok, err = pcall(bus.call, bus, service, path, intf, 'SetDlogLevel', 'a{ss}sy', context.new(), 'info', 1)
        if not ok then
            log:error('Set app [%s] log level [%s] failed, error: %s', app, level, err)
        else
            log:notice('Set app [%s] log level [%s] successfully', app, level)
        end
    end
end

local function test_thermal_mgmt()
    skynet.sleep(6000) -- 等待60秒风扇识别
    log:info('================ test thermal_mgmt start ================')
    local bus = sdbus.open_user(true)
    local apps = {'thermal_mgmt'}
    set_log_level(bus, apps, 'info')
    skynet.sleep(6000)
    local objs = test_init(bus)
    test_fan_type_identify(objs['fan_4_obj'])
    test_ipmi.main(bus, objs['conf'], objs['pumps_conf'],
        objs['c_fan_1'], objs['c_fan_2'], objs['c_pump_1'], objs['c_pump_2'])
    test_thermal_mgmt_set_fan_pwm(bus)
    local metricName = test_thermal_mgmt_get_items(bus)
    test_thermal_mgmt_get_data(bus, metricName)
    test_min_pwm.main(bus, ctx_new, objs['conf'])
    rpc_test.main(bus, objs['conf'], objs['policy_1'], ctx_new)
    cooling_obj.main(bus, objs['conf'], ctx_new)
    data_keeping_test.main(bus, objs['conf'])
    special_rpc_test.main(bus, ctx_new)
    on_reboot_test.main(bus, ctx_new)
    test_pumps_config.main(bus,ctx_new, objs['pumps_conf'], objs['c_pump_1'], objs['c_pump_2'])
    test_conflict_requirement_obj.main(bus)
    test_business_config.main(bus)
    test_customize_config.main(bus)
    skynet.call('thermal_mgmt', 'lua', 'exit')
    log:info('================ test thermal_mgmt complete ================')
end

skynet.start(function()
    clear_test_data()
    prepare_test_data()
    test_common.dbus_launch()
    skynet.uniqueservice('sd_bus')
    skynet.uniqueservice('persistence/service/main')
    skynet.uniqueservice('maca/service/main')
    skynet.uniqueservice('hwdiscovery/service/main')
    skynet.uniqueservice('soctrl/service/main')
    skynet.uniqueservice('bmc_soc/service/main')
    skynet.uniqueservice('ipmi_core/service/main')
    skynet.uniqueservice('main')
    skynet.uniqueservice('fructrl/service/main')
    skynet.uniqueservice('pcie_device/service/main')
    skynet.uniqueservice('general_hardware/service/main')
    skynet.uniqueservice('chassis/service/main')
    skynet.sleep(200)
    skynet.fork(function()
        skynet.sleep(10)
        local ok, err = pcall(test_thermal_mgmt)
        clear_test_data(true)
        if not ok then
            error(err)
        end
    end)
end)
