-- Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
--
-- this file licensed under the Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
-- PURPOSE.
-- See the Mulan PSL v2 for more details.
local class = require 'mc.class'
local log = require 'mc.logging'
local json = require 'cjson'
local thermal_mgmt_utils = require 'thermal_mgmt_utils'
local mdb_config_manage = require 'mc.mdb.micro_component.config_manage'
local business_config = require 'config_mgmt.business_config'
local customize_config = require 'config_mgmt.customize_config'
local interface_config = require 'config_mgmt.interface_config'
local thermal_subsystem = require 'basic_cooling.cooling_device.thermal_subsystem'

local server = class()

function server:ctor(db, bus)
    self.db = db
    self.bus = bus
    self.interface_config_instance = interface_config:get_instance()
    self.thermal_subsystem_instance = thermal_subsystem:get_instance()
end

function server:init()
    self:register()
end

function server:on_import(ctx, data_str, data_type)
    local data
    if data_str then
        data = json.decode(data_str)
    end
    if not data or not data.ConfigData or type(data.ConfigData) ~= 'table' then
        log:error('Import data [%s] is invalid', data_str)
        thermal_mgmt_utils.operation_log(ctx, "Import configuration failed")
        return
    end
    if data_type == 'custom' then
        if not data.ConfigData.CustomSettings or type(data.ConfigData.CustomSettings) ~= 'table' then
            log:error('Import data [%s] is invalid', data_str)
            thermal_mgmt_utils.operation_log(ctx, 'Import configuration failed')
            return
        end
        customize_config.on_import(self, ctx, data.ConfigData.CustomSettings)
    elseif data_type == 'configuration' then
        business_config.on_import(self, ctx, data.ConfigData)
    else
        log:error('Import type [%s] is invalid', data_type)
        thermal_mgmt_utils.operation_log(ctx, 'Import configuration failed')
    end
end

function server:on_export(ctx, type)
    if type ~= 'custom' and type ~= 'configuration' then
        log:error('Export type [%s] is invalid', type)
    end
    local data = {}
    -- 配置导出在thermal组件
    if type == 'configuration' then
        local ret = business_config.on_export(self, ctx)
        data = ret
    elseif type == 'custom' then
        local ret = customize_config.on_export(self, ctx)
        data.CustomSettings = ret
    end
    return json.encode({ ConfigData = data })
end

function server:register()
    mdb_config_manage.on_import(function(...) return self:on_import(...) end)
    mdb_config_manage.on_export(function(...) return self:on_export(...) end)
end

return server