-- Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
-- 
-- this file licensed under the Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN \"AS IS\" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
-- PURPOSE.
-- See the Mulan PSL v2 for more details.

local class = require 'mc.class'
local log = require 'mc.logging'
local client = require 'thermal_mgmt.client'
local singleton = require 'mc.singleton'

local enums = require 'basic_cooling.define.cooling_enums'
local props = require 'basic_cooling.define.cooling_properties'
local base = require 'basic_cooling.data_keeping.base'

local PCIE_CARD_INTERFACE <const> = "bmc.kepler.Systems.PCIeDevices.PCIeCard"

local cards_data_keeping = class(base)

function cards_data_keeping:ctor(bus)
    self.obj_name = 'PCIECard'
    -- 监听的资源树路径
    self.paths_table = {
        "/bmc/kepler/Systems/${SystemId}/PCIeDevices/OCPCards/${Id}",
        "/bmc/kepler/Systems/${SystemId}/PCIeDevices/PCIeCards/${Id}",
        "/bmc/kepler/Systems/${SystemId}/PCIeDevices/PCIeCards/DPUCards/${Id}"
    }
    -- 监听的接口
    self.interfaces_table = {
        [PCIE_CARD_INTERFACE] = {
            [enums.data_keeping_config.FOREACH_FUNC] = client.ForeachPCIeCardObjects,
            [enums.data_keeping_config.PROPERTIES] = {
                [props.PCIE_CARD.NAME] = enums.data_keeping_log.LOG
            }
        }
    }
    
    self.loaded_pciecards = {}
    self.pciecard_path_name_map = {}
end

function cards_data_keeping:init()
    self.super.init(self)
end

-- pciecard对象添加回调
function cards_data_keeping:action_after_obj_added(path, interface)
    local name = self.objs[path][interface][props.PCIE_CARD.NAME]
    if self.pciecard_path_name_map[path] then
        log:error('Duplicated pciecard obj(path:%s)', path)
        return
    end
    
    self.pciecard_path_name_map[path] = name
    self.loaded_pciecards[name] = (not self.loaded_pciecards[name]) and 1 or (self.loaded_pciecards[name] + 1)

    log:notice("Add pciecard obj(path:%s), name: %s, cur number:%s",
        path, name, self.loaded_pciecards[name])
end

-- cards对象移除回调
function cards_data_keeping:action_after_obj_removed(path, interface)
    local name = self.pciecard_path_name_map[path]
    if not name then
        log:error("Pciecard obj(path:%s) not in table", path)
        return
    end

    self.pciecard_path_name_map[path] = nil
    if self.loaded_pciecards[name] >= 1 then
        self.loaded_pciecards[name] = self.loaded_pciecards[name] - 1
    else
        log:error("The number of pciecard obj(name:%s) less then 1", name)
    end
    log:notice("Delete pciecard obj(path:%s), cur number:%u", path, self.loaded_pciecards[name])
end

function cards_data_keeping:action_after_property_changed(path, interface, prop, value)
    if prop ~= props.PCIE_CARD.NAME then
        return
    end
    local pre_pciecard_name = self.pciecard_path_name_map[path]
    if self.loaded_pciecards[pre_pciecard_name] >= 1 then
        self.loaded_pciecards[pre_pciecard_name] = self.loaded_pciecards[pre_pciecard_name] - 1
        log:notice('Update the number of pciecard obj(name:%s), cur number: %s', pre_pciecard_name,
            self.loaded_pciecards[pre_pciecard_name])
    else
        log:error("The number of pciecard obj(name:%s) less then 1", pre_pciecard_name)
    end

    self.loaded_pciecards[value] = (not self.loaded_pciecards[value]) and
        1 or (self.loaded_pciecards[value] + 1)
    self.pciecard_path_name_map[path] = value

    log:notice("Update pciecard obj(path:%s), pre name: %s, name: %s, cur number:%s",
        path, pre_pciecard_name, value, self.loaded_pciecards[value])
end

function cards_data_keeping:is_pciecard_loaded(name)
    return self.loaded_pciecards[name] and self.loaded_pciecards[name] > 0
end

return singleton(cards_data_keeping)