/*
 * Copyright (c) Huawei Technologies Co., Ltd. 2025-2025. All rights reserved.
 *
 * this file licensed under the Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 * http://license.coscl.org.cn/MulanPSL2
 *
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
 * PURPOSE.
 * See the Mulan PSL v2 for more details.
 *
 * Description: 华为自研的存储设备管理库(Storage Management Library)的基本信息, 主要包含业界标准或协议的定义
 */

#ifndef SML_COMMON_H
#define SML_COMMON_H

#include "sml_common_extend.h"

#ifdef __cplusplus
#if __cplusplus
extern "C" {
#endif
#endif /* __cplusplus */

#pragma pack(1)

/*
 * ata smart attribute data structure, vendor specific
 * SFF 8035 spec *
 * https://www.linux-mips.org/pub/linux/mips/people/macro/S.M.A.R.T./SFF-8035i.pdf
 */
typedef struct _ATA_SMART_ATTRIBUTE {
    guint8  id;
    guint16 flags;
    guint8  current;
    guint8  worst;
    guint8  raw[6]; // 属性原始值
    guint8  reserved;
} ATA_SMART_ATTRIBUTE_S;

/*
 * ata smart data structure
 * Table 62 of T13/1699-D (ATA8-ACS) Revision 6a
 * http://www.t13.org/documents/uploadeddocuments/docs2008/d1699r6a-ata8-acs.pdf
 */
typedef struct _ATA_SMART_DATA { // refers to ACS-3 Table-130
    guint16           revNumber;
    ATA_SMART_ATTRIBUTE_S smartAttribute[30];
    guint8            offLineDataCollectionStatus;
    guint8            selftestExecStatus;
    guint16           totalTimeToCompleteOffLine;
    guint8            vendorSpecific366;
    guint8            offLineDataCollectionCapability;
    guint16           smartCapability;
    guint8            errorlogCapability;
    guint8            vendorSpecific371;
    guint8            shortSelfTestPollingTime;
    guint8            extendSelfTestPollingTimeinByte;
    guint8            conveyanceSelfTestPollingTime;
    guint16           extendSelfTestPollingTimeinWord;
    guint8            reserved377_385[9];
    guint8            vendorSpecific386_510[125];
    guint8            chkSum;
} ATA_SMART_DATA_S;

/*
 * HW defined smart data structure
 */
typedef struct _ATA_SMART_DATA_HW_DEFINED {
    ATA_SMART_ATTRIBUTE_S smartAttribute[256]; // 华为自定义log page固定大小为3072Bytes，3072/12=256，因此idx最大为256
} ATA_SMART_DATA_HW_DEFINED_S;

/*
 * ata smart threshold data structure, vendor specific
 * 4.2. of SFF 8035i, page 13
 * https://www.linux-mips.org/pub/linux/mips/people/macro/S.M.A.R.T./SFF-8035i.pdf
 */
typedef struct _ATA_SMART_THRESHOLD_ENTRY {
    guint8 id;
    guint8 threshold;
    guint8 reserved[10];
} ATA_SMART_THRESHOLD_ENTRY_S;

/*
 * Device attribute threshold data structure
 * 4. of SFF 8035i, page 13
 * https://www.linux-mips.org/pub/linux/mips/people/macro/S.M.A.R.T./SFF-8035i.pdf
 */
typedef struct _ATA_SMART_THRESHOLDS {
    guint16                     revNumber;
    ATA_SMART_THRESHOLD_ENTRY_S thresholdEntries[30];
    guint8                      reserved[149];
    guint8                      chkSum;
} ATA_SMART_THRESHOLDS_S;

typedef struct _ATA_SMART_ERRLOG_INPUT {
    guint8 commandReg;
    guint8 featuresReg;
    guint8 status;
    guint8 errorRegister;
    guint16 sectorCount;
} ATA_SMART_ERRLOG_INPUT_S;

/*
 * Error log data structure
 * Table 42 of T13/1321D Rev 1 spec
 * https://www.seagate.com/support/disc/manuals/ata/d1153r17.pdf
 */
typedef struct _ATA_SMART_ERRLOG_ERR {
    guint8 reserved;
    guint8 errorRegister;
    guint8 sectorCount;
    guint8 sectorNumber;
    guint8 cylinderLow;
    guint8 cylinderHigh;
    guint8 driveHead;
    guint8 status;
    guint8 extendedError[ATA_LOG_EXT_ERR_LEN];
    guint8 state;
    guint16 timestamp;
} ATA_SMART_ERRLOG_ERR_S;

/*
 * Command Data Structure
 * Table 41 of T13/1321D Rev 1 spec
 * https://www.seagate.com/support/disc/manuals/ata/d1153r17.pdf
 */
typedef struct _ATA_SMART_ERRLOG_CMD {
    guint8 deviceControlReg;
    guint8 featuresReg;
    guint8 sectorCount;
    guint8 sectorNumber;
    guint8 cylinderLow;
    guint8 cylinderHigh;
    guint8 driveHead;
    guint8 commandReg;
    guint32 timestamp;
} ATA_SMART_ERRLOG_CMD_S;

/*
 * Error log data structure
 * Table 40 of T13/1321D Rev 1 spec
 * https://www.seagate.com/support/disc/manuals/ata/d1153r17.pdf
 */
typedef struct _ATA_SMART_ERRLOG_ENTRY {
    ATA_SMART_ERRLOG_CMD_S commands[ATA_LOG_ERR_CMD_NUM];
    ATA_SMART_ERRLOG_ERR_S error;
} ATA_SMART_ERRLOG_ENTRY_S;

/*
 * SMART error log sector
 * Table 39 of T13/1321D Rev 1 spec
 * https://www.seagate.com/support/disc/manuals/ata/d1153r17.pdf
 */
typedef struct _ATA_SMART_ERRLOG {
    guint8 revNumber;
    guint8 errorLogPointer;
    ATA_SMART_ERRLOG_ENTRY_S errorlog[ATA_LOG_ERR_ERR_NUM];
    guint16 errorCount;
    guint8 reserved[57];
    guint8 chkSum;
} ATA_SMART_ERRLOG_S;

/*
 * Error data structure
 * Table A.10 T13/1699-D Revision 6a
 * http://www.t13.org/documents/uploadeddocuments/docs2008/d1699r6a-ata8-acs.pdf
 */
typedef struct _ATA_SMART_EXTERRLOG_ERROR {
    guint8 deviceControlReg;
    guint8 errorReg;
    guint8 countReg;
    guint8 countRegHi;
    guint8 lbaLowReg;
    guint8 lbaLowRegHi;
    guint8 lbaMidReg;
    guint8 lbaMidRegHi;
    guint8 lbaHighReg;
    guint8 lbaHighRegHi;
    guint8 deviceReg;
    guint8 statusReg;
    guint8 extendedError[ATA_LOG_EXT_ERR_LEN];
    guint8 state;
    guint16 timestamp;
} ATA_SMART_EXTERRLOG_ERROR_S;

/*
 * Command data structure
 * Table A.9 of T13/1699-D Revision 6a
 * http://www.t13.org/documents/uploadeddocuments/docs2008/d1699r6a-ata8-acs.pdf
 */
typedef struct _ATA_SMART_EXTERRLOG_CMD {
    guint8 deviceControlReg;
    guint8 featuresReg;
    guint8 featuresRegHi;
    guint8 countReg;
    guint8 countRegHi;
    guint8 lbaLowReg;
    guint8 lbaLowRegHi;
    guint8 lbaMidReg;
    guint8 lbaMidRegHi;
    guint8 lbaHighReg;
    guint8 lbaHighRegHi;
    guint8 deviceReg;
    guint8 commandReg;
    guint8 reserved;
    guint32 timestamp;
} ATA_SMART_EXTERRLOG_CMD_S;

/*
 * Error log data structure
 * Table A.8 of T13/1699-D Revision 6a
 * http://www.t13.org/documents/uploadeddocuments/docs2008/d1699r6a-ata8-acs.pdf
 */
typedef struct _ATA_SMART_EXTERRLOG_ENTRY {
    ATA_SMART_EXTERRLOG_CMD_S commands[ATA_LOG_ERR_CMD_NUM];
    ATA_SMART_EXTERRLOG_ERROR_S error;
} ATA_SMART_EXTERRLOG_ENTRY_S;

/*
 * Ext. Comprehensive SMART error log
 * Table A.7 of T13/1699-D Revision 6a
 * http://www.t13.org/documents/uploadeddocuments/docs2008/d1699r6a-ata8-acs.pdf
 */
typedef struct _ATA_SMART_EXTERRLOG {
    guint8 version;
    guint8 reserved1;
    guint16 errorLogIndex;
    ATA_SMART_EXTERRLOG_ENTRY_S errorLogs[ATA_LOG_EXT_ERR_NUM];
    guint16 deviceErrorCount;
    guint8 reserved2[9];
    guint8 chkSum;
} ATA_SMART_EXTERRLOG_S;

/*
 * Self-test log descriptor entry
 * Table 45 of T13/1321D Rev 1 spec
 * http://www.t13.org/documents/uploadeddocuments/docs2008/d1699r6a-ata8-acs.pdf
 */
typedef struct _ATA_SMART_EXTSELFTESTLOG_DESC {
    guint8 selfTestType;
    guint8 selfTestStatus;
    guint16 timestamp;
    guint8 chkPoint;
    guint8 failingLba[6];
    guint8 vendorSpecific[15];
} ATA_SMART_EXTSELFTESTLOG_DESC_S;

/*
 * Self-test log data structure
 * Table 44 of T13/1321D Rev 1 spec
 * http://www.t13.org/documents/uploadeddocuments/docs2008/d1699r6a-ata8-acs.pdf
 */
typedef struct _ATA_SMART_EXTSELFTESTLOG {
    guint8 version;
    guint8 reserved1;
    guint16 logDescIndex;
    ATA_SMART_EXTSELFTESTLOG_DESC_S logDescs[ATA_LOG_EXT_SELF_TEST_LOG_NUM];
    guint8 vendorSpecifc[2];
    guint8 reserved2[11];
    guint8 chksum;
} ATA_SMART_EXTSELFTESTLOG_S;

typedef struct _ATA_SMART_SEAGATEGLIST_ITEM {
    guint16 R;    // 硬盘上的物理地址Radius
    guint16 T;    // 硬盘上的物理地址Theta
    guint16 Z;    // 硬盘上的物理地址Z
    guint32 LBA;  // 逻辑块地址
    guint16 POT;  // 上电时间 (Hours)
    guint8 a[4];
} ATA_SMART_SEAGATEGLIST_ITEM_S;

typedef struct _ATA_SMART_SEAGATEGLIST {
    guint8 reserved0[16];
    ATA_SMART_SEAGATEGLIST_ITEM_S items[ATA_LOG_SEAGATE_GLIST_NUM];
    guint8 reserved1[16];
} ATA_SMART_SEAGATEGLIST_S;

typedef struct _ATA_SMART_SEAGATEPLIST_ITEM {
    guint16 R;
    guint16 T;
    guint16 Z;
    guint32 LBA;
    guint16 POT;
    guint8 a[4];
} ATA_SMART_SEAGATEPLIST_ITEM_S;

typedef struct _ATA_SMART_SEAGATEPLIST {
    guint8 reserved0[16];
    ATA_SMART_SEAGATEPLIST_ITEM_S items[ATA_LOG_SEAGATE_PLIST_NUM];
} ATA_SMART_SEAGATEPLIST_S;

typedef struct _ATA_SMART_CRITICALEVENT_ITEM {
    guint16 type;           // 类型
    guint16 POT;            // 上电时间 (Hours)
    guint32 timestamp;      // 时间戳
    guint32 LBA;            // 逻辑块地址
    guint32 EC;             // 错误码
    guint16 temp;           // 温度
    guint16 unknow2;        // ATA CMD
    guint32 R;
    guint16 Z;
    guint16 T;
    guint16 index;
    guint16 reserved;
} ATA_SMART_CRITICALEVENT_ITEM_S;

typedef struct _ATA_SMART_DISKSTATUS_DESC {
    guint16 status;
    gchar* desc;
} ATA_SMART_DISKSTATUS_DESC_S;

typedef struct _ATA_SMART_CMD_DESC {
    guint8 reg;
    gchar* desc;
} ATA_SMART_CMD_DESC_S;

typedef struct _ATA_SEAGATE_FARM_HEADER {
    guint64 signature;
    guint64 majorRev;
    guint64 minorRev;
    guint64 pageNum;
    guint64 logSize;
    guint64 pageSize;
    guint64 headSupport;
    guint64 copiesNum;
    gchar reserved[16320];
} ATA_SEAGATE_FARM_HEADER_S;

typedef struct _ATA_SEAGATE_FARM_DRIVE_INFO {
    guint64 pageNum;
    guint64 copyNum;
    gchar serialNumHi[8];
    gchar serialNumLow[8];
    guint64 wwnHi;
    guint64 wwnLow;
    gchar interfaceType[8];
    guint64 capacity;
    guint64 pSectorSize;
    guint64 lSectorSize;
    guint64 bufferSize;
    guint64 headNum;
    guint64 formFactor;
    guint64 rotationRate;
    gchar fwRevHi[8];
    gchar fwRevLow[8];
    guint64 ataSecurityState;
    guint64 ataFeaturesSupported;
    guint64 ataFeaturesEnabled;
    guint64 powerOnHour;
    guint64 spindlePowerOnHour;
    guint64 headFlightHour;
    guint64 headLoadEvent;
    guint64 powerCycleCount;
    guint64 hardwareResetCount;
    guint64 spinupTime;
    guint64 NVCStatusPowerOn;
    guint64 timeAvailableToSave;
    guint64 startTimeStamp;
    guint64 endTimeStamp;
    gchar reserved[16144];
} ATA_SEAGATE_FARM_DRIVE_INFO_S;

typedef struct _ATA_SEAGATE_FARM_WORKLOAD_STAT {
    guint64 pageNum;
    guint64 copyNum;
    guint64 ratedWP;
    guint64 readCmdNum;
    guint64 writeCmdNum;
    guint64 rReadCmdNum;
    guint64 rWriteCmdNum;
    guint64 otherCmdNum;
    guint64 lSectorWritten;
    guint64 lSectorRead;
    gchar reserved[16304];
} ATA_SEAGATE_FARM_WORKLOAD_STAT_S;

typedef struct _ATA_SEAGATE_FARM_ERROR_STAT {
    guint64 pageNum;
    guint64 copyNum;
    guint64 unrecoverReadErrors;
    guint64 unrecoverWriteErrors;
    guint64 reallocatedSectors;
    guint64 readRecoveryAttempts;
    guint64 mechanicalStartFailures;
    guint64 reallocatedCandidateSectors;
    guint64 ASREvents;
    guint64 interfaceCRCErrors;
    guint64 spinRetryCount;
    guint64 spinRetryCountNormalized;
    guint64 spinRetryCountWorst;
    guint64 IOEDCErrors;
    guint64 CTOTotal;
    guint64 CTOOver5s;
    guint64 CTOOver7_5s;
    guint64 flashLEDEvents;
    guint64 flashLEDCodeAddress;
    guint64 uncorrectableErrors;
    guint64 FruCodeSmartTrip;
    guint64 infoFlashLEDEvent;
    guint64 infoReadWriteRetryEvent;
    guint64 superParityFlyRecovery;
    guint64 ReallocatedSectorsByCause;
    gchar reserved[15952];
} ATA_SEAGATE_FARM_ERROR_STAT_S;

typedef struct _ATA_SEAGATE_FARM_ENV_STAT {
    guint64 pageNum;
    guint64 copyNum;
    guint64 currentTemp;
    guint64 highestTemp;
    guint64 lowestTemp;
    guint64 avgShortTermTemp;
    guint64 avgLongTermTemp;
    guint64 highestAvgShortTermTemp;
    guint64 lowestAvgShortTermTemp;
    guint64 highestAvgLongTermTemp;
    guint64 lowestAvgLongTermTemp;
    guint64 overTempTime;
    guint64 underTempTime;
    guint64 maxOperatTemp;
    guint64 minOperatTemp;
    guint64 overLimitShockEvents;
    guint64 highFlyWriteCount;
    guint64 currentRelativeHumidity;
    guint64 humidityMixedRatio;
    guint64 currentMotorPower;
    gchar reserved[16224];
} ATA_SEAGATE_FARM_ENV_STAT_S;

typedef struct _ATA_SEAGATE_FARM_RELIA_STAT {
    guint64 pageNum;
    guint64 copyNum;
    guint64 lastIDDTimeStamp;
    guint64 lastIDDSubCmd;
    guint64 discSlip[24];
    guint64 errorRateOfZone0[24];
    guint64 GListReclamationsNum;
    guint64 servoStatus;
    guint64 altsListBeforeIDD;
    guint64 altsListAfterIDD;
    guint64 residentGlistBeforeIDD;
    guint64 residentGlistAfterIDD;
    guint64 scrubsListBeforeIDD;
    guint64 scrubsListAfterIDD;
    guint64 DOSScansPerformedNum;
    guint64 LBAsCorrectedByISPNum;
    guint64 validParitySectorsNum;
    guint64 DOSWriteRefreshCount[24];
    guint64 RAWOperationsNum;
    guint64 DVGASkipWriteDetect[24];
    guint64 RVGASkipWriteDetect[24];
    guint64 FVGASkipWriteDetect[24];
    guint64 skipWriteDetectThred[24];
    guint64 errorRateRaw;
    guint64 errorRateNormalized;
    guint64 errorRateWorst;
    guint64 seekErrorRateRaw;
    guint64 seekErrorRateNormalized;
    guint64 seekErrorRateWorst;
    guint64 highPriorityUnloadEvents;
    guint64 microActuatorLockoutAccumulated;
    guint64 ACFFSine1X[24];
    guint64 ACFFCosine1X[24];
    guint64 PZTCalibration[24];
    guint64 MRHeadResistance[24];
    guint64 TMDNum[24];
    guint64 velocityObserver[24];
    guint64 velocityObserverNum[24];
    guint64 currentH2SATTrimmedMeanBitsInError[24][3];
    guint64 currentH2SATIterationsToConverge[24][3];
    guint64 currentH2SATCodewordsAtIterationLevel[24];
    guint64 H2SATAmplitude[24];
    guint64 H2SATAsymmetry[24];
    guint64 flyheightClearanceDelta[24][3];
    guint64 discSlipRecalibrationPerformed;
    guint64 reallocatedSectorsNum[24];
    guint64 reallocationCandidateSectorsNum[24];
    guint64 heliumPressureThreshold;
    guint64 DOSOughtScansCount[24];
    guint64 DOSNeedScansCount[24];
    guint64 DOSWriteScansCount[24];
    guint64 WritePOHInSeconds[24];
    guint64 RVAbsoluteMean;
    guint64 MaxRVAbsoluteMean;
    guint64 IdleTime;
    guint64 DOSWriteCountThreshold[24];
    guint64 SecondMRHeadResistance[24];
    gchar reserved[9624];
}ATA_SEAGATE_FARM_RELIA_STAT_S;

#define ATA_GENERAL_PURPOSE_LOG_DIRECTORY_LENGTH            512

typedef union _IDE_IDENTIFY_DEVICE_T {
    guint16  word[256];      /* raw version of data */
    struct {                 /* formatted version of data */
        guint16  generalConfiguration;
        guint16  obsolete;
        guint16  specificConfiguration;
        guint16  obsolete1;
        guint16  retired1[2];
        guint16  obsolete2;
        guint16  reserved2[2];
        guint16  retired3;
        guint16  serialNumber[10];
        guint16  retired4[2];
        guint16  obsolete5;
        guint16  firmwareRevision[4];
        guint16  modelNumber[20];
        guint16  maxMultipleSize;
        guint16  reserved6;
        guint16  capabilities[2];
        guint16  obsolete6[2];
        guint16  validWords;
        guint16  obsolete7[5];
        guint16  totalUserSectors[2];
        guint16  obsolete8;
        guint16  dmaMode;
        guint16  pioModesSupported;
        guint16  minimumMultiwordDmaTransferCycleTime;
        guint16  manufacturerRecommendedMultiwordDmaTransferCycleTime;
        guint16  minumumPioTransferCycleTimeWithoutFlowControl;
        guint16  minumumPioTransferCycleTimeWithFlowControl;
        guint16  reserved9[2];
        guint16  reserved10[4];
        guint16  queueDepth;
        guint16  reserved11[4];
        guint16  majorVersionNumber;
        guint16  minorVersionNumber;
        guint16  commandSetsSupported[3];
        guint16  commandSetsEnabled[2];
        guint16  commandSetDefaults;
        guint16  ultraDmaModes;
        guint16  timeForSecurityErase;
        guint16  timeForEnahncedSecurityErase;
        guint16  currentAdvancedPowerManagementValue;
        guint16  masterPasswordRevisionCode;
        guint16  hardwareResetResult;
        guint16  acousticManagement;
        guint16  streamMinReqSize;
        guint16  streamTransferTime;
        guint16  streamAccessLatency;
        guint16  streamPerformanceGranularity[2];
        guint16  maximumLBAfor48bitAddressing[4];
        guint16  streamingTransferTime;
        guint16  reserved12;
        guint16  sectorSize;
        guint16  interSeekDelay;
        guint16  reserved13[9];
        guint16  logicalSectorWords[2];
        guint16  reserved14[8];
        guint16  removableMediaStatusNotification;
        guint16  securityStatus;
        guint16  vendorSpecific[31];
        guint16  CFApowerMode;
        guint16  reserved15[15];
        guint16  mediaSerialNumber[30];
        guint16  reserves16[49];
        guint16  integrityWord;
    } data;
} IDE_IDENTIFY_DEVICE_T;
#define IDE_IDENTIFY_DEVICE_S sizeof(IDE_IDENTIFY_DEVICE_T)

typedef struct _SCSI_SENSE_DISECT {
    guint8 error_code;
    guint8 sense_key;
    guint8 asc;
    guint8 ascq;
} SCSI_SENSE_DISECT_S;

typedef struct _SCSI_ERROR_COUNTER {
    guint8 got_pc[7];
    guint8 got_extra_pc;
    guint64 counter[8];
} SCSI_ERROR_COUNTER_S;

typedef struct _SCSI_NON_MEDIUM_ERROR {
    guint8 got_pc0;
    guint8 got_extra_pc;
    guint64 counter_pc0;
    guint8 got_tfe_h;
    guint64 counter_tfe_h; /* Track following errors [Hitachi] */
    guint8 got_pe_h;
    guint64 counter_pe_h;  /* Positioning errors [Hitachi] */
} SCSI_NON_MEDIUM_ERROR_S;

typedef struct _SCSI_ADDITIONAL_CODES_DESC {
    gchar* desc;
    guint8 asc;
    guint8 ascq;
} SCSI_ADDITIONAL_CODES_DESC_S;

// ------------模拟RAID卡诊断事件
typedef struct tag_sml_mock_ctrl_event {
    guint32 event_code;
    guint16 pd_device_id;
    guint16 encl_device_id;

    /* 硬盘状态变化事件 */
    guint8  pd_prev_state;
    guint8  pd_new_state;

    /* 硬盘sense code事件 */
    guint8  pd_sense_key;
    guint8  pd_sense_asc;
    guint8  pd_sense_ascq;
} SML_MOCK_CTRL_EVENT_S;

typedef struct PD_VENDOR_NAME_ATTR_ID {
    const gchar *vendor_name;
    guint8 attr_id;
} PD_VENDOR_NAME_ATTR_ID_S;

typedef struct PD_VENDOR_ID_VENDOR_NAME {
    guint32 vendor_id;
    const gchar *vendor_name;
} PD_VENDOR_ID_VENDOR_NAME_S;

typedef struct PD_VENDOR_PN_VENDOR_NAME {
    const gchar *vendor_pn;
    const gchar *vendor_name;
} PD_VENDOR_PN_VENDOR_NAME_S;

typedef struct _DRIVE_VENDOR_INFO_EXTEND {
    guint32 vendor_id;
    guint32 attr_id_wear;
    gchar* vendor_name;
} DRIVE_VENDOR_INFO_EXTEND_S;

#pragma pack()

#ifdef __cplusplus
#if __cplusplus
}
#endif
#endif /* __cplusplus */
#endif

