/*
 * Copyright (c) Huawei Technologies Co., Ltd. 2025-2025. All rights reserved.
 *
 * this file licensed under the Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 * http://license.coscl.org.cn/MulanPSL2
 *
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
 * PURPOSE.
 * See the Mulan PSL v2 for more details.
 *
 */

#include "platform.h"
#include <unistd.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <stdio.h>
#include "utils/file_securec.h"

#define USEC_PER_SEC 1000000L

#define MS_PER_SECOND 1000
#define US_PER_MILIISECOND 1000
#define MS_PER_HARF_SECOND 500

#define CRC_TABLE_LEN 256
#define EIGHT_BIT_OFFSET 8

#define TMP_PATH_WITH_SLASH     "/tmp/"
#define TMP_PATH_NO_SLASH       "/tmp"
#define SSL_INVALID_DATA 0x00020004              // 无效数据
#define SSL_NO_DATA 0x0002000F                   // 无数据,数据为空

#define MAX_FILENAME_LENGTH 256

#define DO1(buf) crcWorkVar = g_crc_table_32[((guint)crcWorkVar ^ (*(buf)++)) & 0xff] ^ (crcWorkVar >> EIGHT_BIT_OFFSET)

#define DO2(buf) do {   \
    DO1(buf);           \
    DO1(buf);           \
} while (0)

#define DO4(buf) do {   \
    DO2(buf);           \
    DO2(buf);           \
} while (0)

#define DO8(buf) do {   \
    DO4(buf);           \
    DO4(buf);           \
} while (0)

/* CRC 数值表 */
const unsigned long g_crc_table_32[CRC_TABLE_LEN] = {
    0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L, 0x706af48fL,
    0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L, 0xe0d5e91eL, 0x97d2d988L,
    0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L, 0x90bf1d91L, 0x1db71064L, 0x6ab020f2L,
    0xf3b97148L, 0x84be41deL, 0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L,
    0x136c9856L, 0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
    0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L, 0xa2677172L,
    0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL, 0x35b5a8faL, 0x42b2986cL,
    0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L, 0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L,
    0x26d930acL, 0x51de003aL, 0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L,
    0xcfba9599L, 0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
    0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L, 0x01db7106L,
    0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL, 0x9fbfe4a5L, 0xe8b8d433L,
    0x7807c9a2L, 0x0f00f934L, 0x9609a88eL, 0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL,
    0x91646c97L, 0xe6635c01L, 0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL,
    0x6c0695edL, 0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
    0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L, 0xfbd44c65L,
    0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L, 0x4adfa541L, 0x3dd895d7L,
    0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL, 0x346ed9fcL, 0xad678846L, 0xda60b8d0L,
    0x44042d73L, 0x33031de5L, 0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL,
    0xbe0b1010L, 0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
    0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L, 0x2eb40d81L,
    0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L, 0x03b6e20cL, 0x74b1d29aL,
    0xead54739L, 0x9dd277afL, 0x04db2615L, 0x73dc1683L, 0xe3630b12L, 0x94643b84L,
    0x0d6d6a3eL, 0x7a6a5aa8L, 0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L,
    0xf00f9344L, 0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
    0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL, 0x67dd4accL,
    0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L, 0xd6d6a3e8L, 0xa1d1937eL,
    0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L, 0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL,
    0xd80d2bdaL, 0xaf0a1b4cL, 0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L,
    0x316e8eefL, 0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
    0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL, 0xb2bd0b28L,
    0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L, 0x2cd99e8bL, 0x5bdeae1dL,
    0x9b64c2b0L, 0xec63f226L, 0x756aa39cL, 0x026d930aL, 0x9c0906a9L, 0xeb0e363fL,
    0x72076785L, 0x05005713L, 0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L,
    0x92d28e9bL, 0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
    0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L, 0x18b74777L,
    0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL, 0x8f659effL, 0xf862ae69L,
    0x616bffd3L, 0x166ccf45L, 0xa00ae278L, 0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L,
    0xa7672661L, 0xd06016f7L, 0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL,
    0x40df0b66L, 0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
    0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L, 0xcdd70693L,
    0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L, 0x5d681b02L, 0x2a6f2b94L,
    0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL, 0x2d02ef8dL
};

void vos_task_delay(guint32 millisecond)
{
    if (millisecond < MS_PER_SECOND) {
        (void)usleep(millisecond * US_PER_MILIISECOND);
    } else {
        (void)sleep((millisecond + MS_PER_HARF_SECOND) / MS_PER_SECOND);
    }
}

gulong arith_make_crc32(gulong crc, const guint8 *buf, gulong len)
{
    gulong crcWorkVar = crc;
    gulong lenWorkVar = len;
    if (buf == NULL) {
        return 0;
    }
    crcWorkVar = crc ^ 0xffffffffL;
    while (lenWorkVar >= EIGHT_BIT_OFFSET) {
        DO8(buf);
        lenWorkVar -= EIGHT_BIT_OFFSET;
    }

    if (lenWorkVar) {
        do {
            DO1(buf);
        } while (--lenWorkVar);
    }

    return crcWorkVar ^ 0xffffffffL;
}

static guint64 vos_get_sys_us_count(void)
{
    struct timeval CurrentTime;

    if (gettimeofday(&CurrentTime, NULL) == -1) {
        g_print("get time fail: file %s -line %d\r\n", __FILE__, __LINE__);
        return 0;
    }

    return (guint64)((gint64)(CurrentTime.tv_sec) * USEC_PER_SEC + CurrentTime.tv_usec);
}

gulong vos_get_cur_time_stamp(void)
{
    return (gulong)time(0);
}

/*
 * Description: 获取当地系统时间
 * History：2017年8月7日  新生成函数
 */
gint32 dal_localtime_r(const time_t *p_time_stamp, struct tm *p_tm)
{
    time_t time_stamp = 0;
    struct tm *p = NULL;

    if (p_tm == NULL) {
        return RET_ERR;
    }

    time_stamp = (p_time_stamp == NULL) ? (time_t)vos_get_cur_time_stamp() : *p_time_stamp;
    tzset();
    p = localtime_r(&time_stamp, p_tm); /* 取得当地时间 */
    if (p == NULL) {
        return RET_ERR;
    }

    return RET_OK;
}

gulong vos_tick_get(void)
{
    gulong jiffies = 0;

#ifdef ARM_PLATFORM

    /* 用CPU的跑了多少个时钟周期才是准确的 */
    (void)vos_get_sys_jiffies(&jiffies);
    return (gulong)(10 * jiffies); /* 10: 返回ms单位计数 */

#else

    jiffies = (gulong)(vos_get_sys_us_count() / 1000); /* 1000: 返回ms单位计数 */
    return (gulong)jiffies;

#endif
}

/* ****************************************************************************
 函 数 名  : vos_get_file_dir
 功能描述  : 获取文件所在目录;
             本函数不破坏传入的原文件路径，成功获取的目录存放在filepath中
             filepath的内存管理由函数调用者负责
 输入参数  : gchar *filename,gchar *filepath：文件名，文件目录
 输出参数  : 无
**************************************************************************** */
gint32 vos_get_file_dir(const gchar *filename, gchar *filepath, guint32 path_len)
{
    size_t len;
    gint32 ret;

    if ((filename == NULL) || (filepath == NULL)) {
        return FALSE;
    }

    len = strlen(filename);
    if (path_len <= len) {
        g_printf("length of filepath is too short\n");
        return FALSE;
    }

    // 从后面开始查找 到"/"为止,如果传入的是目录则返回当前值
    for (; len > 0; len--) {
        if (filename[len - 1] == '/') {
            ret = memmove_s(filepath, path_len, filename, len);
            if (ret != VOS_OK) {
                g_printf("[%s][%d]:memmove_s failed(%d).\n", __FUNCTION__, __LINE__, ret);
            }
            filepath[len] = 0;
            return TRUE;
        }
    }

    return FALSE;
}

/*
 * Description: 清除字符串中空格
 * History: 1.2016年12月19日,  新生成函数
 */
gchar *string_clear_blank(gchar *string_src, guint32 string_src_size)
{
    gchar *dst = string_src;
    gchar *src = string_src;
    guint32 count = 0;

    if (NULL == string_src) {
        return NULL;
    }

    while (*src != '\0' && count < string_src_size) {
        if (' ' != *src) {
            *dst = *src;
            dst++;
        }
        count++;
        src++;
    }

    *dst = '\0';

    return string_src;
}

/*
 * Description: 由于json-c开源软件归一，取消自研补丁，json_object_get_string函数默认进行反斜杠转义
 *      特封装本函数，取消反斜杠转义，json_object_get_string统一使用这个函数替代
 *      使用白泽Json替换json-c
 */
const gchar* dal_json_object_get_str(Json *obj)
{
    if (JsonIsNull(obj) == true ||  JsonIsString(obj) == true) {
        gchar* str_value = NULL;
        JsonItemStringValueGet(obj, &str_value);
        return str_value;
    }

    return JsonPrint(obj);
}
