/* Copyright (c) Huawei Technologies Co., Ltd. 2025-2025. All rights reserved.
 * openUBMC is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

#include "platform.h"
#include "sml.h"
#include "sml_errcodes.h"
#include "pd_log_process.h"
#include "sas_ie_log_parse.h"

static SCSI_ADDITIONAL_CODES_DESC_S g_additional_error[] = {
    {"No additional sense information",                                    0x00, 0x00},
    {"Filemark detected",                                                  0x00, 0x01},
    {"End-of-partition/medium detected",                                   0x00, 0x02},
    {"Setmark detected",                                                   0x00, 0x03},
    {"Beginning-of-partition/medium detected",                             0x00, 0x04},
    {"End-of-data detected",                                               0x00, 0x05},
    {"I/O process terminated",                                             0x00, 0x06},
    {"Audio play operation in progress",                                   0x00, 0x11},
    {"Audio play operation paused",                                        0x00, 0x12},
    {"Audio play operation successfully completed",                        0x00, 0x13},
    {"Audio play operation stopped due to error",                          0x00, 0x14},
    {"No current audio status to return",                                  0x00, 0x15},
    {"operation in progress",                                              0x00, 0x16},
    {"Cleaning requested",                                                 0x00, 0x17},
    {"Erase operation in progress",                                        0x00, 0x18},
    {"Locate operation in progress",                                       0x00, 0x19},
    {"Rewind operation in progress",                                       0x00, 0x1a},
    {"Set capacity operation in progress",                                 0x00, 0x1b},
    {"Verify operation in progress",                                       0x00, 0x1c},
    {"ATA pass through information available",                             0x00, 0x1d},
    {"No index/sector signal",                                             0x01, 0x00},
    {"No seek complete",                                                   0x02, 0x00},
    {"Peripheral device write fault",                                      0x03, 0x00},
    {"No write current",                                                   0x03, 0x01},
    {"Excessive write errors",                                             0x03, 0x02},
    {"Logical unit not ready, cause not reportable",                       0x04, 0x00},
    {"Logical unit is in process of becoming ready",                       0x04, 0x01},
    {"Logical unit not ready, initializing command required",              0x04, 0x02},
    {"Logical unit not ready, manual intervention required",               0x04, 0x03},
    {"Logical unit not ready, format in progress",                         0x04, 0x04},
    {"Logical unit not ready, rebuild in progress",                        0x04, 0x05},
    {"Logical unit not ready, recalculation in progress",                  0x04, 0x06},
    {"Logical unit not ready, operation in progress",                      0x04, 0x07},
    {"Logical unit not ready, long write in progress",                     0x04, 0x08},
    {"Logical unit not ready, self-test in progress",                      0x04, 0x09},
    {"Logical unit not accessible, asymmetric access state transition",    0x04, 0x0a},
    {"Logical unit not accessible, target port in standby state",          0x04, 0x0b},
    {"Logical unit not accessible, target port in unavailable state",      0x04, 0x0c},
    {"Logical unit not ready, auxiliary memory not accessible",            0x04, 0x10},
    {"Logical unit not ready, notify (enable spinup) required",            0x04, 0x11},
    {"Logical unit not ready, offline",                                    0x04, 0x12},
    {"Logical unit does not respond to selection",                         0x05, 0x00},
    {"No reference position found",                                        0x06, 0x00},
    {"Multiple peripheral devices selected",                               0x07, 0x00},
    {"Logical unit communication failure",                                 0x08, 0x00},
    {"Logical unit communication time-out",                                0x08, 0x01},
    {"Logical unit communication parity error",                            0x08, 0x02},
    {"Logical unit communication CRC error (Ultra-DMA/32)",                0x08, 0x03},
    {"Unreachable copy target",                                            0x08, 0x04},
    {"Track following error",                                              0x09, 0x00},
    {"Tracking servo failure",                                             0x09, 0x01},
    {"Focus servo failure",                                                0x09, 0x02},
    {"Spindle servo failure",                                              0x09, 0x03},
    {"Head select fault",                                                  0x09, 0x04},
    {"Error log overflow",                                                 0x0A, 0x00},
    {"Warning",                                                            0x0B, 0x00},
    {"Warning - specified temperature exceeded",                           0x0B, 0x01},
    {"Warning - enclosure degraded",                                       0x0B, 0x02},
    {"Warning - background self-test failed",                              0x0B, 0x03},
    {"Warning - background pre-scan detected medium error",                0x0B, 0x04},
    {"Warning - background medium scan detected medium error",             0x0B, 0x05},
    {"Write error",                                                        0x0C, 0x00},
    {"Write error - recovered with auto reallocation",                     0x0C, 0x01},
    {"Write error - auto reallocation failed",                             0x0C, 0x02},
    {"Write error - recommend reassignment",                               0x0C, 0x03},
    {"Compression check miscompare error",                                 0x0C, 0x04},
    {"Data expansion occurred during compression",                         0x0C, 0x05},
    {"Block not compressible",                                             0x0C, 0x06},
    {"Write error - recovery needed",                                      0x0C, 0x07},
    {"Write error - recovery failed",                                      0x0C, 0x08},
    {"Write error - loss of streaming",                                    0x0C, 0x09},
    {"Write error - padding blocks added",                                 0x0C, 0x0A},
    {"Auxiliary memory write error",                                       0x0C, 0x0B},
    {"Write error - unexpected unsolicited data",                          0x0C, 0x0C},
    {"Write error - not enough unsolicited data",                          0x0C, 0x0D},
    {"Defects in error window",                                            0x0C, 0x0F},
    {"Error detected by third party temporary initiator",                  0x0D, 0x00},
    {"Third party device failure",                                         0x0D, 0x01},
    {"Copy target device not reachable",                                   0x0D, 0x02},
    {"Incorrect copy target device type",                                  0x0D, 0x03},
    {"Copy target device data underrun",                                   0x0D, 0x04},
    {"Copy target device data overrun",                                    0x0D, 0x05},
    {"Invalid information unit",                                           0x0E, 0x00},
    {"Information unit too short",                                         0x0E, 0x01},
    {"Information unit too long",                                          0x0E, 0x02},
    {"Invalid field in command information unit",                          0x0E, 0x03},
    {"Id CRC or ECC error",                                                0x10, 0x00},
    {"Logical block guard check failed",                                   0x10, 0x01},
    {"Logical block application tag check failed",                         0x10, 0x02},
    {"Logical block reference tag check failed",                           0x10, 0x03},
    {"Unrecovered read error",                                             0x11, 0x00},
    {"Read retries exhausted",                                             0x11, 0x01},
    {"Error too long to correct",                                          0x11, 0x02},
    {"Multiple read errors",                                               0x11, 0x03},
    {"Unrecovered read error - auto reallocate failed",                    0x11, 0x04},
    {"L-EC uncorrectable error",                                           0x11, 0x05},
    {"CIRC unrecovered error",                                             0x11, 0x06},
    {"Data re-synchronization error",                                      0x11, 0x07},
    {"Incomplete block read",                                              0x11, 0x08},
    {"No gap found",                                                       0x11, 0x09},
    {"Miscorrected error",                                                 0x11, 0x0A},
    {"Unrecovered read error - recommend reassignment",                    0x11, 0x0B},
    {"Unrecovered read error - recommend rewrite the data",                0x11, 0x0C},
    {"De-compression CRC error",                                           0x11, 0x0D},
    {"Cannot decompress using declared algorithm",                         0x11, 0x0E},
    {"Error reading UPC/EAN number",                                       0x11, 0x0F},
    {"Error reading ISRC number",                                          0x11, 0x10},
    {"Read error - loss of streaming",                                     0x11, 0x11},
    {"Auxiliary memory read error",                                        0x11, 0x12},
    {"Read error - failed retransmission request",                         0x11, 0x13},
    {"Read error - LBA marked bad by application client",                  0x11, 0x14},
    {"Address mark not found for id field",                                0x12, 0x00},
    {"Address mark not found for data field",                              0x13, 0x00},
    {"Recorded entity not found",                                          0x14, 0x00},
    {"Record not found",                                                   0x14, 0x01},
    {"Filemark or setmark not found",                                      0x14, 0x02},
    {"End-of-data not found",                                              0x14, 0x03},
    {"Block sequence error",                                               0x14, 0x04},
    {"Record not found - recommend reassignment",                          0x14, 0x05},
    {"Record not found - data auto-reallocated",                           0x14, 0x06},
    {"Locate operation failure",                                           0x14, 0x07},
    {"Random positioning error",                                           0x15, 0x00},
    {"Mechanical positioning error",                                       0x15, 0x01},
    {"Positioning error detected by read of medium",                       0x15, 0x02},
    {"Data synchronization mark error",                                    0x16, 0x00},
    {"Data sync error - data rewritten",                                   0x16, 0x01},
    {"Data sync error - recommend rewrite",                                0x16, 0x02},
    {"Data sync error - data auto-reallocated",                            0x16, 0x03},
    {"Data sync error - recommend reassignment",                           0x16, 0x04},
    {"Recovered data with no error correction applied",                    0x17, 0x00},
    {"Recovered data with retries",                                        0x17, 0x01},
    {"Recovered data with positive head offset",                           0x17, 0x02},
    {"Recovered data with negative head offset",                           0x17, 0x03},
    {"Recovered data with retries and/or circ applied",                    0x17, 0x04},
    {"Recovered data using previous sector id",                            0x17, 0x05},
    {"Recovered data without ECC - data auto-reallocated",                 0x17, 0x06},
    {"Recovered data without ECC - recommend reassignment",                0x17, 0x07},
    {"Recovered data without ECC - recommend rewrite",                     0x17, 0x08},
    {"Recovered data without ECC - data rewritten",                        0x17, 0x09},
    {"Recovered data with error correction applied",                       0x18, 0x00},
    {"Recovered data with error corr. & retries applied",                  0x18, 0x01},
    {"Spindle impending failure throughput performance",                   0x5D, 0x59},
    {"Spindle impending failure seek time performance",                    0x5D, 0x5A},
    {"Spindle impending failure spin-up retry count",                      0x5D, 0x5B},
    {"Spindle impending failure drive calibration retry count",            0x5D, 0x5C},
    {"Firmware impending failure general hard drive failure",              0x5D, 0x60},
    {"Firmware impending failure drive error rate too high",               0x5D, 0x61},
    {"Recovered data - data auto-reallocated",                             0x18, 0x02},
    {"Recovered data with CIRC",                                           0x18, 0x03},
    {"Recovered data with L-EC",                                           0x18, 0x04},
    {"Recovered data - recommend reassignment",                            0x18, 0x05},
    {"Recovered data - recommend rewrite",                                 0x18, 0x06},
    {"Recovered data with ECC - data rewritten",                           0x18, 0x07},
    {"Recovered data with linking",                                        0x18, 0x08},
    {"Defect list error",                                                  0x19, 0x00},
    {"Defect list not available",                                          0x19, 0x01},
    {"Defect list error in primary list",                                  0x19, 0x02},
    {"Defect list error in grown list",                                    0x19, 0x03},
    {"Parameter list length error",                                        0x1A, 0x00},
    {"Synchronous data transfer error",                                    0x1B, 0x00},
    {"Defect list not found",                                              0x1C, 0x00},
    {"Primary defect list not found",                                      0x1C, 0x01},
    {"Grown defect list not found",                                        0x1C, 0x02},
    {"Miscompare during verify operation",                                 0x1D, 0x00},
    {"Recovered id with ECC correction",                                   0x1E, 0x00},
    {"Partial defect list transfer",                                       0x1F, 0x00},
    {"Invalid command operation code",                                     0x20, 0x00},
    {"Access denied - initiator pending-enrolled",                         0x20, 0x01},
    {"Access denied - no access rights",                                   0x20, 0x02},
    {"Access denied - invalid mgmt id key",                                0x20, 0x03},
    {"Illegal command while in write capable state",                       0x20, 0x04},
    {"Write type operation while in read capable state (obs)",             0x20, 0x05},
    {"Illegal command while in explicit address mode",                     0x20, 0x06},
    {"Illegal command while in implicit address mode",                     0x20, 0x07},
    {"Access denied - enrollment conflict",                                0x20, 0x08},
    {"Access denied - invalid LU identifier",                              0x20, 0x09},
    {"Access denied - invalid proxy token",                                0x20, 0x0A},
    {"Access denied - ACL LUN conflict",                                   0x20, 0x0B},
    {"Logical block address out of range",                                 0x21, 0x00},
    {"Invalid element address",                                            0x21, 0x01},
    {"Invalid address for write",                                          0x21, 0x02},
    {"Invalid write crossing layer jump",                                  0x21, 0x03},
    {"Servo impending failure drive calibration retry count",              0x5D, 0x4C},
    {"Spindle impending failure general hard drive failure",               0x5D, 0x50},
    {"Spindle impending failure drive error rate too high",                0x5D, 0x51},
    {"Firmware impending failure data error rate too high",                0x5D, 0x62},
    {"Illegal function (use 20 00, 24 00, or 26 00)",                      0x22, 0x00},
    {"Invalid field in cdb",                                               0x24, 0x00},
    {"CDB decryption error",                                               0x24, 0x01},
    {"Invalid cdb field while in explicit block model (obs)",              0x24, 0x02},
    {"Invalid cdb field while in implicit block model (obs)",              0x24, 0x03},
    {"Security audit value frozen",                                        0x24, 0x04},
    {"Security working key frozen",                                        0x24, 0x05},
    {"Nonce not unique",                                                   0x24, 0x06},
    {"Nonce timestamp out of range",                                       0x24, 0x07},
    {"Logical unit not supported",                                         0x25, 0x00},
    {"Invalid field in parameter list",                                    0x26, 0x00},
    {"Parameter not supported",                                            0x26, 0x01},
    {"Parameter value invalid",                                            0x26, 0x02},
    {"Threshold parameters not supported",                                 0x26, 0x03},
    {"Invalid release of persistent reservation",                          0x26, 0x04},
    {"Data decryption error",                                              0x26, 0x05},
    {"Too many target descriptors",                                        0x26, 0x06},
    {"Unsupported target descriptor type code",                            0x26, 0x07},
    {"Too many segment descriptors",                                       0x26, 0x08},
    {"Unsupported segment descriptor type code",                           0x26, 0x09},
    {"Unexpected inexact segment",                                         0x26, 0x0A},
    {"Inline data length exceeded",                                        0x26, 0x0B},
    {"Invalid operation for copy source or destination",                   0x26, 0x0C},
    {"Copy segment granularity violation",                                 0x26, 0x0D},
    {"Invalid parameter while port is enabled",                            0x26, 0x0E},
    {"Invalid data-out buffer integrity check value",                      0x26, 0x0F},
    {"Data decryption key fail limit reached",                             0x26, 0x10},
    {"Incomplete key-associated data set",                                 0x26, 0x11},
    {"Vendor specific key reference not found",                            0x26, 0x12},
    {"Data channel impending failure controller detected",                 0x5D, 0x38},
    {"Data channel impending failure throughput performance",              0x5D, 0x39},
    {"Data channel impending failure seek time performance",               0x5D, 0x3A},
    {"Servo impending failure access times too high",                      0x5D, 0x45},
    {"Servo impending failure start unit times too high",                  0x5D, 0x46},
    {"Servo impending failure channel parametrics",                        0x5D, 0x47},
    {"Write protected",                                                    0x27, 0x00},
    {"Hardware write protected",                                           0x27, 0x01},
    {"Logical unit software write protected",                              0x27, 0x02},
    {"Associated write protect",                                           0x27, 0x03},
    {"Persistent write protect",                                           0x27, 0x04},
    {"Permanent write protect",                                            0x27, 0x05},
    {"Conditional write protect",                                          0x27, 0x06},
    {"Not ready to ready change, medium may have changed",                 0x28, 0x00},
    {"Import or export element accessed",                                  0x28, 0x01},
    {"Format-layer may have changed",                                      0x28, 0x02},
    {"Power on, reset, or bus device reset occurred",                      0x29, 0x00},
    {"Power on occurred",                                                  0x29, 0x01},
    {"SCSI bus reset occurred",                                            0x29, 0x02},
    {"Bus device reset function occurred",                                 0x29, 0x03},
    {"Device internal reset",                                              0x29, 0x04},
    {"Transceiver mode changed to single-ended",                           0x29, 0x05},
    {"Transceiver mode changed to lvd",                                    0x29, 0x06},
    {"I_T nexus loss occurred",                                            0x29, 0x07},
    {"Parameters changed",                                                 0x2A, 0x00},
    {"Mode parameters changed",                                            0x2A, 0x01},
    {"Log parameters changed",                                             0x2A, 0x02},
    {"Reservations preempted",                                             0x2A, 0x03},
    {"Reservations released",                                              0x2A, 0x04},
    {"Registrations preempted",                                            0x2A, 0x05},
    {"Asymmetric access state changed",                                    0x2A, 0x06},
    {"Implicit asymmetric access state transition failed",                 0x2A, 0x07},
    {"Priority changed",                                                   0x2A, 0x08},
    {"Capacity data has changed",                                          0x2A, 0x09},
    {"Timestamp changed",                                                  0x2A, 0x10},
    {"Data encryption parameters changed by another i_t nexus",            0x2A, 0x11},
    {"Data encryption parameters changed by vendor specific event",        0x2A, 0x12},
    {"Data encryption key instance counter has changed",                   0x2A, 0x13},
    {"Copy cannot execute since host cannot disconnect",                   0x2B, 0x00},
    {"Command sequence error",                                             0x2C, 0x00},
    {"Too many windows specified",                                         0x2C, 0x01},
    {"Invalid combination of windows specified",                           0x2C, 0x02},
    {"Current program area is not empty",                                  0x2C, 0x03},
    {"Current program area is empty",                                      0x2C, 0x04},
    {"Illegal power condition request",                                    0x2C, 0x05},
    {"Persistent prevent conflict",                                        0x2C, 0x06},
    {"Previous busy status",                                               0x2C, 0x07},
    {"Previous task set full status",                                      0x2C, 0x08},
    {"Previous reservation conflict status",                               0x2C, 0x09},
    {"Partition or collection contains user objects",                      0x2C, 0x0A},
    {"Not reserved",                                                       0x2C, 0x0B},
    {"Overwrite error on update in place",                                 0x2D, 0x00},
    {"Insufficient time for operation",                                    0x2E, 0x00},
    {"Commands cleared by another initiator",                              0x2F, 0x00},
    {"Commands cleared by power loss notification",                        0x2F, 0x01},
    {"Commands cleared by device server",                                  0x2F, 0x02},
    {"Incompatible medium installed",                                      0x30, 0x00},
    {"Cannot read medium - unknown format",                                0x30, 0x01},
    {"Cannot read medium - incompatible format",                           0x30, 0x02},
    {"Cleaning cartridge installed",                                       0x30, 0x03},
    {"Cannot write medium - unknown format",                               0x30, 0x04},
    {"Cannot write medium - incompatible format",                          0x30, 0x05},
    {"Cannot format medium - incompatible medium",                         0x30, 0x06},
    {"Cleaning failure",                                                   0x30, 0x07},
    {"Cannot write - application code mismatch",                           0x30, 0x08},
    {"Current session not fixated for append",                             0x30, 0x09},
    {"Cleaning request rejected",                                          0x30, 0x0A},
    {"Cleaning tape expired",                                              0x30, 0x0B},
    {"WORM medium - overwrite attempted",                                  0x30, 0x0C},
    {"WORM medium - integrity check",                                      0x30, 0x0D},
    {"Medium not formatted",                                               0x30, 0x10},
    {"Medium format corrupted",                                            0x31, 0x00},
    {"Format command failed",                                              0x31, 0x01},
    {"Zoned formatting failed due to spare linking",                       0x31, 0x02},
    {"No defect spare location available",                                 0x32, 0x00},
    {"Defect list update failure",                                         0x32, 0x01},
    {"Tape length error",                                                  0x33, 0x00},
    {"Enclosure failure",                                                  0x34, 0x00},
    {"Enclosure services failure",                                         0x35, 0x00},
    {"Unsupported enclosure function",                                     0x35, 0x01},
    {"Enclosure services unavailable",                                     0x35, 0x02},
    {"Enclosure services transfer failure",                                0x35, 0x03},
    {"Enclosure services transfer refused",                                0x35, 0x04},
    {"Enclosure services checksum error",                                  0x35, 0x05},
    {"Ribbon, ink, or toner failure",                                      0x36, 0x00},
    {"Rounded parameter",                                                  0x37, 0x00},
    {"Event status notification",                                          0x38, 0x00},
    {"Esn - power management class event",                                 0x38, 0x02},
    {"Esn - media class event",                                            0x38, 0x04},
    {"Esn - device busy class event",                                      0x38, 0x06},
    {"Saving parameters not supported",                                    0x39, 0x00},
    {"Medium not present",                                                 0x3A, 0x00},
    {"Medium not present - tray closed",                                   0x3A, 0x01},
    {"Medium not present - tray open",                                     0x3A, 0x02},
    {"Data channel impending failure spin-up retry count",                 0x5D, 0x3B},
    {"Data channel impending failure drive calibration retry count",       0x5D, 0x3C},
    {"Servo impending failure general hard drive failure",                 0x5D, 0x40},
    {"Servo impending failure drive error rate too high",                  0x5D, 0x41},
    {"Servo impending failure data error rate too high",                   0x5D, 0x42},
    {"Servo impending failure seek error rate too high",                   0x5D, 0x43},
    {"Servo impending failure too many block reassigns",                   0x5D, 0x44},
    {"Medium not present - loadable",                                      0x3A, 0x03},
    {"Medium not present - medium auxiliary memory accessible",            0x3A, 0x04},
    {"Sequential positioning error",                                       0x3B, 0x00},
    {"Tape position error at beginning-of-medium",                         0x3B, 0x01},
    {"Tape position error at end-of-medium",                               0x3B, 0x02},
    {"Tape or electronic vertical forms unit not ready",                   0x3B, 0x03},
    {"Slew failure",                                                       0x3B, 0x04},
    {"Paper jam",                                                          0x3B, 0x05},
    {"Failed to sense top-of-form",                                        0x3B, 0x06},
    {"Failed to sense bottom-of-form",                                     0x3B, 0x07},
    {"Reposition error",                                                   0x3B, 0x08},
    {"Read past end of medium",                                            0x3B, 0x09},
    {"Read past beginning of medium",                                      0x3B, 0x0A},
    {"Position past end of medium",                                        0x3B, 0x0B},
    {"Position past beginning of medium",                                  0x3B, 0x0C},
    {"Medium destination element full",                                    0x3B, 0x0D},
    {"Medium source element empty",                                        0x3B, 0x0E},
    {"End of medium reached",                                              0x3B, 0x0F},
    {"Medium magazine not accessible",                                     0x3B, 0x11},
    {"Medium magazine removed",                                            0x3B, 0x12},
    {"Medium magazine inserted",                                           0x3B, 0x13},
    {"Medium magazine locked",                                             0x3B, 0x14},
    {"Medium magazine unlocked",                                           0x3B, 0x15},
    {"Mechanical positioning or changer error",                            0x3B, 0x16},
    {"Read past end of user object",                                       0x3B, 0x17},
    {"Invalid bits in identify message",                                   0x3D, 0x00},
    {"Logical unit has not self-configured yet",                           0x3E, 0x00},
    {"Logical unit failure",                                               0x3E, 0x01},
    {"Timeout on logical unit",                                            0x3E, 0x02},
    {"Logical unit failed self-test",                                      0x3E, 0x03},
    {"Logical unit unable to update self-test log",                        0x3E, 0x04},
    {"Target operating conditions have changed",                           0x3F, 0x00},
    {"Microcode has been changed",                                         0x3F, 0x01},
    {"Changed operating definition",                                       0x3F, 0x02},
    {"Inquiry data has changed",                                           0x3F, 0x03},
    {"Component device attached",                                          0x3F, 0x04},
    {"Device identifier changed",                                          0x3F, 0x05},
    {"Redundancy group created or modified",                               0x3F, 0x06},
    {"Redundancy group deleted",                                           0x3F, 0x07},
    {"Spare created or modified",                                          0x3F, 0x08},
    {"Spare deleted",                                                      0x3F, 0x09},
    {"Volume set created or modified",                                     0x3F, 0x0A},
    {"Volume set deleted",                                                 0x3F, 0x0B},
    {"Volume set deassigned",                                              0x3F, 0x0C},
    {"Volume set reassigned",                                              0x3F, 0x0D},
    {"Reported luns data has changed",                                     0x3F, 0x0E},
    {"Echo buffer overwritten",                                            0x3F, 0x0F},
    {"Medium loadable",                                                    0x3F, 0x10},
    {"Medium auxiliary memory accessible",                                 0x3F, 0x11},
    {"iSCSI IP address added",                                             0x3F, 0x12},
    {"iSCSI IP address removed",                                           0x3F, 0x13},
    {"iSCSI IP address changed",                                           0x3F, 0x14},
    {"Ram failure (should use 40 nn)",                                     0x40, 0x00},
    {"Data path failure (should use 40 nn)",                               0x41, 0x00},
    {"Power-on or self-test failure (should use 40 nn)",                   0x42, 0x00},
    {"Message error",                                                      0x43, 0x00},
    {"Internal target failure",                                            0x44, 0x00},
    {"ATA device failed Set Features",                                     0x44, 0x71},
    {"Hardware impending failure too many block reassigns",                0x5D, 0x14},
    {"Hardware impending failure access times too high",                   0x5D, 0x15},
    {"Select or reselect failure",                                         0x45, 0x00},
    {"Unsuccessful soft reset",                                            0x46, 0x00},
    {"SCSI parity error",                                                  0x47, 0x00},
    {"Data phase CRC error detected",                                      0x47, 0x01},
    {"SCSI parity error detected during st data phase",                    0x47, 0x02},
    {"Information unit iuCRC error detected",                              0x47, 0x03},
    {"Asynchronous information protection error detected",                 0x47, 0x04},
    {"Protocol service CRC error",                                         0x47, 0x05},
    {"Phy test function in progress",                                      0x47, 0x06},
    {"Some commands cleared by iSCSI protocol event",                      0x47, 0x7F},
    {"Initiator detected error message received",                          0x48, 0x00},
    {"Invalid message error",                                              0x49, 0x00},
    {"Command phase error",                                                0x4A, 0x00},
    {"Data phase error",                                                   0x4B, 0x00},
    {"Invalid target port transfer tag received",                          0x4B, 0x01},
    {"Too much write data",                                                0x4B, 0x02},
    {"Ack/nak timeout",                                                    0x4B, 0x03},
    {"Nak received",                                                       0x4B, 0x04},
    {"Data offset error",                                                  0x4B, 0x05},
    {"Initiator response timeout",                                         0x4B, 0x06},
    {"Logical unit failed self-configuration",                             0x4C, 0x00},
    {"Firmware impending failure channel parametrics",                     0x5D, 0x67},
    {"Firmware impending failure controller detected",                     0x5D, 0x68},
    {"Firmware impending failure throughput performance",                  0x5D, 0x69},
    {"Controller impending failure drive error rate too high",             0x5D, 0x21},
    {"Hardware impending failure general hard drive failure",              0x5D, 0x10},
    {"Hardware impending failure drive error rate too high",               0x5D, 0x11},
    /*
     * ASC 0x4D overridden by an "additional2" array entry
     * so there is no need to have them here.
     */

    {"Overlapped commands attempted",                                      0x4E, 0x00},
    {"Write append error",                                                 0x50, 0x00},
    {"Write append position error",                                        0x50, 0x01},
    {"Position error related to timing",                                   0x50, 0x02},

    {"Hardware impending failure data error rate too high",                0x5D, 0x12},
    {"Hardware impending failure seek error rate too high",                0x5D, 0x13},

    {"Erase failure",                                                      0x51, 0x00},
    {"Erase failure - incomplete erase operation detected",                0x51, 0x01},

    {"Hardware impending failure start unit times too high",               0x5D, 0x16},
    {"Hardware impending failure channel parametrics",                     0x5D, 0x17},

    {"Cartridge fault",                                                    0x52, 0x00},
    {"Media load or eject failed",                                         0x53, 0x00},
    {"Unload tape failure",                                                0x53, 0x01},
    {"Medium removal prevented",                                           0x53, 0x02},
    {"Medium removal prevented by data transfer element",                  0x53, 0x03},
    {"Medium thread or unthread failure",                                  0x53, 0x04},
    {"SCSI to host system interface failure",                              0x54, 0x00},

    {"Firmware impending failure seek time performance",                   0x5D, 0x6A},
    {"Firmware impending failure spin-up retry count",                     0x5D, 0x6B},
    {"Firmware impending failure drive calibration retry count",           0x5D, 0x6C},
    {"Controller impending failure general hard drive failure",            0x5D, 0x20},

    {"System resource failure",                                            0x55, 0x00},
    {"System buffer full",                                                 0x55, 0x01},
    {"Insufficient reservation resources",                                 0x55, 0x02},
    {"Insufficient resources",                                             0x55, 0x03},
    {"Insufficient registration resources",                                0x55, 0x04},
    {"Insufficient access control resources",                              0x55, 0x05},
    {"Auxiliary memory out of space",                                      0x55, 0x06},
    {"Quota error",                                                        0x55, 0x07},
    {"Maximum number of supplemental decryption keys exceeded",            0x55, 0x08},
    {"Unable to recover table-of-contents",                                0x57, 0x00},
    {"Generation does not exist",                                          0x58, 0x00},
    {"Updated block read",                                                 0x59, 0x00},
    {"Operator request or state change input",                             0x5A, 0x00},
    {"Operator medium removal request",                                    0x5A, 0x01},
    {"Operator selected write protect",                                    0x5A, 0x02},
    {"Operator selected write permit",                                     0x5A, 0x03},
    {"Log exception",                                                      0x5B, 0x00},
    {"Threshold condition met",                                            0x5B, 0x01},
    {"Log counter at maximum",                                             0x5B, 0x02},
    {"Log list codes exhausted",                                           0x5B, 0x03},
    {"Rpl status change",                                                  0x5C, 0x00},
    {"Spindles synchronized",                                              0x5C, 0x01},
    {"Spindles not synchronized",                                          0x5C, 0x02},
    {"Failure prediction threshold exceeded",                              0x5D, 0x00},
    {"Hardware impending failure seek time performance",                   0x5D, 0x1A},
    {"Hardware impending failure spin-up retry count",                     0x5D, 0x1B},
    {"Hardware impending failure drive calibration retry count",           0x5D, 0x1C},
    {"Media failure prediction threshold exceeded",                        0x5D, 0x01},
    {"Logical unit failure prediction threshold exceeded",                 0x5D, 0x02},
    {"spare area exhaustion prediction threshold exceeded",                0x5D, 0x03},
    {"Hardware impending failure controller detected",                     0x5D, 0x18},
    {"Hardware impending failure throughput performance",                  0x5D, 0x19},
    {"Controller impending failure channel parametrics",                   0x5D, 0x27},
    {"Controller impending failure drive calibration retry count",         0x5D, 0x2C},

    {"Data channel impending failure start unit times too high",           0x5D, 0x36},

    {"Servo impending failure throughput performance",                     0x5D, 0x49},
    {"Servo impending failure seek time performance",                      0x5D, 0x4A},

    {"Firmware impending failure access times too high",                   0x5D, 0x65},
    {"Firmware impending failure start unit times too high",               0x5D, 0x66},

    {"Failure prediction threshold exceeded (false)",                      0x5D, 0xFF},
    {"Low power condition on",                                             0x5E, 0x00},
    {"Idle condition activated by timer",                                  0x5E, 0x01},
    {"Standby condition activated by timer",                               0x5E, 0x02},
    {"Idle condition activated by command",                                0x5E, 0x03},
    {"Standby condition activated by command",                             0x5E, 0x04},
    {"Power state change to active",                                       0x5E, 0x41},
    {"Power state change to idle",                                         0x5E, 0x42},
    {"Power state change to standby",                                      0x5E, 0x43},
    {"Power state change to sleep",                                        0x5E, 0x45},
    {"Power state change to device control",                               0x5E, 0x47},
    {"Lamp failure",                                                       0x60, 0x00},
    {"Video acquisition error",                                            0x61, 0x00},
    {"Unable to acquire video",                                            0x61, 0x01},
    {"Out of focus",                                                       0x61, 0x02},
    {"Scan head positioning error",                                        0x62, 0x00},
    {"End of user area encountered on this track",                         0x63, 0x00},
    {"Packet does not fit in available space",                             0x63, 0x01},
    {"Illegal mode for this track",                                        0x64, 0x00},
    {"Invalid packet size",                                                0x64, 0x01},
    {"Voltage fault",                                                      0x65, 0x00},
    {"Automatic document feeder cover up",                                 0x66, 0x00},
    {"Automatic document feeder lift up",                                  0x66, 0x01},
    {"Document jam in automatic document feeder",                          0x66, 0x02},
    {"Document miss feed automatic in document feeder",                    0x66, 0x03},
    {"Configuration failure",                                              0x67, 0x00},
    {"Configuration of incapable logical units failed",                    0x67, 0x01},
    {"Add logical unit failed",                                            0x67, 0x02},
    {"Modification of logical unit failed",                                0x67, 0x03},
    {"Exchange of logical unit failed",                                    0x67, 0x04},
    {"Remove of logical unit failed",                                      0x67, 0x05},
    {"Attachment of logical unit failed",                                  0x67, 0x06},
    {"Creation of logical unit failed",                                    0x67, 0x07},
    {"Assign failure occurred",                                            0x67, 0x08},
    {"Multiply assigned logical unit",                                     0x67, 0x09},
    {"Set target port groups command failed",                              0x67, 0x0A},

    {"Controller impending failure data error rate too high",              0x5D, 0x22},
    {"Controller impending failure seek error rate too high",              0x5D, 0x23},
    {"Controller impending failure too many block reassigns",              0x5D, 0x24},
    {"Controller impending failure access times too high",                 0x5D, 0x25},
    {"Controller impending failure start unit times too high",             0x5D, 0x26},

    {"ATA device feature not enabled",                                     0x67, 0x0B},
    {"Logical unit not configured",                                        0x68, 0x00},
    {"Data loss on logical unit",                                          0x69, 0x00},
    {"Multiple logical unit failures",                                     0x69, 0x01},
    {"Parity/data mismatch",                                               0x69, 0x02},
    {"Informational, refer to log",                                        0x6A, 0x00},
    {"State change has occurred",                                          0x6B, 0x00},
    {"Data channel impending failure general hard drive failure",          0x5D, 0x30},
    {"Data channel impending failure drive error rate too high",           0x5D, 0x31},
    {"Data channel impending failure data error rate too high",            0x5D, 0x32},
    {"Data channel impending failure seek error rate too high",            0x5D, 0x33},
    {"Data channel impending failure too many block reassigns",            0x5D, 0x34},
    {"Data channel impending failure access times too high",               0x5D, 0x35},

    {"Redundancy level got better",                                        0x6B, 0x01},
    {"Redundancy level got worse",                                         0x6B, 0x02},
    {"Rebuild failure occurred",                                           0x6C, 0x00},
    {"Recalculate failure occurred",                                       0x6D, 0x00},
    {"Command to logical unit failed",                                     0x6E, 0x00},
    {"Copy protection key exchange failure - authentication failure",      0x6F, 0x00},
    {"Copy protection key exchange failure - key not present",             0x6F, 0x01},
    {"Copy protection key exchange failure - key not established",         0x6F, 0x02},
    {"Read of scrambled sector without authentication",                    0x6F, 0x03},
    {"Media region code is mismatched to logical unit region",             0x6F, 0x04},
    {"Drive region must be permanent/region reset count error",            0x6F, 0x05},
    {"Insufficient block count for binding nonce recording",               0x6F, 0x06},
    {"Conflict in binding nonce recording",                                0x6F, 0x07},

    /*
     * ASC 0x70 overridden by an "additional2" array entry
     * so there is no need to have them here.
     */

    {"Decompression exception long algorithm id",                          0x71, 0x00},
    {"Session fixation error",                                             0x72, 0x00},
    {"Session fixation error writing lead-in",                             0x72, 0x01},
    {"Session fixation error writing lead-out",                            0x72, 0x02},
    {"Session fixation error - incomplete track in session",               0x72, 0x03},
    {"Empty or partially written reserved track",                          0x72, 0x04},
    {"No more track reservations allowed",                                 0x72, 0x05},
    {"RMZ extension is not allowed",                                       0x72, 0x06},

    {"Servo impending failure spin-up retry count",                        0x5D, 0x4B},
    {"Data channel impending failure channel parametrics",                 0x5D, 0x37},

    {"Servo impending failure controller detected",                        0x5D, 0x48},
    {"Firmware impending failure seek error rate too high",                0x5D, 0x63},
    {"Firmware impending failure too many block reassigns",                0x5D, 0x64},

    {"No more test zone extensions are allowed",                           0x72, 0x07},
    {"CD control error",                                                   0x73, 0x00},
    {"Power calibration area almost full",                                 0x73, 0x01},
    {"Power calibration area is full",                                     0x73, 0x02},
    {"Power calibration area error",                                       0x73, 0x03},
    {"Program memory area update failure",                                 0x73, 0x04},
    {"Program memory area is full",                                        0x73, 0x05},
    {"RMA/PMA is almost full",                                             0x73, 0x06},
    {"Current power calibration area almost full",                         0x73, 0x10},
    {"Current power calibration area is full",                             0x73, 0x11},

    {"Controller impending failure controller detected",                   0x5D, 0x28},
    {"Controller impending failure throughput performance",                0x5D, 0x29},
    {"Controller impending failure seek time performance",                 0x5D, 0x2A},
    {"Controller impending failure spin-up retry count",                   0x5D, 0x2B},

    {"RDZ is full",                                                        0x73, 0x17},
    {"Security error",                                                     0x74, 0x00},
    {"Unable to decrypt data",                                             0x74, 0x01},
    {"Unencrypted data encountered while decrypting",                      0x74, 0x02},
    {"Incorrect data encryption key",                                      0x74, 0x03},
    {"Cryptographic integrity validation failed",                          0x74, 0x04},
    {"Error decrypting data",                                              0x74, 0x05},
    {"Unknown signature verification key",                                 0x74, 0x06},

    {"Spindle impending failure data error rate too high",                 0x5D, 0x52},
    {"Spindle impending failure seek error rate too high",                 0x5D, 0x53},
    {"Spindle impending failure too many block reassigns",                 0x5D, 0x54},
    {"Spindle impending failure access times too high",                    0x5D, 0x55},
    {"Spindle impending failure start unit times too high",                0x5D, 0x56},
    {"Spindle impending failure channel parametrics",                      0x5D, 0x57},
    {"Spindle impending failure controller detected",                      0x5D, 0x58},

    {"Encryption parameters not useable",                                  0x74, 0x07},
    {"Digital signature validation failure",                               0x74, 0x08},
    {"Logical unit access not authorized",                                 0x74, 0x71},

    {"Firmware impending failure too many block reassign",                 0x5D, 0xFC},
    {"Controller impending failure hard drive failure",                    0x5D, 0xFD},
    {"Hardware impending failure throughput performance",                  0x5D, 0xFE},
    {NULL, 0, 0}
};

#define VENDOR_IE_LOG_STRING_LEN 128

typedef struct vendor_ie_log {
    guint8 attr_id;
    const gchar name_str[VENDOR_IE_LOG_STRING_LEN];
    const gchar key_str[VENDOR_IE_LOG_STRING_LEN];
} VENDOR_IE_LOG;

// samsung与Huawei相同或类似的统计项，数据采集字段命名参考Huawei，其他采用首字母缩写
static VENDOR_IE_LOG g_samsung_vendor_ie_log_map[] = {
    {0x00, "Most Recent Temperature Reading",        "NA"},
    {0x01, "Raw Read Error Rate",                    "rr_err_rate_cur"},
    {0x09, "Power On Hours",                         "POT"},
    {0x0C, "Power Cycle Count",                      "DPCC"},
    {0xAD, "Worst Wear Leveling Count Block",        "WWLC"},
    {0xB2, "Used Spare LBA Count (in core)",         "USLBAC"},
    {0xB4, "Read Fail Block Count (in total)",       "BRFC"},
    {0xB5, "Program Fail Block Count (in total)",    "BPFC"},
    {0xB6, "Erase Fail Block Count (in total)",      "BEFC"},
    {0xBB, "Uncorrectable ECC Errors Count",         "UNC_EC"},
    {0xBE, "Temperature Exceed Count",               "TE"},
    {0xC2, "Maximum Temperature",                    "HT"},
    {0xC7, "Link CRC Error Count",                   "Link_CRC"},
    {0xC8, "Power Loss Protection Capacitor Status", "Cstatus"},
    {0xC9, "Backup Power Supply Status",             "BPSstatus"},
    {0xCA, "Exception Mode Status",                  "EMstatus"},
    {0xE9, "Host Write Count",                       "host_write_block"},
    {0xEA, "Host Read Count",                        "host_read_block"},
    {0xEB, "Physical Write Count",                   "phy_write_block"},
    {0xEC, "Corrected DRAM ECC Errors",              "CE"}
};

/*
 * Description : 处理huawei SAS SSD IE log vendor specific信息
 * History：2020-11-17  新生成函数
 */
static void pd_log_handle_sas_huawei_ie_log(FILE *file, PD_LOG_S *pd_log, void *out_put, Json *pd_log_jso)
{
    gint32 i, page_len, param_code;
    guint64 param_data, temp_value;
    guchar *p_data = NULL;
    const gint8 ie_param0_size = 8;  // parameter code 0 有8个字节
    const gint8 ie_param1_size = 12; // parameter code 1 有12个字节

    p_data = pd_log->SASDevice.ie.data;
    page_len = (p_data[2] << 8) + p_data[3]; // 2、3字节为IE Log数据长度
    if (page_len < (ie_param0_size + ie_param1_size)) {
        pd_log_fprintf(file, "there is no informational exceptions vendor Specific log.\n");
        return;
    }

    p_data = pd_log->SASDevice.ie.data + SCSI_LOG_PAGE_HEADER_SIZE + ie_param0_size;

    param_code = (p_data[0] << 8) + p_data[1]; // 0、1字节为parameter code
    if (param_code == 1) {
        param_data = 0;
        for (i = 4; i < ie_param1_size; i++) {
            temp_value = param_data;
            temp_value = temp_value << 8;
            param_data = temp_value + p_data[i];
        }
        pd_log_fprintf(file, "  Bad block count: %d\n", param_code, param_data);
    }

    return;
}

/*
 * Description : 获取三星 SAS SSD硬盘SMART属性名称
 * History：2020-11-17  新生成函数
 */
static const gchar *get_sas_ssd_samsung_smart_attr_name(guint8 attr_id)
{
    gint32 i;
    for (i = 0; i < G_N_ELEMENTS(g_samsung_vendor_ie_log_map); i++) {
        if (attr_id == g_samsung_vendor_ie_log_map[i].attr_id) {
            return g_samsung_vendor_ie_log_map[i].name_str;
        }
    }

    return "Unknown_Attribute";
}

/*
 * Description : 获取三星 SAS SSD硬盘SMART属性数据采集字段名称
 * History：2020-11-17  新生成函数
 */
static const gchar *get_sas_ssd_samsung_smart_key_str(guint8 attr_id)
{
    gint32 i;
    for (i = 0; i < G_N_ELEMENTS(g_samsung_vendor_ie_log_map); i++) {
        if (attr_id == g_samsung_vendor_ie_log_map[i].attr_id) {
            return g_samsung_vendor_ie_log_map[i].key_str;
        }
    }

    return "Unknown";
}

/*
 * Description : 处理sumsung SAS SSD IE log vendor specific数据
 * History：2020-11-17  新生成函数
 */
static void pd_log_handle_sas_samsung_ie_log_data(FILE *file, const guchar *p_data, void *out_put,
    Json *pd_log_jso)
{
    gint32 i, param_code;
    guint64 param_data, temp_value;
    const gint8 ie_param_size = 12; // 每个vendor log数据长度12字节
    gint32 factory_value, power_on_value, worst_value, power_on_temperature;
    gchar tmp_str[TMP_STR_LEN] = {0};

    param_code = (p_data[0] << 8) + p_data[1]; // 0、1字节为parameter code
    param_data = 0;
    for (i = 4; i < ie_param_size; i++) {
        temp_value = param_data;
        temp_value = temp_value << 8;
        param_data = temp_value + p_data[i];
    }

    // 0xC8包含四项数据，数据解析后再记录log
    if (param_code == 0xC8) {
        factory_value = (p_data[4] << 8) + p_data[5];
        pd_log_fprintf(file, "0x%02x(%3d)   %s(factory value)         0x%04x\n", param_code, param_code,
            get_sas_ssd_samsung_smart_attr_name(param_code), factory_value);

        power_on_value = (p_data[6] << 8) + p_data[7];
        pd_log_fprintf(file, "0x%02x(%3d)   %s(power on value)        0x%04x\n", param_code, param_code,
            get_sas_ssd_samsung_smart_attr_name(param_code), power_on_value);

        worst_value = (p_data[8] << 8) + p_data[9];
        pd_log_fprintf(file, "0x%02x(%3d)   %s(worst value)           0x%04x\n", param_code, param_code,
            get_sas_ssd_samsung_smart_attr_name(param_code), worst_value);

        power_on_temperature = (p_data[10] << 8) + p_data[11];
        pd_log_fprintf(file, "0x%02x(%3d)   %s(power on temperature)  0x%04x\n", param_code, param_code,
            get_sas_ssd_samsung_smart_attr_name(param_code), power_on_temperature);
    } else {
        pd_log_fprintf(file, "0x%02x(%3d)   %-60s  0x%016" G_GUINT64_HEX_FORMAT "\n", param_code, param_code,
            get_sas_ssd_samsung_smart_attr_name(param_code), param_data);
    }

    // 数据采集不特殊处理0xC8统计项，由数据使用者进行数据解析
    (void)snprintf_s(tmp_str, TMP_STR_LEN, TMP_STR_LEN - 1, "%" G_GUINT64_FORMAT, param_data);
    Json *json_string_obj = NULL;
    JsonStringCreate(tmp_str, &json_string_obj);
    JsonObjectItemSet(pd_log_jso, get_sas_ssd_samsung_smart_key_str(param_code), json_string_obj);

    return;
}

/*
 * Description : 处理sumsung SAS SSD IE log vendor specific信息
 * History：2020-11-17  新生成函数
 */
static void pd_log_handle_sas_samsung_ie_log(FILE *file, PD_LOG_S *pd_log, void *out_put, Json *pd_log_jso)
{
    gint32 i, page_len;
    guchar *p_data = NULL;
    gint32 ie_vendor_param_cnt;     // IE log中vendor log条目梳理
    const gint8 ie_param_size = 12; // 每个vendor log数据长度12字节
    Json *samsumg_ie_log_jso = NULL;

    p_data = pd_log->SASDevice.ie.data;
    page_len = (p_data[2] << 8) + p_data[3]; // 2、3字节为IE Log数据长度
    ie_vendor_param_cnt = (page_len - ie_param_size) / ie_param_size;
    if (ie_vendor_param_cnt < 1) {
        pd_log_fprintf(file, "there is no informational exceptions vendor Specific log.\n");
        return;
    }

    if (JsonObjectCreate(&samsumg_ie_log_jso) != JSON_OK) {
        debug_log(DLOG_ERROR, "%s: JsonObjectCreate failed.\n", __FUNCTION__);
        return;
    }

    pd_log_fprintf(file, "Informational Exceptions Log vendor specific:\n");
    pd_log_fprintf(file, "ID#         Attribute Name                                                Vlaue\n");
    p_data = pd_log->SASDevice.ie.data + SCSI_LOG_PAGE_HEADER_SIZE + ie_param_size;

    for (i = 0; i < ie_vendor_param_cnt; i++) {
        pd_log_handle_sas_samsung_ie_log_data(file, p_data, out_put, samsumg_ie_log_jso);
        p_data += ie_param_size;
    }

    if (pd_log_jso != NULL) {
        JsonObjectItemSet(pd_log_jso, "vendor_specific_log", samsumg_ie_log_jso);
    } else {
        JsonObjectRelease(samsumg_ie_log_jso);
    }

    return;
}

/*
 * Description : 处理SAS SSD IE log vendor specific信息
 * History：2020-11-17  新生成函数
 */
static void pd_log_handle_sas_vendor_ie_log(FILE *file, PD_LOG_S *pd_log, void *out_put, Json *pd_log_jso)
{
    // RAMAXEL和UMIS 与HUAWEI协议一致
    if (strstr(pd_log->pd_manufacturer, PD_VENDOR_NAME_HUAWEI) != NULL ||
        strstr(pd_log->pd_manufacturer, PD_VENDOR_NAME_RAMAXEL) != NULL ||
        strstr(pd_log->pd_manufacturer, PD_VENDOR_NAME_UMIS) != NULL) {
        pd_log_handle_sas_huawei_ie_log(file, pd_log, out_put, pd_log_jso);
    } else if (strstr(pd_log->pd_manufacturer, PD_VENDOR_NAME_SAMSUNG) != NULL) {
        pd_log_handle_sas_samsung_ie_log(file, pd_log, out_put, pd_log_jso);
    }
}

/*
 * Description : 校验IE Log数据
 * History：2021-1-12  新生成函数
 */
static gint32 check_ie_log_data(FILE *file, PD_LOG_S *pd_log)
{
    guint8 *ie_log_data = NULL;

    if ((pd_log->SASDevice.ie.result != SML_SUCCESS && pd_log->SASDevice.ie.result != SML_ERR_PD_SCSI_RESP_TRUNCATED) ||
        (pd_log->SASDevice.ie.data == NULL) || (pd_log->SASDevice.ie.data_length == 0)) {
        return RET_ERR;
    }

    if (pd_log->SASDevice.ie.data_length - SCSI_LOG_PAGE_HEADER_SIZE < SCSI_LOG_PAGE_HEADER_SIZE) {
        pd_log_fprintf(file, "Badly formed Informational Exceptions log page\n");
        return RET_ERR;
    }

    ie_log_data = pd_log->SASDevice.ie.data;
    if ((ie_log_data[0] & 0x3f) != SCSI_LOG_PAGE_IE) {
        pd_log_fprintf(file, "Informational Exceptions Log Sense Failed, page mismatch\n");
        return RET_ERR;
    }

    return RET_OK;
}

/*
 * Description : 根据asc进行硬盘故障诊断
 * History：2021-1-12  新生成函数
 */
static gint32 pd_diagnose_by_asc(PD_LOG_S *pd_log, void *out_put, gint32 asc)
{
    SML_PD_FAULT_ANALYSIS *result = (SML_PD_FAULT_ANALYSIS *)out_put;
    const gint32 threshold = 0x5D;

    if (result != NULL && threshold == asc) {
        result->o_fault_bitmap.pd_smart_log_error = 1;
        (void)snprintf_s(result->io_buf, sizeof(result->io_buf), sizeof(result->io_buf) - 1, "asc is 0x%02x", asc);
        debug_log(DLOG_INFO, "IE log (asc = 0x%02x) error of %s.", asc, pd_log->pd_device_name);
        return RET_ERR;
    }

    return RET_OK;
}

/*
 * Description : 将IE log asc和ascq写入json
 * History：2021-1-12  新生成函数
 */
static void put_asc_and_ascq_to_json(gint32 asc, gint32 ascq, Json *pd_log_jso)
{
    gchar tmp_str[TMP_STR_LEN] = {0};
    Json *ie_log_node = NULL;
    if (JsonObjectCreate(&ie_log_node) != JSON_OK) {
        return;
    }

    (void)memset_s(tmp_str, TMP_STR_LEN, 0, TMP_STR_LEN);
    (void)snprintf_s(tmp_str, TMP_STR_LEN, TMP_STR_LEN - 1, "0x%02x", asc);
    Json *json_string_obj = NULL;
    JsonStringCreate(tmp_str, &json_string_obj);
    JsonObjectItemSet(ie_log_node, "asc", json_string_obj);
    (void)memset_s(tmp_str, TMP_STR_LEN, 0, TMP_STR_LEN);
    (void)snprintf_s(tmp_str, TMP_STR_LEN, TMP_STR_LEN - 1, "0x%02x", ascq);
    Json *json_ascq_string_obj = NULL;
    JsonStringCreate(tmp_str, &json_ascq_string_obj);
    JsonObjectItemSet(ie_log_node, "ascq", json_ascq_string_obj);
    guint32 size = 0;
    JsonArraySizeGet(pd_log_jso, &size);
    JsonArrayItemInsert(pd_log_jso, size, ie_log_node);
}

/*
 * Description : 处理IE log数据
 * History：2021-1-12  新生成函数
 */
static gint32 pd_log_handle_sas_ie_log_data(FILE *file, PD_LOG_S *pd_log, void *out_put, Json *ie_log_array)
{
    guint8 ucSign = 0;
    gint32 i = 0, j;
    guint32 extra = 0;
    gint32 tmp = 0;
    gint32 asc = 0;
    gint32 ascq = 0;
    SCSI_ADDITIONAL_CODES_DESC_S *eip = NULL;
    gint32 flag = 0;

    guchar *p_data = pd_log->SASDevice.ie.data + SCSI_LOG_PAGE_HEADER_SIZE;    // 指向IE parameter data;
    gint32 num = (gint32)(pd_log->SASDevice.ie.data_length - SCSI_LOG_PAGE_HEADER_SIZE); // IE parameter data长度,去掉头

    for (i = num; i > 0; i -= (gint32)extra) {
        extra = p_data[3] + SCSI_LOG_PAGE_HEADER_SIZE; // 获取IE parameter data长度
        tmp = (p_data[0] << 8) + p_data[1];

        flag = (tmp != 0 || extra <= 5);
        if (flag) {
            continue;
        }

        asc = p_data[4];
        ascq = p_data[5];

        // 诊断故障
        if (pd_diagnose_by_asc(pd_log, out_put, asc) != RET_OK) {
            JsonObjectRelease(ie_log_array);
            return RET_ERR;
        }

        for (j = 0; g_additional_error[j].desc; ++j) {
            eip = &g_additional_error[j];
            flag = (eip->asc == asc) && (eip->ascq == ascq);
            if (flag) {
                pd_log_fprintf(file, "  IE asc = 0x%02x ascq = 0x%02x\t%s\n", asc, ascq, eip->desc);
                ucSign = 1;
                break;
            }
        }

        if (ucSign == 0) {
            pd_log_fprintf(file, "  IE asc = 0x%02x ascq = 0x%02x     Unknown sense information\n", asc, ascq);
        }

        put_asc_and_ascq_to_json(asc, ascq, ie_log_array);

        if (extra < pd_log->SASDevice.ie.data_length) {
            p_data += extra;
        }
    }

    return RET_OK;
}

/*
 * Description : 处理Informational Exceptions Log数据
 * History：2018年5月11日  新生成函数
 * 2021年1月12日   SAS SSD日志数据采集
 */
void pd_log_handle_sas_ie_log(FILE *file, PD_LOG_S *pd_log, void *out_put, Json *pd_log_jso)
{
    if (check_ie_log_data(file, pd_log) != RET_OK) {
        return;
    }

    Json *ie_log_array = NULL;
    if (JsonArrayCreate(&ie_log_array) != JSON_OK) {
        return;
    }

    if (pd_log_handle_sas_ie_log_data(file, pd_log, out_put, ie_log_array) != RET_OK) {
        return;
    }

    if (pd_log_jso != NULL) {
        JsonObjectItemSet(pd_log_jso, "informational_exceptions_log", ie_log_array);
    } else {
        JsonObjectRelease(ie_log_array);
    }

    if (pd_log->pd_media_type == PD_MEDIA_TYPE_SSD) {
        pd_log_handle_sas_vendor_ie_log(file, pd_log, out_put, pd_log_jso);
    }

    return;
}
