-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local ipmi = require 'ipmi'
local mdb = require 'mc.mdb'
local skynet = require 'skynet'
local class = require 'mc.class'
local context = require 'mc.context'
local open_db = require 'storage.db'
local app_base = require 'mc.service_app_base'
local open_local_db = require 'storage.local_db'
local object_manage = require 'mc.mdb.object_manage'
local persist_client = require 'persistence.persist_client_lib'
local orm_classes = require 'storage.orm_classes'
local ok, datas = pcall(require, 'storage.datas')
if not ok then
    datas = nil -- 如果没有datas配置，证明当前组件不需要datas，仅打开数据库
end

local StorageConfigTypes = require 'storage.json_types.StorageConfig'
local StorageDiagnoseTypes = require 'storage.json_types.StorageDiagnose'
local PhyBitErrorTypes = require 'storage.json_types.PhyBitError'
local RecordDriveSpareBlockTypes = require 'storage.json_types.RecordDriveSpareBlock'
local ReleaseStorageControllerTypes = require 'storage.json_types.ReleaseStorageController'
local ReleaseStorageDriveTypes = require 'storage.json_types.ReleaseStorageDrive'
local PropertiesTypes = require 'mdb.bmc.kepler.Object.PropertiesInterface'
local SystemsStorageControllerTypes = require 'storage.json_types.SystemsStorageController'
local DriverTypes = require 'storage.json_types.Driver'
local ChipTypes = require 'storage.json_types.Chip'
local ConsistencyCheckTypes = require 'storage.json_types.ConsistencyCheck'
local VolumeManageTypes = require 'storage.json_types.VolumeManage'
local ControllerStatusTypes = require 'storage.json_types.ControllerStatus'
local BatteryStatusTypes = require 'storage.json_types.BatteryStatus'
local RetirementTypes = require 'storage.json_types.Retirement'
local HardwareTypes = require 'storage.json_types.Hardware'
local PhyErrorTypes = require 'storage.json_types.PhyError'
local DiskArrayTypes = require 'storage.json_types.DiskArray'
local SystemsStorageDriveTypes = require 'storage.json_types.SystemsStorageDrive'
local AddrInfoTypes = require 'storage.json_types.AddrInfo'
local DriveStatusTypes = require 'storage.json_types.DriveStatus'
local SASSMARTTypes = require 'storage.json_types.SASSMART'
local MetricTypes = require 'storage.json_types.Metric'
local DriveSubHealthTypes = require 'storage.json_types.DriveSubHealth'
local NVMeTypes = require 'storage.json_types.NVMe'
local NVMeSMARTTypes = require 'storage.json_types.NVMeSMART'
local DriveDiagnoseTypes = require 'storage.json_types.DriveDiagnose'
local DrivesTypes = require 'storage.json_types.Drives'
local ReplicaDriveTypes = require 'storage.json_types.ReplicaDrive'
local VolumeTypes = require 'storage.json_types.Volume'
local BatteryTypes = require 'storage.json_types.Battery'
local SASMetricsTypes = require 'storage.json_types.SASMetrics'
local DiskPartitionTypes = require 'storage.json_types.DiskPartition'
local StorageSettingTypes = require 'storage.json_types.StorageSetting'

local StorageConfig = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage', {
    {name = 'bmc.kepler.Systems.Storage.StorageConfig', interface = StorageConfigTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Diagnose', interface = StorageDiagnoseTypes.interface},
    {name = 'bmc.kepler.Debug.Storage.PhyBitError', interface = PhyBitErrorTypes.interface},
    {name = 'bmc.kepler.Debug.Storage.RecordDriveSpareBlock', interface = RecordDriveSpareBlockTypes.interface},
    {name = 'bmc.kepler.Release.Storage.Controller', interface = ReleaseStorageControllerTypes.interface},
    {name = 'bmc.kepler.Release.Storage.Drive', interface = ReleaseStorageDriveTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function StorageConfig:ctor(SystemId)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage'
end

local Controller = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/Controllers/:Id', {
    {name = 'bmc.kepler.Systems.Storage.Controller', interface = SystemsStorageControllerTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Controller.Driver', interface = DriverTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Controller.Chip', interface = ChipTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Controller.ConsistencyCheck', interface = ConsistencyCheckTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Controller.VolumeManage', interface = VolumeManageTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Controller.ControllerStatus', interface = ControllerStatusTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Controller.BatteryStatus', interface = BatteryStatusTypes.interface},
    {name = 'bmc.kepler.LifeCycle.Retirement', interface = RetirementTypes.interface},
    {name = 'bmc.kepler.Inventory.Hardware', interface = HardwareTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Controller:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. Id
end

local SASPhy = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/Controllers/:Id1/SASPhy/:Id2', {
    {name = 'bmc.kepler.Systems.Storage.PhyError', interface = PhyErrorTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function SASPhy:ctor(SystemId, Id1, Id2)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. Id1 .. '/SASPhy/' .. Id2
end

local DiskArray = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/Controllers/:ControllerId/DiskArrays/:Id',
    {
        {name = 'bmc.kepler.Systems.Storage.DiskArray', interface = DiskArrayTypes.interface},
        {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
    })

function DiskArray:ctor(SystemId, ControllerId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. ControllerId .. '/DiskArrays/' .. Id
end

local Drive = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/Drives/:Id', {
    {name = 'bmc.kepler.Systems.Storage.Drive', interface = SystemsStorageDriveTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Drive.AddrInfo', interface = AddrInfoTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Drive.DriveStatus', interface = DriveStatusTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Drive.SAS.SMART', interface = SASSMARTTypes.interface},
    {name = 'bmc.kepler.Metric', interface = MetricTypes.interface},
    {name = 'bmc.kepler.Inventory.Hardware', interface = HardwareTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Drive.DriveSubHealth', interface = DriveSubHealthTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Drive.NVMe', interface = NVMeTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Drive.NVMe.SMART', interface = NVMeSMARTTypes.interface},
    {name = 'bmc.kepler.Systems.Storage.Drive.Diagnose', interface = DriveDiagnoseTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Drive:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Drives/' .. Id
end

local Drives = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/Drives', {
    {name = 'bmc.kepler.Systems.Storage.Drives', interface = DrivesTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Drives:ctor(SystemId)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Drives'
end

local ReplicaDrive = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/ReplicaDrives/:Id', {
    {name = 'bmc.kepler.Systems.Storage.ReplicaDrive', interface = ReplicaDriveTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function ReplicaDrive:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/ReplicaDrives/' .. Id
end

local Volume = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/Controllers/:ControllerId/Volumes/:Id', {
    {name = 'bmc.kepler.Systems.Storage.Volume', interface = VolumeTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Volume:ctor(SystemId, ControllerId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. ControllerId .. '/Volumes/' .. Id
end

local Battery = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/Controllers/:Id/Battery', {
    {name = 'bmc.kepler.Systems.Storage.Battery', interface = BatteryTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Battery:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. Id .. '/Battery'
end

local Port = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/Controllers/:ControllerId/Ports/:Id', {
    {name = 'bmc.kepler.Systems.Storage.SASMetrics', interface = SASMetricsTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Port:ctor(SystemId, ControllerId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. ControllerId .. '/Ports/' .. Id
end

local DiskPartition = mdb.register_object('/bmc/kepler/Systems/:SystemId/Storage/DiskPartitions/:Id', {
    {name = 'bmc.kepler.Systems.Storage.DiskPartition', interface = DiskPartitionTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DiskPartition:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/DiskPartitions/' .. Id
end

local StorageSetting = mdb.register_object('/bmc/kepler/Chassis/:ChassisId/Storage', {
    {name = 'bmc.kepler.Chassis.Storage.StorageSetting', interface = StorageSettingTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function StorageSetting:ctor(ChassisId)
    self.path = '/bmc/kepler/Chassis/' .. ChassisId .. '/Storage'
end

local model = require 'class.model'

local storage_service = class(app_base.Service)

storage_service.package = 'storage'

function storage_service:CreateStorageConfig(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage'
    return object_manage.create_object('StorageConfig', path, path, prop_setting_cb)
end

function storage_service:CreateController(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. Id
    return object_manage.create_object('Controller', path, path, prop_setting_cb)
end

function storage_service:CreateSASPhy(SystemId, Id1, Id2, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. Id1 .. '/SASPhy/' .. Id2
    return object_manage.create_object('SASPhy', path, path, prop_setting_cb)
end

function storage_service:CreateDiskArray(SystemId, ControllerId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. ControllerId .. '/DiskArrays/' .. Id
    return object_manage.create_object('DiskArray', path, path, prop_setting_cb)
end

function storage_service:CreateDrive(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Drives/' .. Id
    return object_manage.create_object('Drive', path, path, prop_setting_cb)
end

function storage_service:CreateDrives(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Drives'
    return object_manage.create_object('Drives', path, path, prop_setting_cb)
end

function storage_service:CreateReplicaDrive(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/ReplicaDrives/' .. Id
    return object_manage.create_object('ReplicaDrive', path, path, prop_setting_cb)
end

function storage_service:CreateVolume(SystemId, ControllerId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. ControllerId .. '/Volumes/' .. Id
    return object_manage.create_object('Volume', path, path, prop_setting_cb)
end

function storage_service:CreateBattery(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. Id .. '/Battery'
    return object_manage.create_object('Battery', path, path, prop_setting_cb)
end

function storage_service:CreatePort(SystemId, ControllerId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/Controllers/' .. ControllerId .. '/Ports/' .. Id
    return object_manage.create_object('Port', path, path, prop_setting_cb)
end

function storage_service:CreateDiskPartition(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Storage/DiskPartitions/' .. Id
    return object_manage.create_object('DiskPartition', path, path, prop_setting_cb)
end

function storage_service:CreateStorageSetting(ChassisId, prop_setting_cb)
    local path = '/bmc/kepler/Chassis/' .. ChassisId .. '/Storage'
    return object_manage.create_object('StorageSetting', path, path, prop_setting_cb)
end

function storage_service:ImplStorageConfigPhyBitErrorMockData(cb)
    model.ImplStorageConfigPhyBitErrorMockData(cb)
end

function storage_service:ImplStorageConfigRecordDriveSpareBlockMockRecordSpareBlock(cb)
    model.ImplStorageConfigRecordDriveSpareBlockMockRecordSpareBlock(cb)
end

function storage_service:ImplStorageConfigReleaseStorageControllerGetControllerInfo(cb)
    model.ImplStorageConfigReleaseStorageControllerGetControllerInfo(cb)
end

function storage_service:ImplStorageConfigReleaseStorageDriveGetDriveDetails(cb)
    model.ImplStorageConfigReleaseStorageDriveGetDriveDetails(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetBootDevices(cb)
    model.ImplControllerSystemsStorageControllerSetBootDevices(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetJBODState(cb)
    model.ImplControllerSystemsStorageControllerSetJBODState(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetSmarterCopyBackState(cb)
    model.ImplControllerSystemsStorageControllerSetSmarterCopyBackState(cb)
end

function storage_service:ImplControllerSystemsStorageControllerRestoreDefaultSettings(cb)
    model.ImplControllerSystemsStorageControllerRestoreDefaultSettings(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetCopybackState(cb)
    model.ImplControllerSystemsStorageControllerSetCopybackState(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetMaintainPDFailHistoryState(cb)
    model.ImplControllerSystemsStorageControllerSetMaintainPDFailHistoryState(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetWorkMode(cb)
    model.ImplControllerSystemsStorageControllerSetWorkMode(cb)
end

function storage_service:ImplControllerSystemsStorageControllerImportForeignConfig(cb)
    model.ImplControllerSystemsStorageControllerImportForeignConfig(cb)
end

function storage_service:ImplControllerSystemsStorageControllerDumpLog(cb)
    model.ImplControllerSystemsStorageControllerDumpLog(cb)
end

function storage_service:ImplControllerSystemsStorageControllerClearForeignConfig(cb)
    model.ImplControllerSystemsStorageControllerClearForeignConfig(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetNoBatteryWriteCache(cb)
    model.ImplControllerSystemsStorageControllerSetNoBatteryWriteCache(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetReadCachePercent(cb)
    model.ImplControllerSystemsStorageControllerSetReadCachePercent(cb)
end

function storage_service:ImplControllerSystemsStorageControllerSetWriteCachePolicy(cb)
    model.ImplControllerSystemsStorageControllerSetWriteCachePolicy(cb)
end

function storage_service:ImplControllerConsistencyCheckEnable(cb)
    model.ImplControllerConsistencyCheckEnable(cb)
end

function storage_service:ImplControllerConsistencyCheckDisable(cb)
    model.ImplControllerConsistencyCheckDisable(cb)
end

function storage_service:ImplControllerConsistencyCheckSetParameters(cb)
    model.ImplControllerConsistencyCheckSetParameters(cb)
end

function storage_service:ImplControllerVolumeManageCreateVolumeInNewArray(cb)
    model.ImplControllerVolumeManageCreateVolumeInNewArray(cb)
end

function storage_service:ImplControllerVolumeManageDeleteVolume(cb)
    model.ImplControllerVolumeManageDeleteVolume(cb)
end

function storage_service:ImplControllerVolumeManageCreateVolumeInExisingtArray(cb)
    model.ImplControllerVolumeManageCreateVolumeInExisingtArray(cb)
end

function storage_service:ImplControllerVolumeManageCreateCachecadeVolume(cb)
    model.ImplControllerVolumeManageCreateCachecadeVolume(cb)
end

function storage_service:ImplControllerRetirementDataWipe(cb)
    model.ImplControllerRetirementDataWipe(cb)
end

function storage_service:ImplControllerRetirementGetReport(cb)
    model.ImplControllerRetirementGetReport(cb)
end

function storage_service:ImplDriveSystemsStorageDriveSetLocationIndicatorState(cb)
    model.ImplDriveSystemsStorageDriveSetLocationIndicatorState(cb)
end

function storage_service:ImplDriveSystemsStorageDriveSetFaultIndicatorState(cb)
    model.ImplDriveSystemsStorageDriveSetFaultIndicatorState(cb)
end

function storage_service:ImplDriveSystemsStorageDriveSetFirmwareStatus(cb)
    model.ImplDriveSystemsStorageDriveSetFirmwareStatus(cb)
end

function storage_service:ImplDriveSystemsStorageDriveSetHotspareType(cb)
    model.ImplDriveSystemsStorageDriveSetHotspareType(cb)
end

function storage_service:ImplDriveSystemsStorageDriveSetBootPriority(cb)
    model.ImplDriveSystemsStorageDriveSetBootPriority(cb)
end

function storage_service:ImplDriveSystemsStorageDriveSetPatrolState(cb)
    model.ImplDriveSystemsStorageDriveSetPatrolState(cb)
end

function storage_service:ImplDriveSystemsStorageDriveCryptoErase(cb)
    model.ImplDriveSystemsStorageDriveCryptoErase(cb)
end

function storage_service:ImplDriveMetricGetData(cb)
    model.ImplDriveMetricGetData(cb)
end

function storage_service:ImplDriveMetricGetItems(cb)
    model.ImplDriveMetricGetItems(cb)
end

function storage_service:ImplDriveMetricSynDataAcquisitionEnable(cb)
    model.ImplDriveMetricSynDataAcquisitionEnable(cb)
end

function storage_service:ImplDriveNVMeGetSmartInfo(cb)
    model.ImplDriveNVMeGetSmartInfo(cb)
end

function storage_service:ImplDriveNVMeSendNvmeMICommand(cb)
    model.ImplDriveNVMeSendNvmeMICommand(cb)
end

function storage_service:ImplDriveNVMeGetTelemetryData(cb)
    model.ImplDriveNVMeGetTelemetryData(cb)
end

function storage_service:ImplDriveNVMeGetIdentifyData(cb)
    model.ImplDriveNVMeGetIdentifyData(cb)
end

function storage_service:ImplDriveDriveDiagnoseCollectDiagnoseData(cb)
    model.ImplDriveDriveDiagnoseCollectDiagnoseData(cb)
end

function storage_service:ImplDriveDriveDiagnoseGetDiagnoseData(cb)
    model.ImplDriveDriveDiagnoseGetDiagnoseData(cb)
end

function storage_service:ImplDrivesDrivesCollectIODeteriorationDiagInfo(cb)
    model.ImplDrivesDrivesCollectIODeteriorationDiagInfo(cb)
end

function storage_service:ImplDrivesDrivesGetDrivesSubHealthDiagInfo(cb)
    model.ImplDrivesDrivesGetDrivesSubHealthDiagInfo(cb)
end

function storage_service:ImplDrivesDrivesGetNvmeDriveRawData(cb)
    model.ImplDrivesDrivesGetNvmeDriveRawData(cb)
end

function storage_service:ImplDrivesDrivesGetNvmeDriveCount(cb)
    model.ImplDrivesDrivesGetNvmeDriveCount(cb)
end

function storage_service:ImplDrivesDrivesSetDriveSubHealthDiagResult(cb)
    model.ImplDrivesDrivesSetDriveSubHealthDiagResult(cb)
end

function storage_service:ImplDrivesDrivesStartCollectLog(cb)
    model.ImplDrivesDrivesStartCollectLog(cb)
end

function storage_service:ImplVolumeVolumeSetName(cb)
    model.ImplVolumeVolumeSetName(cb)
end

function storage_service:ImplVolumeVolumeSetReadPolicy(cb)
    model.ImplVolumeVolumeSetReadPolicy(cb)
end

function storage_service:ImplVolumeVolumeSetWritePolicy(cb)
    model.ImplVolumeVolumeSetWritePolicy(cb)
end

function storage_service:ImplVolumeVolumeSetBootable(cb)
    model.ImplVolumeVolumeSetBootable(cb)
end

function storage_service:ImplVolumeVolumeSetIOPolicy(cb)
    model.ImplVolumeVolumeSetIOPolicy(cb)
end

function storage_service:ImplVolumeVolumeSetBGIEnable(cb)
    model.ImplVolumeVolumeSetBGIEnable(cb)
end

function storage_service:ImplVolumeVolumeSetAccessPolicy(cb)
    model.ImplVolumeVolumeSetAccessPolicy(cb)
end

function storage_service:ImplVolumeVolumeSetDiskCachePolicy(cb)
    model.ImplVolumeVolumeSetDiskCachePolicy(cb)
end

function storage_service:ImplVolumeVolumeStartForegroundInit(cb)
    model.ImplVolumeVolumeStartForegroundInit(cb)
end

function storage_service:ImplVolumeVolumeCancelForegroundInit(cb)
    model.ImplVolumeVolumeCancelForegroundInit(cb)
end

function storage_service:ImplVolumeVolumeSetCachecadeEnable(cb)
    model.ImplVolumeVolumeSetCachecadeEnable(cb)
end

function storage_service:ImplVolumeVolumeSetAccelerator(cb)
    model.ImplVolumeVolumeSetAccelerator(cb)
end

function storage_service:ImplVolumeVolumeSetCapacitySize(cb)
    model.ImplVolumeVolumeSetCapacitySize(cb)
end

function storage_service:ImplVolumeVolumeSetStripSize(cb)
    model.ImplVolumeVolumeSetStripSize(cb)
end

---@param mdb_object object
---@param MetricName string
---@param TagPath string
function storage_service:DriveMetricCollectSignal(mdb_object, MetricName, TagPath)
    self.bus:signal(mdb_object.path, 'bmc.kepler.Metric', 'CollectSignal', 'a{ss}ss', context.get_context() or {},
        MetricName, TagPath)
end

function storage_service:get_bus()
    return self.bus
end

function storage_service:register_ipmi_cmd(ipmi_cmd, cb)
    self.ipmi_cmds[ipmi_cmd.name] = ipmi.register_ipmi_cmd(self.bus, self.service_name, ipmi_cmd,
        cb or self[ipmi_cmd.name])
end

function storage_service:unregister_ipmi_cmd(ipmi_cmd)
    local cmd_obj = self.ipmi_cmds[ipmi_cmd.name]
    if not cmd_obj then
        return
    end

    cmd_obj:unregister()
    self.ipmi_cmds[ipmi_cmd.name] = nil
end

function storage_service:ctor()
    self.ipmi_cmds = {}
    self.signal_slots = {}
    self.name = self.name or storage_service.package
    self.db = open_db(':memory:', datas)
    if skynet.getenv('TEST_DATA_DIR') then
        self.reset_local_db = open_local_db(skynet.getenv('TEST_DATA_DIR') .. '/storage_reset.db', datas, 'reset')
    else
        self.reset_local_db = open_local_db('/opt/bmc/pram/persistence.local/storage.db', datas, 'reset')
    end

    orm_classes.init(self.db)
    self.bus:request_name(app_base.Service.get_service_name(self.name))
    model.init(self.bus)
    storage_service.bus = self.bus
end

function storage_service:pre_init()
    storage_service.super.pre_init(self)
    self.persist = persist_client.new(self.bus, self.db, self, {
        ['t_storage_config_table_info'] = true,
        ['t_controller_info'] = true,
        ['t_storage_drive'] = true,
        ['t_ReplicaDrives_info'] = true,
        ['t_batterys_info'] = true,
        ['t_storage_setting_table_info'] = true,
        ['t_nvme_info'] = true,
        ['t_vpd_connect'] = true
    })
    object_manage.set_persist_client(self.persist)
end

function storage_service:init()
    storage_service.super.init(self)
end

return storage_service
