-- Copyright (c) 2025 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local mdb = require 'mc.mdb'
local class = require 'mc.class'
local app_base = require 'mc.client_app_base'
local mdb_service = require 'mc.mdb.mdb_service'
local subscribe_signal = require 'mc.mdb.subscribe_signal'
local org_freedesktop_dbus = require 'sd_bus.org_freedesktop_dbus'

local match_rule = org_freedesktop_dbus.MatchRule
local get_non_virtual_interface_objects = mdb_service.get_non_virtual_interface_objects
local foreach_non_virtual_interface_objects = mdb_service.foreach_non_virtual_interface_objects

local BlockIO = require 'storage.json_types.BlockIO'
local Accessor = require 'storage.json_types.Accessor'
local PCIeDevice = require 'storage.json_types.PCIeDevice'
local PCIe = require 'storage.json_types.PCIe'
local PCIeEndpoint = require 'storage.json_types.PCIeEndpoint'
local PCIeTransport = require 'storage.json_types.PCIeTransport'
local PCIeRoutingEntry = require 'storage.json_types.PCIeRoutingEntry'
local Events = require 'storage.json_types.Events'
local FruCtrl = require 'storage.json_types.FruCtrl'
local SmBios = require 'storage.json_types.SmBios'
local Bios = require 'storage.json_types.Bios'
local PcieAddrInfo = require 'storage.json_types.PcieAddrInfo'
local Component = require 'storage.json_types.Component'
local SmsStatus = require 'storage.json_types.SmsStatus'
local Sms = require 'storage.json_types.Sms'
local Connector = require 'storage.json_types.Connector'
local Board = require 'storage.json_types.Board'
local Unit = require 'storage.json_types.Unit'

---@class storage_client: BasicClient
local storage_client = class(app_base.Client)

function storage_client:GetBlockIOObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip.BlockIO', true)
end

function storage_client:ForeachBlockIOObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip.BlockIO', cb, true)
end

function storage_client:GetAccessorObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Accessor', true)
end

function storage_client:ForeachAccessorObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Accessor', cb, true)
end

function storage_client:GetPCIeDeviceObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.PCIeDevices.PCIeDevice', true)
end

function storage_client:ForeachPCIeDeviceObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.PCIeDevices.PCIeDevice', cb, true)
end

function storage_client:GetPCIeObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.MctpBinding.PCIe', true)
end

function storage_client:ForeachPCIeObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.MctpBinding.PCIe', cb, true)
end

function storage_client:GetPCIeEndpointObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeEndpoint', true)
end

function storage_client:ForeachPCIeEndpointObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeEndpoint', cb, true)
end

function storage_client:GetPCIeTransportObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeTransport', true)
end

function storage_client:ForeachPCIeTransportObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeTransport', cb, true)
end

function storage_client:GetPCIeRoutingEntryObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeRoutingEntry', true)
end

function storage_client:ForeachPCIeRoutingEntryObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeRoutingEntry', cb, true)
end

function storage_client:GetEventsObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Events', true)
end

function storage_client:ForeachEventsObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Events', cb, true)
end

function storage_client:GetFruCtrlObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.FruCtrl', true)
end

function storage_client:ForeachFruCtrlObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.FruCtrl', cb, true)
end

function storage_client:GetSmBiosObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.SmBios', true)
end

function storage_client:ForeachSmBiosObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.SmBios', cb, true)
end

function storage_client:GetBiosObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Bios', true)
end

function storage_client:ForeachBiosObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Bios', cb, true)
end

function storage_client:GetPcieAddrInfoObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.PcieAddrInfo', true)
end

function storage_client:ForeachPcieAddrInfoObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.PcieAddrInfo', cb, true)
end

function storage_client:GetComponentObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Component', true)
end

function storage_client:ForeachComponentObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Component', cb, true)
end

function storage_client:GetConnectorObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Connector', true)
end

function storage_client:ForeachConnectorObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Connector', cb, true)
end

function storage_client:GetBoardObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Board', true)
end

function storage_client:ForeachBoardObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Board', cb, true)
end

function storage_client:OnBlockIOInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.BlockIO')
end

function storage_client:OnBlockIOInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.BlockIO')
end

function storage_client:OnAccessorInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Accessor')
end

function storage_client:OnAccessorInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Accessor')
end

function storage_client:OnPCIeDeviceInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.PCIeDevices.PCIeDevice')
end

function storage_client:OnPCIeDeviceInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.PCIeDevices.PCIeDevice')
end

function storage_client:OnPCIeInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.MctpBinding.PCIe')
end

function storage_client:OnPCIeInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.MctpBinding.PCIe')
end

function storage_client:OnPCIeEndpointInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeEndpoint')
end

function storage_client:OnPCIeEndpointInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeEndpoint')
end

function storage_client:OnPCIeTransportInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeTransport')
end

function storage_client:OnPCIeTransportInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeTransport')
end

function storage_client:OnPCIeRoutingEntryInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeRoutingEntry')
end

function storage_client:OnPCIeRoutingEntryInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeRoutingEntry')
end

function storage_client:OnEventsInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Events')
end

function storage_client:OnEventsInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Events')
end

function storage_client:OnFruCtrlInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.FruCtrl')
end

function storage_client:OnFruCtrlInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.FruCtrl')
end

function storage_client:OnSmBiosInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.SmBios')
end

function storage_client:OnSmBiosInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.SmBios')
end

function storage_client:OnBiosInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Bios')
end

function storage_client:OnBiosInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Bios')
end

function storage_client:OnPcieAddrInfoInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.PcieAddrInfo')
end

function storage_client:OnPcieAddrInfoInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.PcieAddrInfo')
end

function storage_client:OnComponentPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Component', {'ReplaceFlag'})
end

function storage_client:OnComponentInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Component')
end

function storage_client:OnComponentInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Component')
end

function storage_client:GetSmsSmsStatusObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
        ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or '/bmc/kepler/Systems/${SystemId}/Sms',
        'bmc.kepler.Systems.Sms.SmsStatus')
end

function storage_client:GetSmsStatusObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Sms.SmsStatus', true,
        {'/bmc/kepler/Systems/${SystemId}/Sms'})
end

function storage_client:ForeachSmsStatusObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Sms.SmsStatus', cb, true,
        {'/bmc/kepler/Systems/${SystemId}/Sms'})
end

function storage_client:OnSmsStatusPropertiesChanged(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms.SmsStatus', {'State'})
end

function storage_client:OnSmsStatusInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Sms.SmsStatus')
end

function storage_client:OnSmsStatusInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms.SmsStatus')
end

function storage_client:GetSmsSmsObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
        ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or '/bmc/kepler/Systems/${SystemId}/Sms',
        'bmc.kepler.Systems.Sms')
end

function storage_client:GetSmsObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Sms', true,
        {'/bmc/kepler/Systems/${SystemId}/Sms'})
end

function storage_client:ForeachSmsObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Sms', cb, true,
        {'/bmc/kepler/Systems/${SystemId}/Sms'})
end

function storage_client:OnSmsInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Sms')
end

function storage_client:OnSmsInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms')
end

function storage_client:OnConnectorInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Connector')
end

function storage_client:OnConnectorInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Connector')
end

function storage_client:OnBoardInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Board')
end

function storage_client:OnBoardInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Board')
end

function storage_client:GetHddBackplaneUnitObject(path_params)
    return mdb.try_get_object(self:get_bus(),
        path_params and
            ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Boards/HddBackplane/' .. path_params['Id']) or
            '/bmc/kepler/Systems/${SystemId}/Boards/HddBackplane/${Id}', 'bmc.kepler.Systems.Board.Unit')
end

function storage_client:GetUnitObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Board.Unit', true,
        {'/bmc/kepler/Systems/${SystemId}/Boards/HddBackplane/${Id}'})
end

function storage_client:ForeachUnitObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Board.Unit', cb, true,
        {'/bmc/kepler/Systems/${SystemId}/Boards/HddBackplane/${Id}'})
end

function storage_client:OnUnitInterfacesAdded(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Boards/HddBackplane/' ..
                                   path_params['Id']) or '/bmc/kepler/Systems/${SystemId}/Boards/HddBackplane/${Id}'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Board.Unit')
end

function storage_client:OnUnitInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Boards/HddBackplane/' ..
                                   path_params['Id']) or '/bmc/kepler/Systems/${SystemId}/Boards/HddBackplane/${Id}'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Board.Unit')
end

---@param Request string
---@return Sms.SetForwardWhiteListRsp
function storage_client:SmsSmsSetForwardWhiteList(ctx, path_params, Request)
    local req = Sms.SetForwardWhiteListReq.new(Request):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetForwardWhiteListRsp.new(obj:SetForwardWhiteList(ctx, req:unpack(true)))
end

function storage_client:PSmsSmsSetForwardWhiteList(ctx, path_params, Request)
    return pcall(function()
        local req = Sms.SetForwardWhiteListReq.new(Request):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetForwardWhiteListRsp.new(obj:SetForwardWhiteList(ctx, req:unpack(true)))
    end)
end

---@param Request string
---@param RetryTimes integer
---@return Sms.ForwardRequestRsp
function storage_client:SmsSmsForwardRequest(ctx, path_params, Request, RetryTimes)
    local req = Sms.ForwardRequestReq.new(Request, RetryTimes):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.ForwardRequestRsp.new(obj:ForwardRequest(ctx, req:unpack(true)))
end

function storage_client:PSmsSmsForwardRequest(ctx, path_params, Request, RetryTimes)
    return pcall(function()
        local req = Sms.ForwardRequestReq.new(Request, RetryTimes):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.ForwardRequestRsp.new(obj:ForwardRequest(ctx, req:unpack(true)))
    end)
end

---@param RequestMethod string
---@param RequestToken string
---@param RequestBody string
---@return Sms.ForwardEventRsp
function storage_client:SmsSmsForwardEvent(ctx, path_params, RequestMethod, RequestToken, RequestBody)
    local req = Sms.ForwardEventReq.new(RequestMethod, RequestToken, RequestBody):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.ForwardEventRsp.new(obj:ForwardEvent(ctx, req:unpack(true)))
end

function storage_client:PSmsSmsForwardEvent(ctx, path_params, RequestMethod, RequestToken, RequestBody)
    return pcall(function()
        local req = Sms.ForwardEventReq.new(RequestMethod, RequestToken, RequestBody):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.ForwardEventRsp.new(obj:ForwardEvent(ctx, req:unpack(true)))
    end)
end

---@param ResourceType string
---@return Sms.UpdateSpecificResourceRsp
function storage_client:SmsSmsUpdateSpecificResource(ctx, path_params, ResourceType)
    local req = Sms.UpdateSpecificResourceReq.new(ResourceType):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.UpdateSpecificResourceRsp.new(obj:UpdateSpecificResource(ctx, req:unpack(true)))
end

function storage_client:PSmsSmsUpdateSpecificResource(ctx, path_params, ResourceType)
    return pcall(function()
        local req = Sms.UpdateSpecificResourceReq.new(ResourceType):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.UpdateSpecificResourceRsp.new(obj:UpdateSpecificResource(ctx, req:unpack(true)))
    end)
end

---@param ChannelType integer
---@param IsPowerOffPer boolean
---@return Sms.SetChannelTypeRsp
function storage_client:SmsSmsSetChannelType(ctx, path_params, ChannelType, IsPowerOffPer)
    local req = Sms.SetChannelTypeReq.new(ChannelType, IsPowerOffPer):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetChannelTypeRsp.new(obj:SetChannelType(ctx, req:unpack(true)))
end

function storage_client:PSmsSmsSetChannelType(ctx, path_params, ChannelType, IsPowerOffPer)
    return pcall(function()
        local req = Sms.SetChannelTypeReq.new(ChannelType, IsPowerOffPer):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetChannelTypeRsp.new(obj:SetChannelType(ctx, req:unpack(true)))
    end)
end

---@param IsPowerOffPer boolean
---@return Sms.GetChannelTypeRsp
function storage_client:SmsSmsGetChannelType(ctx, path_params, IsPowerOffPer)
    local req = Sms.GetChannelTypeReq.new(IsPowerOffPer):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.GetChannelTypeRsp.new(obj:GetChannelType(ctx, req:unpack(true)))
end

function storage_client:PSmsSmsGetChannelType(ctx, path_params, IsPowerOffPer)
    return pcall(function()
        local req = Sms.GetChannelTypeReq.new(IsPowerOffPer):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.GetChannelTypeRsp.new(obj:GetChannelType(ctx, req:unpack(true)))
    end)
end

---@param Enabled boolean
---@return Sms.SetCdevChannelEnabledRsp
function storage_client:SmsSmsSetCdevChannelEnabled(ctx, path_params, Enabled)
    local req = Sms.SetCdevChannelEnabledReq.new(Enabled):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetCdevChannelEnabledRsp.new(obj:SetCdevChannelEnabled(ctx, req:unpack(true)))
end

function storage_client:PSmsSmsSetCdevChannelEnabled(ctx, path_params, Enabled)
    return pcall(function()
        local req = Sms.SetCdevChannelEnabledReq.new(Enabled):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetCdevChannelEnabledRsp.new(obj:SetCdevChannelEnabled(ctx, req:unpack(true)))
    end)
end

function storage_client:SubscribePCIeEndpointOnMessage(cb)
    local sig = match_rule.signal('OnMessage', 'bmc.kepler.Systems.Mctp.PCIeEndpoint')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function storage_client:SubscribePCIeEndpointMessageReceived(cb)
    local sig = match_rule.signal('MessageReceived', 'bmc.kepler.Systems.Mctp.PCIeEndpoint')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function storage_client:SubscribeFruCtrlBeforePowerOnSignal(cb)
    local sig = match_rule.signal('BeforePowerOnSignal', 'bmc.kepler.Systems.FruCtrl')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function storage_client:ctor()
    self.signal_slots = {}
end

---@type storage_client
return storage_client.new('storage')
